/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Copy the kernel somewhere and jump in.
 */

#include <flux/x86/types.h>		/* vm_size_t */
#include <flux/x86/far_ptr.h>		/* far_pointer_32 */
#include <flux/x86/proc_reg.h>		/* cli/sti */
#include <flux/x86/base_gdt.h>		/* LINEAR_CS */
#include <flux/x86/base_vm.h>		/* kvtophys */
#include <flux/c/malloc.h>		/* mustmalloc */
#include <flux/c/string.h>		/* bcopy */
#include <flux/c/stdio.h>		/* printf */

#include "boot.h"			/* boot_info */

extern char do_boot[], do_boot_end[];

void boot_start(void)
{
	vm_size_t stub_size;
	void *stub;
	struct far_pointer_32 ptr;
	vm_offset_t copy_source;
	vm_size_t copy_size;

	boot_info_dump();

	printf("Booting kernel...\n");

	cli();

	/* All of the data structures that are important to the kernel,
	   are in memory guaranteed not to conflict with the kernel image
	   or with anything else.
	   However, the kernel image itself is not in its final position,
	   because that might be right on top of us,
	   or on top of anything we allocated before reserving the kernel image region.
	   Therefore, we must allocate (non-conflicting) memory for a small stub
	   to copy the kernel image to its final position and invoke it.  */
	stub_size = do_boot_end - do_boot;
	stub = mustmalloc(stub_size);
	bcopy(do_boot, stub, stub_size);

	ptr.seg = LINEAR_CS;
	ptr.ofs = kvtophys(stub);

	/* The kernel image source and destination regions may overlap,
	   so we may have to copy either forwards or backwards.  */
	copy_source = kvtophys(boot_kern_image);
	copy_size = boot_kern_hdr.load_end_addr - boot_kern_hdr.load_addr;
	if (copy_source > boot_kern_hdr.load_addr)
	{
		asm volatile("
			cld
			ljmp	%0
		" :
		  : "mr" (ptr),		/* XXX r is inappropriate but gcc wants it */
		    "a" (boot_kern_hdr.entry),
		    "S" (copy_source),
		    "D" (boot_kern_hdr.load_addr),
		    "c" (copy_size),
		    "b" (kvtophys(boot_info)),
		    "d" (LINEAR_DS));
	}
	else
	{
		printf("(copying backwards...)\n"); /* XXX */
		asm volatile("
			std
			ljmp	%0
		" :
		  : "mr" (ptr),		/* XXX r is inappropriate but gcc wants it */
		    "a" (boot_kern_hdr.entry),
		    "S" (copy_source + copy_size - 1),
		    "D" (boot_kern_hdr.load_addr + copy_size - 1),
		    "c" (copy_size),
		    "b" (kvtophys(boot_info)),
		    "d" (LINEAR_DS));
	}
}
