/* 
 * Copyright (c) 1995,1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

#include <flux/x86/pc/rtc.h>		/* RTC_BASEHI */
#include <flux/x86/pc/phys_lmm.h>	/* phys_lmm_add */
#include <flux/x86/types.h>		/* vm_offset_t */
#include <flux/x86/multiboot.h>		/* multiboot structs */
#include <flux/x86/base_cpu.h>		/* base_cpu_setup */
#include <flux/x86/base_vm.h>		/* kvtophys */
#include <flux/exec/exec.h>		/* exec_load */
#include <flux/exec/a.out.h>		/* struct exec */
#include <flux/c/sys/reboot.h>		/* RB_ASKNAME */
#include <flux/c/malloc.h>		/* malloc_reserve */
#include <flux/c/string.h>		/* memcpy */
#include <flux/c/stdio.h>		/* printf */
#include <flux/c/assert.h>		/* assert */
#include <flux/lmm.h>			/* lmm_remove_free */

#include "boottype.h"
#include "boot.h"
#include "intr.h"

static vm_offset_t phys_mem_lower;
static vm_offset_t phys_mem_upper;

struct multiboot_info *boot_info;
static struct multiboot_module *boot_mods;

struct multiboot_header boot_kern_hdr;
void *boot_kern_image;
static  struct exec_info boot_kern_info;

extern struct lmod
{
	void *start;
	void *end;
	char *string;
} boot_modules[];

static vm_offset_t      symaddr;
static vm_size_t        symsize;
static vm_offset_t      straddr;
static vm_size_t        strsize;

/* Find the extended memory available and add it to malloc's free list.  */
static void grab_ext_mem(void)
{
	extern char _end[];

	vm_offset_t free_bot, free_top;

	/*
	 * Find lower and upper bounds of physical memory.
	 * We look in the NVRAM to find the size of base memory and
	 * the size of extended memory.
	 */
	phys_mem_lower = (rtcin(RTC_BASEHI) << 8) | rtcin(RTC_BASELO);
	phys_mem_upper = (rtcin(RTC_DEXTHI) << 8) | rtcin(RTC_DEXTLO);

	/* Figure out how much mem is actually free. */
	free_bot = kvtophys(_end);
	free_top = 1024*phys_mem_upper;
	free_top += 0x100000;		/* skip 1M of I/O and ROM */
	assert(free_bot > 0x100000);
	assert(free_bot < free_top);

	/* Add it to the free list. */
	phys_lmm_init();
	phys_lmm_add(free_bot, free_top - free_bot);

	printf("using extended memory %08x-%08x\n", free_bot, free_top);
}

/* Unlike the Linux boot mechanism, the older Mach and BSD systems
   do not have any notion of a general command line,
   only a fixed, kludgy (and diverging) set of flags and values.
   Here we make a best-effort attempt
   to piece together a sane Linux-style command line
   from the boothowto and bootdev values provided by the boot loader.
   Naturally, the interpretation changes depending on who loaded us;
   that's what the boottype is for (see crt0.S).  */
static void init_cmdline(void)
{
	static char buf[30];
	char *major;

	/* First handle the option flags.
	   Just supply the flags as they would appear on the command line
	   of the Mach or BSD reboot command line,
	   and let the kernel handle or ignore them as appropriate.  */
	if (boothowto & RB_ASKNAME)
		strcat(buf, "-a ");
	if (boothowto & RB_SINGLE)
		strcat(buf, "-s ");
	if (boothowto & RB_DFLTROOT)
		strcat(buf, "-r ");
	if (boothowto & RB_HALT)
		strcat(buf, "-b ");
	if (boottype == BOOTTYPE_MACH)
	{
		/* These were determined from what the command-line parser does
		   in boot.c in the original Mach boot loader.  */
		if (boothowto & MACH_RB_KDB)
			strcat(buf, "-d ");
	}
	else
	{
		/* These were determined from what the command-line parser does
		   in boot.c in the FreeBSD 2.0 boot loader.  */
		if (boothowto & BSD_RB_KDB)
			strcat(buf, "-d ");
		if (boothowto & BSD_RB_CONFIG)
			strcat(buf, "-c ");
	}

	/* Now indicate the root device with a "root=" option.  */
	major = 0;
	if (boottype == BOOTTYPE_MACH)
	{
		static char *devs[] = {"hd", "fd", "wt", "sd", "ha"};
		if (B_TYPE(bootdev) < sizeof(devs)/sizeof(devs[0]))
			major = devs[B_TYPE(bootdev)];
	}
	else
	{
		static char *devs[] = {"wd", "hd", "fd", "wt", "sd"};
		if (B_TYPE(bootdev) < sizeof(devs)/sizeof(devs[0]))
			major = devs[B_TYPE(bootdev)];
	}
	if (major)
	{
		sprintf(buf + strlen(buf), "root=%s%ld%c",
			major, B_UNIT(bootdev),
			'a' + (char)B_PARTITION(bootdev));
	}

	/* Insert the command line into the boot_info structure.  */
	boot_info->cmdline = (vm_offset_t)kvtophys(buf);
	boot_info->flags |= MULTIBOOT_CMDLINE;
}

static
int kimg_read(void *handle, vm_offset_t file_ofs, void *buf, vm_size_t size, vm_size_t *out_actual)
{
	/* XXX limit length */
	memcpy(buf, boot_modules[0].start + file_ofs, size);
	*out_actual = size;
	return 0;
}

static
int kimg_read_exec_1(void *handle, vm_offset_t file_ofs, vm_size_t file_size,
		     vm_offset_t mem_addr, vm_size_t mem_size,
		     exec_sectype_t section_type)
{
	if (!(section_type & EXEC_SECTYPE_ALLOC))
		return 0;

	assert(mem_size > 0);
	if (mem_addr < boot_kern_hdr.load_addr)
		boot_kern_hdr.load_addr = mem_addr;
	if (mem_addr+file_size > boot_kern_hdr.load_end_addr)
		boot_kern_hdr.load_end_addr = mem_addr+file_size;
	if (mem_addr+mem_size > boot_kern_hdr.bss_end_addr)
		boot_kern_hdr.bss_end_addr = mem_addr+mem_size;

	return 0;
}

static
int kimg_read_exec_2(void *handle, vm_offset_t file_ofs, vm_size_t file_size,
		     vm_offset_t mem_addr, vm_size_t mem_size,
		     exec_sectype_t section_type)
{
	if (!(section_type & EXEC_SECTYPE_ALLOC))
		return 0;

	assert(mem_size > 0);
	assert(mem_addr >= boot_kern_hdr.load_addr);
	assert(mem_addr+file_size <= boot_kern_hdr.load_end_addr);
	assert(mem_addr+mem_size <= boot_kern_hdr.bss_end_addr);

	memcpy(boot_kern_image + mem_addr - boot_kern_hdr.load_addr,
		boot_modules[0].start + file_ofs, file_size);

	return 0;
}

/*
 * Callback for reading the symbol table.
 */
static
int kimg_read_exec_3(void *handle, vm_offset_t file_ofs, vm_size_t file_size,
		     vm_offset_t mem_addr, vm_size_t mem_size,
		     exec_sectype_t section_type)
{
	if (section_type & EXEC_SECTYPE_AOUT_SYMTAB) {
		symaddr = boot_modules[0].start + file_ofs;
		symsize = file_size;
	}
	else if (section_type & EXEC_SECTYPE_AOUT_STRTAB) {
		straddr = boot_modules[0].start + file_ofs;
		strsize = file_size;
	}

	return 0;
}

static void init_symtab(struct multiboot_header *h)
{
	int err;
	void *symtab;

	/*
	 * XXX we only deal with aoutly files.
	 */
	if (! (h->flags & MULTIBOOT_AOUT_KLUDGE))
		return;
	
	/*
	 * Figure out where the symtab is.
	 */
	if ((err = exec_load(kimg_read, kimg_read_exec_3, 0,
			     &boot_kern_info)) != 0)
		panic("cannot load kernel image 3: error code %d", err);

	/*
	 * Copy the symtab into non-conflicting memory.
	 * We assume the strtab is right after the symtab.
	 * In a.out the first word of the strtab is the size,
	 * for the symtab we add size info in our copy.
	 */
	symtab = mustmalloc(sizeof(vm_size_t) + symsize + strsize);
	*((vm_size_t *)symtab) = symsize;
	memcpy(symtab + sizeof(vm_size_t), (void *)symaddr, symsize + strsize);

	/*
	 * Register the symtab in the boot_info.
	 */
	boot_info->flags |= MULTIBOOT_AOUT_SYMS;
	boot_info->syms.a.tabsize = symsize + sizeof(vm_size_t);
	boot_info->syms.a.strsize = strsize;
	boot_info->syms.a.addr    = (vm_offset_t)kvtophys(symtab);

	printf("symtab at %#x, %d bytes; strtab at %#x, %d bytes\n",
	       symaddr, symsize,
	       straddr, strsize);
}

void raw_start(void)
{
	struct multiboot_header *h;
	int i, err;

	base_cpu_setup();
	interrupt_boot();

	/* Get some memory to work in.  */
	grab_ext_mem();

	if (boot_modules[0].start == 0)
		panic("This boot image contains no boot modules!?!?");

	/* Scan for the multiboot_header.  */
	for (i = 0; ; i += 4)
	{
		if (i >= MULTIBOOT_SEARCH)
			panic("kernel image has no multiboot_header");
		h = (struct multiboot_header*)(boot_modules[0].start+i);
		if (h->magic == MULTIBOOT_MAGIC
		    && !(h->magic + h->flags + h->checksum))
			break;
	}
	if (h->flags & MULTIBOOT_MUSTKNOW & ~MULTIBOOT_MEMORY_INFO)
		panic("unknown multiboot_header flag bits %08x",
			h->flags & MULTIBOOT_MUSTKNOW & ~MULTIBOOT_MEMORY_INFO);
	boot_kern_hdr = *h;

	if (h->flags & MULTIBOOT_AOUT_KLUDGE)
	{
		boot_kern_image = (void*)h + h->load_addr - h->header_addr;
	}
	else
	{
		/*
		 * No a.out-kludge information available;
		 * attempt to interpret the exec header instead,
		 * using the simple interpreter in libexec.a.
		 */

		/* Perform the "load" in two passes.
		   In the first pass, find the number of sections the load image contains
		   and reserve the physical memory containing each section.
		   Also, initialize the boot_kern_hdr to reflect the extent of the image.
		   In the second pass, load the sections into a temporary area
		   that can be copied to the final location all at once by do_boot.S.  */

		boot_kern_hdr.load_addr = 0xffffffff;
		boot_kern_hdr.load_end_addr = 0;
		boot_kern_hdr.bss_end_addr = 0;

		if ((err = exec_load(kimg_read, kimg_read_exec_1, 0,
				     &boot_kern_info)) != 0)
			panic("cannot load kernel image 1: error code %d", err);
		boot_kern_hdr.entry = boot_kern_info.entry;

		/* It's OK to malloc this before reserving the memory the kernel will occupy,
		   because do_boot.S can deal with overlapping source and destination.  */
		assert(boot_kern_hdr.load_addr < boot_kern_hdr.load_end_addr);
		assert(boot_kern_hdr.load_end_addr < boot_kern_hdr.bss_end_addr);
		boot_kern_image = malloc(boot_kern_hdr.load_end_addr - boot_kern_hdr.load_addr);

		if ((err = exec_load(kimg_read, kimg_read_exec_2, 0,
				     &boot_kern_info)) != 0)
			panic("cannot load kernel image 2: error code %d", err);
		assert(boot_kern_hdr.entry == boot_kern_info.entry);
	}

	/* Reserve the memory that the kernel will eventually occupy.
	   All malloc calls after this are guaranteed to stay out of this region.  */
	lmm_remove_free(&malloc_lmm,
			(void *)phystokv(boot_kern_hdr.load_addr),
			phystokv(boot_kern_hdr.bss_end_addr)
			- phystokv(boot_kern_hdr.load_addr));

	printf("kernel at %08x-%08x text+data %d bss %d\n",
		boot_kern_hdr.load_addr, boot_kern_hdr.bss_end_addr,
		boot_kern_hdr.load_end_addr - boot_kern_hdr.load_addr,
		boot_kern_hdr.bss_end_addr - boot_kern_hdr.load_end_addr);
	assert(boot_kern_hdr.load_addr < boot_kern_hdr.load_end_addr);
	assert(boot_kern_hdr.load_end_addr < boot_kern_hdr.bss_end_addr);
	if (boot_kern_hdr.load_addr < 0x1000)
		panic("kernel wants to be loaded too low!");
#if 0
	if (boot_kern_hdr.bss_end_addr > phys_mem_max)
		panic("kernel wants to be loaded beyond available physical memory!");
#endif
	if ((boot_kern_hdr.load_addr < 0x100000)
	    && (boot_kern_hdr.bss_end_addr > 0xa0000))
		panic("kernel wants to be loaded on top of I/O space!");

	boot_info = (struct multiboot_info*)mustcalloc(sizeof(*boot_info), 1);

	/* Build a command line to pass to the kernel.  */
	init_cmdline();

	/* Add memory information */
	boot_info->flags |= MULTIBOOT_MEMORY;
	boot_info->mem_upper = phys_mem_upper;
	boot_info->mem_lower = phys_mem_lower;

	/* Indicate to the kernel which BIOS disk device we booted from.
	   The Mach and BSD boot loaders obscure this information somewhat;
	   we have to extract it from the mangled bootdev value.
	   We assume that any unit other than floppy means BIOS hard drive.
	   XXX If we boot from FreeBSD's netboot, we shouldn't set this.  */
	boot_info->flags |= MULTIBOOT_BOOT_DEVICE;
	if (boottype == BOOTTYPE_MACH)
		boot_info->boot_device[0] = B_TYPE(bootdev) == 1 ? 0 : 0x80;
	else
		boot_info->boot_device[0] = B_TYPE(bootdev) == 2 ? 0 : 0x80;
	boot_info->boot_device[0] += B_UNIT(bootdev);
	boot_info->boot_device[1] = 0xff;
	boot_info->boot_device[2] = B_PARTITION(bootdev);
	boot_info->boot_device[3] = 0xff;

	/* Find the symbol table to supply to the kernel, if possible.  */
	init_symtab(h);

	/* Initialize the boot module entries in the boot_info.  */
	boot_info->flags |= MULTIBOOT_MODS;
	for (i = 1; boot_modules[i].start; i++);
	boot_info->mods_count = i-1;
	boot_mods = (struct multiboot_module*)mustcalloc(
		boot_info->mods_count * sizeof(*boot_mods), 1);
	boot_info->mods_addr = kvtophys(boot_mods);
	for (i = 0; i < boot_info->mods_count; i++)
	{
		struct lmod *lm = &boot_modules[1+i];
		struct multiboot_module *bm = &boot_mods[i];

		assert(lm->end > lm->start);

		/* Try to leave the boot module where it is and pass its address.  */
		bm->mod_start = kvtophys(lm->start);
		bm->mod_end = kvtophys(lm->end);

		/* However, if the current location of the boot module
		   overlaps with the final location of the kernel image,
		   we have to move the boot module somewhere else.  */
		if ((bm->mod_start < boot_kern_hdr.load_end_addr)
		    && (bm->mod_end > boot_kern_hdr.load_addr))
		{
			vm_size_t size = lm->end - lm->start;
			void *newaddr = mustmalloc(size);

			printf("moving boot module %d from %08x to %08x\n",
				i, kvtophys(lm->start), kvtophys(newaddr));
			memcpy(newaddr, lm->start, size);

			bm->mod_start = kvtophys(newaddr);
			bm->mod_end = bm->mod_start + size;
		}

		/* Also provide the string associated with the module.  */
		printf("lm->string '%s'\n", lm->string);
		{
			char *newstring = mustmalloc(strlen(lm->string)+1);
			strcpy(newstring, lm->string);
			bm->string = kvtophys(newstring);
		}

		bm->reserved = 0;
	}

	boot_start();
}

