/*
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Software interrupt support.
 */

#ifndef OSKIT
#define OSKIT
#endif

#include <flux/fdev.h>
#include <flux/debug.h>
#include <flux/c/stddef.h>
#include <flux/x86/proc_reg.h>

#define NSOFT	10

struct soft_handler {
	void	(*func)(void);
	struct	soft_handler *next;
};

static struct soft_handler *soft_head, *soft_free;
static struct soft_handler soft_data[NSOFT];

int fdev_soft_intr_pending = 0;

/*
 * Initialize software interrupts.
 */
void
fdev_soft_init()
{
	int i;

	for (i = 0; i < NSOFT; i++)
		soft_data[i].next = soft_data + i + 1;
	soft_data[--i].next = NULL;
	soft_free = &soft_data[0];
}

/*
 * Register a software interrupt handler.
 */
void
fdev_register_soft_handler(void (*func)(void))
{
	struct soft_handler *sh;
	unsigned int flags;

	if (func == 0)
		panic("%s:%d: null function", __FILE__, __LINE__);

	flags = get_eflags();
	cli();
	for (sh = soft_head; sh; sh = sh->next)
		if (sh->func == func) {
			set_eflags(flags);
			return;
		}
	sh = soft_free;
	if (sh == 0)
		panic("%s:%d: ran out of softints", __FILE__, __LINE__);
	sh->func = func;
	soft_free = sh->next;
	sh->next = soft_head;
	soft_head = sh;
	fdev_soft_intr_pending++;
	set_eflags(flags);
}

/*
 * Generic software interrupt handler.
 */
void
fdev_soft_intr()
{
	struct soft_handler *sh;
	void (*func)(void);

	for (sh = soft_head; sh; sh = sh->next) {
		cli();
		soft_head = sh->next;
		func = sh->func;
		sh->next = soft_free;
		soft_free = sh;
		sti();

		if (func == 0)
			panic("%s:%d: null function", __FILE__, __LINE__);

		(*func)();
	}
}
