/*
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Timer interrupt support.
 */

#include <flux/fdev.h>
#include <flux/debug.h>
#include <flux/x86/pio.h>
#include <flux/x86/proc_reg.h>
#include <flux/x86/pc/pit.h>

#define NTIMERS	10
#define TIMER_FREQ 100

struct timer_handler {
	void	(*func)(void);
	struct	timer_handler *next;
};

static struct timer_handler *timer_head, *timer_free;
static struct timer_handler timer_data[NTIMERS];

/*
 * Generic timer interrupt handler.
 */
static void
timer_intr()
{
	struct timer_handler *th;

	for (th = timer_head; th; th = th->next)
		(*th->func)();
}

/*
 * Initialize timers.
 */
void
fdev_timer_init()
{
	int i;

	for (i = 0; i < NTIMERS; i++)
		timer_data[i].next = timer_data + i + 1;
	timer_data[--i].next = 0;
	timer_free = &timer_data[0];

	/*
	 * Initialize timer to interrupt at TIMER_FREQ Hz.
	 */
	init_pit(TIMER_FREQ);

	/*
	 * Install interrupt handler.
	 */
	if (fdev_intr_alloc(0, timer_intr, 0, 0))
		panic("fdev_timer_init: couldn't install intr handler");
}

/*
 * Register a timer handler to be called at the specified frequency.
 */
void
fdev_register_timer_handler(void (*func)(void), int freq)
{
	struct timer_handler *th;
	unsigned int flags;

	if (timer_free == 0)
		panic("%s:%d: ran out of entries", __FILE__, __LINE__);
	/*
	 * XXX
	 */
	if (freq != TIMER_FREQ)
		panic("%s:%d: bad freq", __FILE__, __LINE__);
	if (func == 0)
		panic("%s:%d: null function", __FILE__, __LINE__);

	flags = get_eflags();
	cli();
	th = timer_free;
	timer_free = th->next;
	th->next = timer_head;
	timer_head = th;
	th->func = func;
	set_eflags(flags);
}
