/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Implement the device interface for Linux block drivers.
 *
 * XXX: The read/write routines presently assume a linear buffer
 * and do not make use of the buffer operations vectors.
 */

#ifndef OSKIT
#define OSKIT
#endif

#define OSKIT_INCLUDE

#include <malloc.h>

#include <flux/x86/types.h>
#include <flux/fdev.h>
#include <flux/page.h>
#include <flux/debug.h>

#include <linux/fs.h>
#include <linux/blk.h>
#include <linux/fcntl.h>
#include <linux/major.h>
#include <linux/kdev_t.h>

#include "block.h"
#include "linux_emul.h"

/*
 * Name to Linux device number mapping table.
 */
static struct {
	char	*name;
	kdev_t	dev;
} name_to_dev[] = {
	{ "sd0", MKDEV(SCSI_DISK_MAJOR, 0) },
	{ "sd1", MKDEV(SCSI_DISK_MAJOR, 1) },
	{ "sd2", MKDEV(SCSI_DISK_MAJOR, 2) },
	{ "sd3", MKDEV(SCSI_DISK_MAJOR, 3) },
	{ "sd4", MKDEV(SCSI_DISK_MAJOR, 4) },
	{ "sd5", MKDEV(SCSI_DISK_MAJOR, 5) },
	{ "sd6", MKDEV(SCSI_DISK_MAJOR, 6) },
	{ "sd7", MKDEV(SCSI_DISK_MAJOR, 7) },
	{ "sda", MKDEV(SCSI_DISK_MAJOR, 0) },
	{ "sdb", MKDEV(SCSI_DISK_MAJOR, 1) },
	{ "sdc", MKDEV(SCSI_DISK_MAJOR, 2) },
	{ "sdd", MKDEV(SCSI_DISK_MAJOR, 3) },
	{ "sde", MKDEV(SCSI_DISK_MAJOR, 4) },
	{ "sdf", MKDEV(SCSI_DISK_MAJOR, 5) },
	{ "sdg", MKDEV(SCSI_DISK_MAJOR, 6) },
	{ "sdh", MKDEV(SCSI_DISK_MAJOR, 7) },
};
#define NAME_TABLE_SIZE	(sizeof(name_to_dev) / sizeof(name_to_dev[0]))

/*
 * List of open devices.
 */
static struct peropen *open_list;

/*
 * Forward declarations.
 */
static void close(fdev_t *);
static int read(fdev_t *, void *,
		fdev_buf_vec_t *, vm_offset_t,
		unsigned, unsigned *);
static int write(fdev_t *, void *,
		 fdev_buf_vec_t *, vm_offset_t,
		 unsigned, unsigned *);

/*
 * Driver operations vector.
 */
static struct fdev_drv_ops block_drv_ops = {
	close, read, write
};

/*
 * Remove a device from the open list.
 */
static inline void
open_list_remove(struct peropen *po)
{
	struct peropen *o, **pp;

	for (o = open_list, pp = &open_list; o; pp = &o->next, o = o->next) {
		if (o == po) {
			*pp = po->next;
			return;
		}
	}
}

/*
 * Open a block device.
 */
int
linux_block_open(const char *name, unsigned flags, fdev_t **dev)
{
	int err, i, len, mode, part;
	kdev_t kdev;
	struct file file;
	struct gendisk *gd;
	struct peropen *po;

	if (flags & ~FDEV_OPEN_ALL)
		return (FDEV_INVALID_PARAMETER);

	/*
	 * Convert name to device number.
	 */
	for (i = 0; i < NAME_TABLE_SIZE; i++) {
		len = strlen(name_to_dev[i].name);
		if (!strncmp(name_to_dev[i].name, name, len))
			break;
	}

	if (i == NAME_TABLE_SIZE
	    || !blkdevs[MAJOR(name_to_dev[i].dev)].fops)
		return (FDEV_NO_SUCH_DEVICE);

	kdev = name_to_dev[i].dev;

	/*
	 * Linux partition support.
	 *
	 * NOTE: This is only here for testing.
	 * It is NOT for general use; use the partition library.
	 */
	i = strlen(name);
	if (i - len > 1)
		return (FDEV_NO_SUCH_DEVICE);

	if (i - len == 1) {
		if (*(name + len) >= 'a' && *(name + len) <= 'f')
			part = *(name + len) - 'a' + 1;
		else if (*(name + len) >= '0' && *(name + len) <= '9')
			part = *(name + len) - '0';
		else
			return (FDEV_NO_SUCH_DEVICE);
	} else
		part = 0;

	/*
	 * Construct device number.
	 */

	for (gd = gendisk_head; gd; gd = gd->next) {
		if (gd->major == MAJOR(kdev)) {
			if (part >= gd->max_p)
				return (FDEV_NO_SUCH_DEVICE);
	
			printf("major = %d, minor = %d\n",
			       MAJOR(kdev), MINOR(kdev));
			printf("minor shift = %d\n", gd->minor_shift);
			kdev = MKDEV(MAJOR(kdev),
				     MINOR(kdev) << gd->minor_shift);
			kdev |= part;
			break;
		}
	}

	switch (flags & (FDEV_OPEN_READ|FDEV_OPEN_WRITE)) {

	case FDEV_OPEN_READ:
		mode = O_RDONLY;
		break;

	case FDEV_OPEN_WRITE:
		mode = O_WRONLY;
		break;

	case FDEV_OPEN_READ|FDEV_OPEN_WRITE:
		mode = O_RDWR;
		break;

	default:
		return (FDEV_INVALID_PARAMETER);
	}

	/*
	 * Search the open list.
	 */
	while (1) {
		for (po = open_list; po; po = po->next)
			if (po->inode.i_rdev == kdev && po->mode == mode)
				break;
		if (po) {
			if (po->busy) {
				po->want = 1;
				linux_event_block(po);
				continue;
			}
			po->count++;
			return (0);
		}
		break;
	}

	/*
	 * Allocate and initialize a device structure.
	 */
	*dev = fdev_mem_alloc(sizeof(fdev_t) + sizeof(struct peropen),
			      FDEV_PHYS_WIRED, 0);
	if (!*dev)
		return (FDEV_MEMORY_SHORTAGE);
	po = (struct peropen *)(*dev + 1);
	(*dev)->data = po;
	(*dev)->drv_ops = &block_drv_ops;
	po->dev = *dev;
	po->fops = blkdevs[MAJOR(kdev)].fops;
	po->count = 1;
	po->busy = 1;
	po->want = 0;
	po->mode = mode;

	po->inode.i_rdev = kdev;

	po->inode.i_emul_data = po;
	if (gd)
		po->size = gd->part[MINOR(kdev)].nr_sects;
	else {
		if (!blk_size[MAJOR(kdev)]
		    || !blk_size[MAJOR(kdev)][MINOR(kdev)])
			panic("%s:%d: bad block size", __FILE__, __LINE__);

		po->size = (blk_size[MAJOR(kdev)][MINOR(kdev)]
			    << (BLOCK_SIZE_BITS - DISK_BLOCK_BITS));
	}

	if (blksize_size[MAJOR(kdev)]
	    && blksize_size[MAJOR(kdev)][MINOR(kdev)])
		po->bsize = blksize_size[MAJOR(kdev)][MINOR(kdev)];
	else
		po->bsize = DISK_BLOCK_SIZE;
	po->bmask = po->bsize - 1;

	for (i = DISK_BLOCK_SIZE, po->bshift = DISK_BLOCK_BITS; i < po->bsize; 
	     i <<= ++po->bshift)
		;
	po->next = open_list;
	open_list = po->next;

	if (!po->fops->open) {
		po->busy = 0;
		return (0);
	}

	file.f_mode = mode;
	file.f_flags = 0;

	printf("inode = %x\n", &po->inode);

	err = (*po->fops->open)(&po->inode, &file);

	if (err) {
		printf("LINUX_BLOCK_OPEN: ERROR! (%d)\n", err);
		open_list_remove(po);
		if (po->want) {
			po->want = 0;
			linux_event_unblock(po);
		}
		err = linux_to_fdev_error(err);
		fdev_mem_free(*dev, FDEV_PHYS_WIRED,
			      sizeof(fdev_t) + sizeof(struct peropen));
	} else {
		printf("LINUX_BLOCK_OPEN: NO ERROR!\n");
		po->busy = 0;
		if (po->want) {
			po->want = 0;
			linux_event_unblock(po);
		}
	}
	return (err);
}

/*
 * Close a device.
 */
static void
close(fdev_t *dev)
{
	struct file file;
	struct peropen *po;

	po = dev->data;
	if (po == NULL)
		panic("%s:%d: null peropen", __FILE__, __LINE__);
	if (po->count == 0)
		panic("%s:%d: bad count", __FILE__, __LINE__);

	while (po->busy) {
		po->want = 1;
		linux_event_block(po);
	}
	if (--po->count == 0) {
		if (po->fops->release) {
			po->busy = 1;
			file.f_mode = po->mode;
			file.f_flags = 0;
			(*po->fops->release)(&po->inode, &file);
		}
		open_list_remove(po);
		if (po->want) {
			po->want = 0;
			linux_event_unblock(po);
		}
		fdev_mem_free(dev, FDEV_PHYS_WIRED,
			      sizeof(fdev_t) + sizeof(struct peropen));
	}
}

/*
 * Read from a device.
 */
static int
read(fdev_t *dev, void *buf, fdev_buf_vec_t *bufvec,
     vm_offset_t off, unsigned count, unsigned *amount_read)
{
	int err;
	struct file file;
	struct peropen *po;
	unsigned bn, sz;

	po = dev->data;
	if (po == NULL)
		panic("%s:%d: null peropen", __FILE__, __LINE__);
	if (!po->fops->read || po->mode == O_WRONLY)
		return (FDEV_BAD_OPERATION);
	bn = off >> DISK_BLOCK_BITS;
	if (count == 0 || bn == po->size) {
		*amount_read = 0;
		return (0);
	}
	if (bn > po->size)
		return (FDEV_INVALID_PARAMETER);
	if ((off & (DISK_BLOCK_SIZE - 1)) == 0)
		sz = (count + (DISK_BLOCK_SIZE - 1)) >> DISK_BLOCK_BITS;
	else {
		sz = 1;
		if (count > DISK_BLOCK_SIZE - (off & (DISK_BLOCK_SIZE - 1)))
			sz += (count - (off & (DISK_BLOCK_SIZE - 1)) 
			       + (DISK_BLOCK_SIZE - 1)) >> DISK_BLOCK_BITS;
	}
	if (po->size - bn < sz)
		sz = po->size - bn;
	if ((sz << DISK_BLOCK_BITS) - (off & (DISK_BLOCK_SIZE - 1)) < count)
		count = (sz << DISK_BLOCK_BITS) - (off & (DISK_BLOCK_SIZE - 1));
	file.f_mode = po->mode;
	file.f_flags = 0;
	file.f_pos = off;
	err = (*po->fops->read)(&po->inode, &file, buf, count);
	if (err) {
		err = linux_to_fdev_error(err);
		*amount_read = 0;	/* XXX */
	} else
		*amount_read = count;
	return (err);
}

/*
 * Write to a device.
 */
static int
write(fdev_t *dev, void *buf, fdev_buf_vec_t *bufvec,
      vm_offset_t off, unsigned count, unsigned *amount_written)
{
	int err;
	struct file file;
	struct peropen *po;
	unsigned bn, sz;

	po = dev->data;
	if (po == NULL)
		panic("%s:%d: null peropen", __FILE__, __LINE__);
	if (!po->fops->write || po->mode == O_RDONLY)
		return (FDEV_BAD_OPERATION);
	bn = off >> DISK_BLOCK_BITS;
	if (count == 0 || bn == po->size) {
		*amount_written = 0;
		return (0);
	}
	if (bn > po->size)
		return (FDEV_INVALID_PARAMETER);
	if ((off & (DISK_BLOCK_SIZE - 1)) == 0)
		sz = (count + (DISK_BLOCK_SIZE - 1)) >> DISK_BLOCK_BITS;
	else {
		sz = 1;
		if (count > DISK_BLOCK_SIZE - (off & (DISK_BLOCK_SIZE - 1)))
			sz += (count - (off & (DISK_BLOCK_SIZE - 1)) 
			       + (DISK_BLOCK_SIZE - 1)) >> DISK_BLOCK_BITS;
	}
	if (po->size - bn < sz)
		sz = po->size - bn;
	if ((sz << DISK_BLOCK_BITS) - (off & (DISK_BLOCK_SIZE - 1)) < count)
		count = (sz << DISK_BLOCK_BITS) - (off & (DISK_BLOCK_SIZE - 1));
	file.f_mode = po->mode;
	file.f_flags = 0;
	file.f_pos = off;
	err = (po->fops->write)(&po->inode, &file, buf, count);
	if (err) {
		err = linux_to_fdev_error(err);
		*amount_written = 0;	/* XXX */
	} else
		*amount_written = count;
	return (err);
}
