/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Glue for Linux block drivers.
 */

#ifndef OSKIT
#define OSKIT
#endif

#define OSKIT_INCLUDE

#include <flux/fdev.h>
#include <flux/page.h>

#include <linux/fs.h>
#include <linux/blk.h>
#include <linux/fcntl.h>
#include <linux/major.h>
#include <linux/kdev_t.h>
#include <linux/string.h>

#include "block.h"

/*
 * Maximum number of block buffers we allocate at a time.
 */
#define MAX_BUF	32

/*
 * Flags for fdev_mem_alloc when allocating a buffer.
 */
#define BUF_MEM_FLAGS	(FDEV_PHYS_WIRED|FDEV_PHYS_CONTIG|FDEV_VIRT_EQ_PHYS)

/*
 * Driver file operations table.
 */
struct device_struct blkdevs[MAX_BLKDEV];

/*
 * Driver request function table.
 */
struct blk_dev_struct blk_dev[MAX_BLKDEV] = {
	{ NULL, NULL },		/* 0 no_dev */
	{ NULL, NULL },		/* 1 dev mem */
	{ NULL, NULL },		/* 2 dev fd */
	{ NULL, NULL },		/* 3 dev ide0 or hd */
	{ NULL, NULL },		/* 4 dev ttyx */
	{ NULL, NULL },		/* 5 dev tty */
	{ NULL, NULL },		/* 6 dev lp */
	{ NULL, NULL },		/* 7 dev pipes */
	{ NULL, NULL },		/* 8 dev sd */
	{ NULL, NULL },		/* 9 dev st */
	{ NULL, NULL },		/* 10 */
	{ NULL, NULL },		/* 11 */
	{ NULL, NULL },		/* 12 */
	{ NULL, NULL },		/* 13 */
	{ NULL, NULL },		/* 14 */
	{ NULL, NULL },		/* 15 */
	{ NULL, NULL },		/* 16 */
	{ NULL, NULL },		/* 17 */
	{ NULL, NULL },		/* 18 */
	{ NULL, NULL },		/* 19 */
	{ NULL, NULL },		/* 20 */
	{ NULL, NULL },		/* 21 */
	{ NULL, NULL }		/* 22 dev ide1 */
};

/*
 * blk_size contains the size of all block-devices in units of 1024 byte
 * sectors:
 *
 * blk_size[MAJOR][MINOR]
 *
 * if (!blk_size[MAJOR]) then no minor size checking is done.
 */
int *blk_size[MAX_BLKDEV] = { NULL, NULL, };

/*
 * blksize_size contains the size of all block-devices:
 *
 * blksize_size[MAJOR][MINOR]
 *
 * if (!blksize_size[MAJOR]) then 1024 bytes is assumed.
 */
int *blksize_size[MAX_BLKDEV] = { NULL, NULL, };

/*
 * hardsect_size contains the size of the hardware sector of a device.
 *
 * hardsect_size[MAJOR][MINOR]
 *
 * if (!hardsect_size[MAJOR])
 *		then 512 bytes is assumed.
 * else
 *		sector_size is hardsect_size[MAJOR][MINOR]
 * This is currently set by some scsi device and read by the msdos fs driver
 * This might be a some uses later.
 */
int *hardsect_size[MAX_BLKDEV] = { NULL, NULL, };

/*
 * This specifies how many sectors to read ahead on the disk.
 * This is unused in the OSKIT.  It is here to make drivers compile.
 */
int read_ahead[MAX_BLKDEV] = {0, };

/*
 * Wait queue on which drivers sleep waiting for request structures.
  */
struct wait_queue *wait_for_request = NULL;

/*
 * Enqueue an I/O request on a driver's queue.
 */
static inline void
enqueue_request(struct request *req)
{
	struct blk_dev_struct *dev;
	struct request *tmp;

	dev = blk_dev + MAJOR(req->rq_dev);

	linux_cli();

	tmp = dev->current_request;
	if (!tmp) {
		dev->current_request = req;
		(*dev->request_fn)();

		linux_sti();

		return;
	}
	while (tmp->next) {
		if ((IN_ORDER(tmp, req) || !IN_ORDER(tmp, tmp->next))
		    && IN_ORDER(req, tmp->next))
			break;
		tmp = tmp->next;
	}
	req->next = tmp->next;
	tmp->next = req;
	if (scsi_major(MAJOR(req->rq_dev)))
		(*dev->request_fn)();

	linux_sti();
}

#define FREE_BUFFERS(num) {						\
	int i;								\
									\
	for (i = 0; i < (num); i++)					\
		fdev_mem_free(bhp[i]->b_data, BUF_MEM_FLAGS, po->bsize);\
	fdev_mem_free(bhp, FDEV_PHYS_WIRED,				\
		      (sizeof(struct buffer_head *)			\
		       + sizeof(struct buffer_head)) * nbuf);		\
	if (req)							\
		fdev_mem_free(req, FDEV_PHYS_WIRED, sizeof(*req));	\
}

/*
 * Read/write data from/to a device.
 */
static int
block_io(int rw, struct inode *inode, struct file *filp, char *buf, int count)
{
	int amt, err, i, nb, nbuf, resid;
	struct buffer_head *bh, **bhp;
	struct peropen *po;
	struct request *req;
	struct semaphore sem;
	unsigned blk, nsect, off, sn;

	if (count < 0)
		return (-LINUX_EINVAL);

	po = inode->i_emul_data;
	resid = count;
	blk = filp->f_pos >> po->bshift;
	off = 0;
	err = 0;
	req = NULL;

	/*
	 * Read/write any partial block to align
	 * the offset on a logical block boundary.
	 * This is required because some drivers
	 * check for it.
	 */
	if (filp->f_pos & po->bmask) {
		sn = blk << (po->bshift - 9);
		amt = ((filp->f_pos + resid + 511) >> 9) - sn;
		if (amt > (po->bsize >> 9))
			amt = po->bsize >> 9;
		if (amt > po->size - sn)
			amt = po->size - sn;
		amt <<= 9;
		bh = bread(inode->i_rdev, blk, amt);
		if (!bh)
			return (-LINUX_EIO);
		amt = resid;
		if (amt > bh->b_size - (filp->f_pos & po->bmask))
			amt = bh->b_size - (filp->f_pos & po->bmask);
		if (rw == READ) {
			err = fdev_default_buf_copyout((bh->b_data
						+ (filp->f_pos & po->bmask)),
					       buf, 0, amt);
			if (err)
				err = -LINUX_EFAULT;
		} else {
			err = fdev_default_buf_copyin(buf, 0,
					      (bh->b_data
					       + (filp->f_pos & po->bmask)),
					      amt);
			if (!err) {
				bh->b_state |= 1 << BH_Dirty;
				bh->b_state &= ~(1 << BH_Uptodate);
				ll_rw_block(WRITE, 1, &bh);
				if (!buffer_uptodate(bh))
					err = -LINUX_EIO;
			} else
				err = -LINUX_EFAULT;
		}
		__brelse(bh);
		if (err)
			return (err);
		resid -= amt;
		off += amt;
		blk++;
	}

	/*
	 * Read/write full sectors.
	 */
	if (resid >= 512) {
		/*
		 * Allocate buffers.
		 */
		nsect = resid >> 9;
		nbuf = (nsect + (po->bsize >> 9) - 1) / (po->bsize >> 9);
		if (nbuf > MAX_BUF)
			nbuf = MAX_BUF;
		bhp = fdev_mem_alloc((sizeof(struct buffer_head *)
				      + sizeof(struct buffer_head)) * nbuf,
				     FDEV_PHYS_WIRED, 0);
		if (!bhp)
			return (-LINUX_ENOMEM);
		bh = (struct buffer_head *)(bhp + nbuf);
		for (i = 0; i < nbuf; i++) {
			bhp[i] = bh + i;
			bhp[i]->b_dev = inode->i_rdev;
			bhp[i]->b_wait = NULL;
			bhp[i]->b_data = fdev_mem_alloc(po->bsize,
							BUF_MEM_FLAGS,
							po->bsize);
			if (!bh->b_data) {
				FREE_BUFFERS(i);
				return (-LINUX_ENOMEM);
			}
		}
		req = fdev_mem_alloc(sizeof(*req), FDEV_PHYS_WIRED, 0);
		if (!req) {
			FREE_BUFFERS(nbuf);
			return (-LINUX_ENOMEM);
		}
		do {
			/*
			 * Construct buffer list.
			 */
			req->nr_sectors = 0;
			for (i = 0; i < nbuf && nsect; i++) {
				bhp[i]->b_state = 1 << BH_Lock;
				bhp[i]->b_blocknr = blk + i;
				bhp[i]->b_reqnext = bhp[i] + 1;
				if (po->bsize <= (nsect << 9))
					bhp[i]->b_size = po->bsize;
				else
					bhp[i]->b_size = nsect << 9;
				if (rw == WRITE) {
					bhp[i]->b_state |= 1 << BH_Dirty;
					err = fdev_default_buf_copyin(buf, off,
							      bhp[i]->b_data,
							      bhp[i]->b_size);
					if (err) {
						FREE_BUFFERS(nbuf);
						return (-LINUX_EFAULT);
					}
					off += bhp[i]->b_size;
				}
				req->nr_sectors += bhp[i]->b_size >> 9;
				nsect -= bhp[i]->b_size >> 9;
				resid -= bhp[i]->b_size;
			}
		

			bhp[i-1]->b_reqnext = NULL;
			nb = i;

			/*
			 * Construct request.
			 */
			req->sector = blk << (po->bshift - 9);
			req->current_nr_sectors = bhp[0]->b_size >> 9;
			req->buffer = bhp[0]->b_data;
			req->rq_status = RQ_ACTIVE;
			req->rq_dev = inode->i_rdev;
			req->cmd = rw;
			req->errors = 0;
			req->sem = &sem;
			req->bh = bhp[0];
			req->bhtail = bhp[nb - 1];
			req->next = NULL;
			sem.count = 0;
			sem.wait = NULL;

			blk += nb;

			/*
			 * Enqueue request and wait for I/O to complete.
			 */
			enqueue_request(req);
			__down(&sem);
			if (req->errors) {
				FREE_BUFFERS(nbuf);
				return (-LINUX_EIO);
			}

			if (rw == READ) {
				for (i = 0; i < nb; i++) {
					err = fdev_default_buf_copyout(bhp[i]->b_data,
							       buf, off,
							       bhp[i]->b_size);
					if (err) {
						FREE_BUFFERS(nbuf);
						return (-LINUX_EFAULT);
					}
					off += bhp[i]->b_size;
				}
			}
		} while (nsect);

		FREE_BUFFERS(nbuf);
	}

	/*
	 * Read/write final partial count.
	 */
	if (resid) {
		bh = bread(inode->i_rdev, blk, 512);
		if (!bh)
			return (-LINUX_EIO);
		if (rw == READ) {
			err = fdev_default_buf_copyout(bh->b_data, buf, off, resid);
			if (err)
				err = -LINUX_EFAULT;
		} else {
			err = fdev_default_buf_copyin(buf, off, bh->b_data, resid);
			if (!err) {
				bh->b_state |= 1 << BH_Dirty;
				bh->b_state &= ~(1 << BH_Uptodate);
				ll_rw_block(WRITE, 1, &bh);
				if (!buffer_uptodate(bh))
					err = -LINUX_EIO;
			} else
				err = -LINUX_EFAULT;
		}
		__brelse(bh);
	}

	return (err);
}

/*
 * Generic routine to read from a block device.
 */
int
block_read(struct inode *inode, struct file *file, char *buf, int count)
{
	return (block_io(READ, inode, file, buf, count));
}

/*
 * Generic routine to write to a block device.
 */
int
block_write(struct inode *inode, struct file *file, const char *buf, int count)
{
	return (block_io(WRITE, inode, file, buf, count));
}

/*
 * Register a block driver.
 */
int
register_blkdev(unsigned major, const char *name, struct file_operations *fops)
{
	printf("REGISTER_BLKDEV: major = %d, name = %s\n",
	       major, name);

	if (major == 0) {
		for (major = MAX_BLKDEV - 1; major > 0; major--)
			if (!blkdevs[major].fops)
				break;
		if (major == 0)
			return (-LINUX_EBUSY);
	} else {
		if (major >= MAX_BLKDEV)
			return (-LINUX_EINVAL);
		if (blkdevs[major].fops && blkdevs[major].fops != fops)
			return (-LINUX_EBUSY);
	}

	printf("REGISTER_BLKDEV: major = %d, name = %s REGISTERED!\n",
	       major, name);

	blkdevs[major].name = name;
	blkdevs[major].fops = fops;
	return (0);
}

/*
 * Unregister a block driver.
 */
int
unregister_blkdev(unsigned major, const char *name)
{
	if (major >= MAX_BLKDEV
	    || !blkdevs[major].fops
	    || strcmp(blkdevs[major].name, name))
		return (-LINUX_EINVAL);

	blkdevs[major].fops = NULL;
	return (0);
}

/*
 * Allocate a block buffer.
 */
struct buffer_head *
getblk(kdev_t dev, int block, int size)
{
	struct buffer_head *bh;

	bh = fdev_mem_alloc(sizeof(*bh), FDEV_PHYS_WIRED, 0);
	if (!bh)
		return (NULL);
	bh->b_data = fdev_mem_alloc(size, BUF_MEM_FLAGS, size);
	if (!bh->b_data) {
		fdev_mem_free(bh, FDEV_PHYS_WIRED, sizeof(*bh));
		return (NULL);
	}
	bh->b_dev = dev;
	bh->b_size = size;
	bh->b_state = 1 << BH_Lock;
	bh->b_blocknr = block;
	return (bh);
}

/*
 * Release a buffer.
 */
void
__brelse(struct buffer_head *bh)
{
	fdev_mem_free(bh->b_data, BUF_MEM_FLAGS, bh->b_size);
	fdev_mem_free(bh, FDEV_PHYS_WIRED, sizeof(*bh));
}

/*
 * Read data from a device.
 */
struct buffer_head *
bread(kdev_t dev, int block, int size)
{
	struct buffer_head *bh;

	bh = getblk(dev, block, size);
	if (!bh)
		return (NULL);
	ll_rw_block(READ, 1, &bh);
	wait_on_buffer(bh);
	if (!buffer_uptodate(bh)) {
		__brelse(bh);
		return (NULL);
	}
	return (bh);
}

#define NREQ	50

/*
 * Perform I/O request on a list of buffers.
 */
void
ll_rw_block(int rw, int nr, struct buffer_head **bh)
{
	int i, bsize, bshift;
	struct request *req;
	static int initdone = 0;
	static struct request requests[NREQ];	/* XXX */

	if (!initdone) {
		for (i = 0; i < NREQ; i++)
			requests[i].rq_status = RQ_INACTIVE;
		initdone = 1;
	}

	/*
	 * Find an unused request structure.
	 */
	linux_cli();

	while (1) {
		for (i = 0; i < NREQ; i++)
			if (requests[i].rq_status == RQ_INACTIVE)
				break;
		if (i < NREQ)
			break;
		linux_event_block(&wait_for_request);
	}

	linux_sti();

	req = &requests[i];

	/*
	 * Compute device block size.
	 */
	bsize = BLOCK_SIZE;
	if (blksize_size[MAJOR(bh[0]->b_dev)]
	    && blksize_size[MAJOR(bh[0]->b_dev)][MINOR(bh[0]->b_dev)])
		bsize = blksize_size[MAJOR(bh[0]->b_dev)][MINOR(bh[0]->b_dev)];
	for (i = bsize, bshift = 0; i != 1; bshift++, i >>= 1)
		;

	/*
	 * Construct request.
	 */
	for (i = 0, req->nr_sectors = 0; i < nr - 1; i++) {
		req->nr_sectors += bh[i]->b_size >> 9;
		bh[i]->b_reqnext = bh[i + 1];
	}
	req->nr_sectors += bh[i]->b_size >> 9;
	bh[i]->b_reqnext = NULL;
	req->rq_status = RQ_ACTIVE;
	req->rq_dev = bh[0]->b_dev;
	req->cmd = rw;
	req->errors = 0;
	req->sector = bh[0]->b_blocknr << (bshift - 9);
	req->current_nr_sectors = bh[0]->b_size >> 9;
	req->buffer = bh[0]->b_data;
	req->bh = bh[0];
	req->bhtail = bh[nr - 1];
	req->next = NULL;

	/*
	 * Queue request.
	 */
	enqueue_request(req);
}

/*
 * This routine checks whether a removable media has been changed,
 * and invalidates all buffer-cache-entries in that case. This
 * is a relatively slow routine, so we have to try to minimize using
 * it. Thus it is called only upon a 'mount' or 'open'. This
 * is the best way of combining speed and utility, I think.
 * People changing diskettes in the middle of an operation deserve
 * to loose :-)
 */
int
check_disk_change(kdev_t dev)
{
	unsigned i;
	struct file_operations *fops;

	i = MAJOR(dev);
	if (i >= MAX_BLKDEV || (fops = blkdevs[i].fops) == NULL)
		return (0);
	if (fops->check_media_change == NULL)
		return (0);
	if (!(*fops->check_media_change)(dev))
		return (0);

	printf("Disk change detected on device %s\n", kdevname(dev));

	if (fops->revalidate)
		(*fops->revalidate)(dev);

	return (1);
}

/*
 * Called by genhd.c to initialize block devices.
 */
void
blk_dev_init()
{
}
