/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Linux kernel memory allocation.
 */

#ifndef OSKIT
#define OSKIT
#endif

#define OSKIT_INCLUDE

#include <flux/fdev.h>
#include <flux/page.h>
#include <flux/x86/pc/phys_lmm.h>

#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/delay.h>

/*
 * Allocate wired down kernel memory.
 */
void *
linux_kmalloc(unsigned int size, int priority)
{
	int flags;
	void *p;

	(flags
	 = FDEV_AUTO_SIZE|FDEV_PHYS_WIRED|FDEV_VIRT_EQ_PHYS|FDEV_PHYS_CONTIG);
	if (priority & GFP_ATOMIC)
		flags |= FDEV_NONBLOCKING;
	if (priority & GFP_DMA)
		flags |= FDEV_ISA_DMA_MEMORY;
	p = fdev_mem_alloc(size, flags, 0);
	return (p);
}

/*
 * Free kernel memory.
 */
void
linux_kfree(void *p)
{
	fdev_mem_free(p, FDEV_AUTO_SIZE, 0);
}

/*
 * Allocate physically contiguous pages.
 */
unsigned long
__get_free_pages(int priority, unsigned long order, unsigned long max_addr)
{
	int flags;
	void *p;

	flags = FDEV_PHYS_WIRED|FDEV_VIRT_EQ_PHYS|FDEV_PHYS_CONTIG;
	if (priority & GFP_ATOMIC)
		flags |= FDEV_NONBLOCKING;
	if ((priority & GFP_DMA) || (max_addr && max_addr <= 16 * 1024 * 1024))
		flags |= FDEV_ISA_DMA_MEMORY;
	p = fdev_mem_alloc(PAGE_SIZE << order, flags, PAGE_SIZE);
	return ((unsigned long)p);
}

/*
 * Free pages.
 */
void
free_pages(unsigned long addr, unsigned long order)
{
	fdev_mem_free((void *)addr, 0, PAGE_SIZE << order);
}

/*
 * Allocate virtually contiguous memory.
 * The underlying physical pages do not need to be contiguous.
 */
void *
vmalloc(unsigned long size)
{
	int order;

	for (order = 0; (PAGE_SIZE << order) < size; order++)
		;
	return ((void *)__get_free_pages(0, order, ~0UL));
}
