/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Linux timer support.
 */

#include <flux/x86/proc_reg.h>

#ifndef OSKIT
#define OSKIT
#endif

#define OSKIT_INCLUDE

#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/interrupt.h>

unsigned long volatile jiffies = 0;

/*
 * Mask of active timers.
 */
unsigned long timer_active = 0;

/*
 * List of timeout routines.
 */
struct timer_struct timer_table[32];

/*
 * The head for the timer-list has a "expires" field of MAX_UINT,
 * and the sorting routine counts on this..
 */
static struct timer_list timer_head =
{
  &timer_head, &timer_head, ~0, 0, NULL
};

#define SLOW_BUT_DEBUGGING_TIMERS 0

void
add_timer(struct timer_list *timer)
{
	unsigned long flags;
	struct timer_list *p;

#if SLOW_BUT_DEBUGGING_TIMERS
	if (timer->next || timer->prev) {
		printk("add_timer() called with non-zero list from %p\n",
			__builtin_return_address(0));
		return;
	}
#endif
	p = &timer_head;
#if 0
	flags = linux_save_flags();
	linux_cli();
#else
	flags = get_eflags();
	cli();
#endif
	do {
		p = p->next;
	} while (timer->expires > p->expires);
	timer->next = p;
	timer->prev = p->prev;
	p->prev = timer;
	timer->prev->next = timer;
#if 0
	linux_restore_flags(flags);
#else
	set_eflags(flags);
#endif
}

int
del_timer(struct timer_list *timer)
{
	unsigned long flags;
#if SLOW_BUT_DEBUGGING_TIMERS
	struct timer_list * p;

	p = &timer_head;
#if 0
	flags = linux_save_flags();
	linux_cli();
#else
	flags = get_eflags();
	cli();
#endif
	while ((p = p->next) != &timer_head) {
		if (p == timer) {
			timer->next->prev = timer->prev;
			timer->prev->next = timer->next;
			timer->next = timer->prev = NULL;
#if 0
			linux_restore_flags(flags);
#else
			set_eflags(flags);
#endif
			return 1;
		}
	}
	if (timer->next || timer->prev)
		printk("del_timer() called from %p with timer not initialized\n",
			__builtin_return_address(0));
#if 0
	linux_restore_flags(flags);
#else
	set_eflags(flags);
#endif
	return 0;
#else
	struct timer_list * next;
	int ret = 0;
#if 0
	flags = linux_save_flags();
	linux_cli();
#else
	flags = get_eflags();
	cli();
#endif
	if ((next = timer->next) != NULL) {
		(next->prev = timer->prev)->next = next;
		timer->next = timer->prev = NULL;
		ret = 1;
	}
#if 0
	linux_restore_flags(flags);
#else
	set_eflags(flags);
#endif
	return ret;
#endif
}

/*
 * Timer software interrupt handler.
 */
void
timer_bh()
{
	unsigned long mask;
	struct timer_struct *tp;
	struct timer_list * timer;

#if 0
	linux_cli();
#else
	cli();
#endif
	while ((timer = timer_head.next) != &timer_head
	       && timer->expires <= jiffies) {
		void (*fn)(unsigned long) = timer->function;
		unsigned long data = timer->data;

		timer->next->prev = timer->prev;
		timer->prev->next = timer->next;
		timer->next = timer->prev = NULL;
#if 0
		linux_sti();
#else
		sti();
#endif
		fn(data);
#if 0
		linux_cli();
#else
		cli();
#endif
	}
#if 0
	linux_sti();
#else
	sti();
#endif

	for (mask = 1, tp = timer_table; mask; tp++, mask <<= 1) {
		if (mask > timer_active)
			break;
		if ((mask & timer_active)
		    && tp->expires > jiffies) {
			timer_active &= ~mask;
			(*tp->fn)();
#if 0
			linux_sti();
#else
			sti();
#endif
		}
	}
}

/*
 * Timer interrupt handler.
 */
void
linux_timer_intr()
{
	unsigned long mask;
	struct timer_struct *tp;

	jiffies++;

	for (mask = 1, tp = timer_table; mask; tp++, mask += mask) {
		if (mask > timer_active)
			break;
		if (!(mask & timer_active))
			continue;
		if (tp->expires > jiffies)
			continue;
		mark_bh(TIMER_BH);
	}
	if (timer_head.next->expires <= jiffies)
		mark_bh(TIMER_BH);
	if (tq_timer != &tq_last)
		mark_bh(TQUEUE_BH);
}
