/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

#include <string.h>
#include <malloc.h>
#include <flux/x86/multiboot.h>
#include <flux/x86/base_vm.h>
#include <flux/x86/pc/phys_lmm.h>
#include <flux/x86/pc/base_multiboot.h>

struct multiboot_info boot_info;

#define skip(hole_min, hole_max)					\
	if ((max > (hole_min)) && (min < (hole_max)))			\
	{								\
		if (min < (hole_min)) max = (hole_min);			\
		else { min = (hole_max); goto retry; }			\
	}

void base_multiboot_init_mem(void)
{
	vm_offset_t min;
	extern char _start[], end[];

	/* Memory regions to skip.  */
	vm_offset_t cmdline_start_pa = boot_info.flags & MULTIBOOT_CMDLINE
		? boot_info.cmdline : 0;
	vm_offset_t cmdline_end_pa = cmdline_start_pa
		? cmdline_start_pa+strlen((char*)phystokv(cmdline_start_pa))+1
		: 0;

	/* Initialize the base memory allocator
	   according to the PC's physical memory regions.  */
	phys_lmm_init();

	/* Add to the free list all the memory the boot loader told us about,
	   carefully avoiding the areas occupied by boot information.
	   as well as our own executable code, data, and bss.
	   Start at the end of the BIOS data area.  */
	min = 0x500;
	do
	{
		vm_offset_t max = 0xffffffff;

		/* Skip the I/O and ROM area.  */
		skip(boot_info.mem_lower * 1024, 0x100000);

		/* Stop at the end of upper memory.  */
		skip(0x100000 + boot_info.mem_upper * 1024, 0xffffffff);

		/* Skip our own text, data, and bss.  */
		skip(kvtophys(_start), kvtophys(end));

		/* Skip the important stuff the bootloader passed to us.  */
		skip(cmdline_start_pa, cmdline_end_pa);
		if ((boot_info.flags & MULTIBOOT_MODS)
		    && (boot_info.mods_count > 0))
		{
			struct multiboot_module *m = (struct multiboot_module*)
				phystokv(boot_info.mods_addr);
			unsigned i;

			skip(boot_info.mods_addr,
			     boot_info.mods_addr +
			     boot_info.mods_count * sizeof(*m));
			for (i = 0; i < boot_info.mods_count; i++)
			{
				if (m[i].string != 0)
				{
					char *s = (char*)phystokv(m[i].string);
					unsigned len = strlen(s);
					skip(m[i].string, m[i].string+len+1);
				}
				skip(m[i].mod_start, m[i].mod_end);
			}
		}

		/* We actually found a contiguous memory block
		   that doesn't conflict with anything else!  Whew!
		   Add it to the free list.  */
		phys_lmm_add(min, max - min);

		/* Continue searching just past the end of this region.  */
		min = max;

		/* The skip() macro jumps to this label
		   to restart with a different (higher) min address.  */
		retry:
	}
	while (min < 0xffffffff);
}

