/*
 * Copyright (c) 1994 The University of Utah and
 * the Computer Systems Laboratory (CSL).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Wrap the memdebug_alloc, varying the parameters based on which
 * allocator is called.
 */

#include "memdebug.h"

/*
 * Debugging the library is easier if you don't include
 * smalloc() etc, as they get used in a lot of the
 * startup routines in libsac.
 */
#define WRAP_SMALLOC_ETC 1

/*
 * malloc wrapper
 */
void *
memdebug_traced_malloc(size_t bytes, char *file, int line)
{
	return memdebug_alloc(0, bytes, MALLOC_INIT, 
			       MALLOC_CALLER, 
			       file, line);
}

void* 
malloc(size_t bytes)
{
 	return memdebug_traced_malloc(bytes,
				      0, 0);
}

/*
 * memalign wrapper
 */
void *
memdebug_traced_memalign(size_t alignment, size_t size, char *file, int line)
{
	return memdebug_alloc(alignment, size, MALLOC_INIT,
			       MALLOC_CALLER, 
			       file, line);
}

void *
memalign(size_t alignment, size_t size)
{
	return memdebug_traced_memalign(alignment, size, 
					0, 0);
}




/*
 * calloc wrapper.
 */
void* 
memdebug_traced_calloc(size_t n, size_t elem_size, char *file, int line)
{
	return memdebug_alloc(0, n * elem_size, 0, 
			       MALLOC_CALLER,
			       file, line);
}

void* 
calloc(size_t n, size_t elem_size)
{
	return memdebug_traced_calloc(n, elem_size,
				      0, 0);
}



/*
 * free wrapper
 */
void
memdebug_traced_free(void* mem, char *file, int line)
{
	memdebug_free(mem, FREE_WIPE, 
		      MALLOC_CALLER, -1,
		      file, line);
}

void 
free(void* mem)
{
	memdebug_traced_free(mem, 
			     0, 0);
}



/*
 * realloc wrapper.
 */
void* 
memdebug_traced_realloc(void* oldmem, size_t bytes, char *file, int line)
{
	void *newmem = memdebug_traced_malloc(bytes, file, line);
	if (!newmem) return 0;
	if (oldmem)
	{
		memcpy(newmem, oldmem, bytes);
		memdebug_traced_free(oldmem, file, line);
	}
	return newmem;
}
	
void* 
realloc(void* oldmem, size_t bytes)
{
	return memdebug_traced_realloc(oldmem, bytes, 
				       0, 0);
}

#if WRAP_SMALLOC_ETC
/*
 * smalloc wrapper. 
 */
void*
memdebug_traced_smalloc (size_t size, char *file, int line)
{
	return memdebug_alloc(0, size, SMALLOC_INIT, 
			       SMALLOC_CALLER,
			       file, line);
}

void* 
smalloc (size_t size)
{
	return memdebug_traced_smalloc(size, 0, 0);
}



/*
 * sfree wrapper
 */
void 
memdebug_traced_sfree(void* mem, size_t chunksize, char *file, int line)
{
	memdebug_free(mem, SFREE_WIPE, 
		      SMALLOC_CALLER, chunksize,
		      file, line);
}

void 
sfree(void* mem, size_t chunksize)
{
	memdebug_traced_sfree(mem, chunksize,
			      0, 0);
}


/*
 * smemalign wrapper
 */
void *
memdebug_traced_smemalign(size_t alignment, size_t size, char *file, int line)
{
	return memdebug_alloc(alignment, size, SMALLOC_INIT,
			       SMALLOC_CALLER, 
			       file, line);
}

void *
smemalign(size_t alignment, size_t size)
{
	return memdebug_traced_smemalign(alignment, size, 
					 0, 0);
}
#endif
