#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/mm.h>

#include <asm/page.h> /* __PAGE_OFFSET */

#include <l4/syscalls.h>
#include <l4/kdebug.h>
#include <services/server/pfs.h>
#include <services/client/dm_anon.h>

#include <services/lib/nameservice.h>
#include <services/lib/sm_handle.h>

#include <services/sys/services.h>

#ifdef PFS_ON_LINUX
#include <stdio.h>

#include <unistd.h>
#include <sys/mman.h>
#include <errno.h>
#endif

#ifdef PFS_ON_LINUX
int fd;
char buffer[4096];
ssize_t size;
#endif

#define START_MEM __MEM_START
#define SIZE_MEM  0x00600000

#define RD_IMAGE  0x20000000
#define RD_SIZE   0x00400000

#define MAX_HANDLES 512

#define BUFFER_SIZE 4096

void* ramdisk_image;

unsigned long main_stack[8192];

struct task_struct _current;
struct task_struct *current = &_current;
struct fs_struct current_fs;

extern kdev_t ROOT_DEV;
int root_mountflags = 0;

sm_service_t dm_anon;
dsid_t anon_ds_id;

/* 
 * new stuff since we don't have a virtual file system 
 */
struct super_block* root_super;

void pfs_thread(void);
void task_init(void);
int init_block_stub(kdev_t dev, char* name, void* service);

int main(void) {
	unsigned long memory_start, memory_end;
	void *start;
	int i;

	memory_start = START_MEM;
	memory_end = START_MEM + SIZE_MEM;
	printk(PFS_EXT2"\n");

	//enter_kdebug("PFS");

	if (!nameservice_register_service(PFS_EXT2))
	{
		printk("Could not register name\n");
		exit(0);
	}
	
	dm_anon = nameservice_get_service(DM_ANON);
	if (l4_is_invalid_id(dm_anon)) 
	{
		printk("Anonymous dataspace not running\n");
	}


	/* now we grab memory from the anonymous ds */
	{
		sm_exc_t exc;
		l4_snd_fpage_t fp;
		/* map memory from anon_dm */
		dm_anon_ds_open(dm_anon, &anon_ds_id, &exc);
		if (flick_is_exception(&exc)) {
			panic("Could not open anonymous dataspace\n");
		}
		
		//enter_kdebug("map anon dm");
		for (i = 0; i<SIZE_MEM; i+=0x1000) {
			dm_anon_ds_map_page(dm_anon, 
					    l4_fpage(START_MEM + i, 12, 0, 0),
					    anon_ds_id, 
					    i,
					    &fp,
					    &exc); 
			if (flick_is_exception(&exc)) {
				printk("error getting page %x for ds %x from anon dm\n", i, anon_ds_id);
				enter_kdebug("grab page");
			}
		}

		//enter_kdebug("map anon dm done");
	}


	memory_start = free_area_init(memory_start, memory_end);
	memory_start = kmem_cache_init(memory_start, memory_end);
	mem_init(memory_start,memory_end);
	kmem_cache_sizes_init();

	dcache_init();
	buffer_init();
	inode_init();
	blk_dev_init();

	rd_init();
	init_block_stub(MKDEV(IDE1_MAJOR, 0), "hdc", 0);

	task_init();
	if (initialize_sm_handle_system(MAX_HANDLES) != SM_OK)
		panic("could not initialize handle system\n");

	/* we have to init the appropriate fs - otherwise read_super fails */
	init_ext2_fs();

#if 0

#ifndef PFS_ON_LINUX
#error "Can't load diskimage without linux"
#else
	fd = open("/home/volkmar/ram1", O_RDONLY);
	if (fd == -1) {
		printk("error opening file\n");
		exit(1);
	}
	
	ramdisk_image = mmap((void*)RD_IMAGE, RD_SIZE, PROT_READ, 
			     MAP_FIXED | MAP_PRIVATE, fd, 0);

	if (ramdisk_image == MAP_FAILED) {
		printk("error mapping file");
		exit(1);
	}
	rd_load(ramdisk_image);
#endif

#ifndef PFS_ON_LINUX
#warning unmap the ramdisk image
#else
	munmap(ramdisk_image, RD_SIZE);
	close(fd);
#endif
	ROOT_DEV = MKDEV(RAMDISK_MAJOR, 0);
#else

	ROOT_DEV = MKDEV(IDE1_MAJOR, 0);

#endif /* 0 */



	mount_root();

	printk("super block: root: %x\n", root_super->s_root);

	printk("PFS: do the main loop\n");
	
	pfs_thread();
	
	exit(1);
}


/* this is the main handler-thread function loop - whatever */
void pfs_thread(void) 
{
    sm_request_t request;
    l4_ipc_buffer_t msgbuf;
    l4_msgdope_t result;
    int ret;
    dword_t buffer[BUFFER_SIZE / 4];
    
    printk("starting thread %x (buffer: %x)\n", l4_myself(), buffer);

    flick_init_request(&request, &msgbuf);
    flick_set_number_rcvstring(&request, 1);
    flick_server_set_rcvstring(&request, 0, BUFFER_SIZE, buffer);
    
    request.local_data = buffer;

    while(1) {
	result = flick_server_wait(&request);
	while(!L4_IPC_ERROR(result)) {
	    if (request.msgbuf->buffer[0] < req_pfs_file_write)
		    ret = pfs_name_server(&request);
	    else
		    ret = pfs_file_server(&request);

	    flick_server_set_rcvstring(&request, 0, BUFFER_SIZE, buffer);

	    switch(ret) {
	    case DISPATCH_ACK_SEND:
		result = flick_server_reply_and_wait(&request);
		break;
	    case DISPATCH_PROPAGATE:
		panic("propagation not implemented yet");
		break;
	    case DISPATCH_NO_REPLY:
		result = flick_server_wait(&request);
		break;
	    default:
		panic("something screwed with the server-loop");
		break;
	    }
	}
	printk("pfsext2: flick ipc error (%x)\n", result);
	enter_kdebug("pfsext2 ipc err");
    }
}


void task_init()
{
	current->rlim[RLIMIT_FSIZE] = (struct rlimit){LONG_MAX, LONG_MAX};
	current->fs = &current_fs;
	current->fs->umask = 0666;
}

void exit(int code)
{
	panic("exit %d", code);
	while(1);
}
	
