#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/dcache.h>
#include <linux/mm.h>
#include <linux/smp_lock.h>

#include <linux/ext2_fs.h>
#include <linux/dcache.h>
#include <linux/malloc.h>

#include <services/server/pfs.h>
#include <pfs/pfs.h>

#include <l4/kdebug.h>

#include <services/lib/sm_handle.h>


#define FOBJ_TO_INODE(var) (struct inode*)(var)

#define MSG_BUF_SIZE  4096

/**********************************************************************
 *
 * some helpers
 *
 */
#define NAME_OFFSET(de) ((int) ((de)->d_name - (char *) (de)))
#define ROUND_UP(x) (((x)+sizeof(long)-1) & ~(sizeof(long)-1))

struct linux_dirent {
	unsigned long	d_ino;
	unsigned long	d_off;
	unsigned short	d_reclen;
	char		d_name[1];
};

struct getdents_callback {
	struct linux_dirent * current_dir;
	struct linux_dirent * previous;
	int count;
	int error;
};

static int filldir(void * __buf, const char * name, int namlen, off_t offset, ino_t ino)
{
	struct linux_dirent * dirent;
	struct getdents_callback * buf = (struct getdents_callback *) __buf;
	int reclen = ROUND_UP(NAME_OFFSET(dirent) + namlen + 1);

	buf->error = -EINVAL;	/* only used if we fail.. */
	if (reclen > buf->count)
		return -EINVAL;
	dirent = buf->previous;
	if (dirent)
		dirent->d_off = offset;
	dirent = buf->current_dir;
	buf->previous = dirent;
	memcpy(dirent->d_name, name, namlen);
	dirent->d_name[namlen] = 0;
	dirent->d_ino = ino;
	dirent->d_reclen = reclen;
#if 0
	printk("pfs (%x): %s (%x, %d, %d/%d)\n", dirent, name, 
	       dirent->d_name, namlen, reclen, dirent->d_reclen);
#endif
	((char *) dirent) += reclen;
	buf->current_dir = dirent;
	buf->count -= reclen;
	return 0;
}




extern int ext2_readdir(struct file * filp, void * dirent, filldir_t filldir);


/**********************************************************************
 * 
 * the server functions
 *
 */
#if 0
sdword_t pfs_name_server_readdir(sm_request_t *request, pfs_fobj_t dir, sqword_t *pos, dword_t *version, sdword_t *len, sdword_t *error, l4_strdope_t *dirent, sm_exc_t *_ev)
{
	struct file filp;
	struct getdents_callback buf;
	int ret;

	struct dentry dentry;
	dentry.d_inode = iget(root_super, dir);

	if (!dentry.d_inode) 
		return -EBADF;

	filp.f_pos = *pos;
	filp.f_version = *version;
	filp.f_dentry = &dentry;

	// buf is only the temporary descriptor
	// to fill the send-buffer
	buf.current_dir = request->local_data;
	buf.previous = NULL;
	buf.count = *len > MSG_BUF_SIZE ? MSG_BUF_SIZE : *len;
	buf.error = 0;
	
	ret = ext2_readdir(&filp, &buf, pfs_filldir);

	*pos = filp.f_pos;
	*version = filp.f_version;
	*error = buf.error;
	*len = *len - buf.count; /* maxlen - filled bytes => len */
	
	if ((dirent->snd_size = *len)) {
		dirent->snd_str = (dword_t)request->local_data;
	}
	else
		dirent->snd_str = 0;
#if 0
	printk("pfs: send buffer\n");
	for (i=0; i<20; i++) {
		printk("%x ", ((dword_t*)dirent->snd_str)[i]);
	}
#endif
	printk("\npfs: readdir (ret: %d, size: %d, buffer: %x)\n", ret, dirent->snd_size, request->local_data);
	return ret;
}

#endif

sdword_t pfs_name_server_lookup(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, pfs_fobj_t *fobj, sdword_t *ino, sm_exc_t *_ev)
{
	struct inode *idir;
	struct dentry dentry;
	int ret;

	printk("pfs: lookup %x\n", dir);
	idir = iget(root_super, dir);
	if (!idir)
		return -EBADF;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);

	ret = ext2_lookup(idir, &dentry);
	
	/* if it was ok - return the inode number or 0 for not exist */
	if (!ret) {
		if (dentry.d_inode) 
			*ino = dentry.d_inode->i_ino;
		else 
			*ino = 0;
		printk("pfs: lookup returned %x\n", *ino);
	}
	iput(idir);
	return ret;

}

sdword_t pfs_name_server_mknod(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, sdword_t mode, sdword_t rdev, sm_exc_t *_ev)
{
	struct dentry dentry;
	struct inode* inode = iget(root_super, dir);
	int ret;
	printk("pfs: mkdir %s (%x)\n", name, inode);
	if (!inode)
		return -EIO;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);
	ret = ext2_mknod(inode, &dentry, mode, rdev);

	iput(inode);
	return ret;
}

sdword_t pfs_name_server_mkdir(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, sdword_t mode, sm_exc_t *_ev)
{
	int ret;
	struct dentry dentry;
	struct inode* inode = iget(root_super, dir);
	printk("pfs: mkdir %s (%x)\n", name, inode);
	if (!inode)
		return -EIO;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);

	ret = ext2_mkdir(inode, &dentry, mode);

	iput(inode);

	return ret;
}

sdword_t pfs_name_server_rmdir(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, sm_exc_t *_ev)
{
	int ret;
	struct dentry dentry;
	struct inode* inode = iget(root_super, dir);
	printk("pfs: rmdir %s (%x)\n", name, inode);
	if (!inode)
		return -EIO;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);

	ext2_lookup(inode, &dentry);

	ret = ext2_rmdir(inode, &dentry);

	iput(inode);

	return ret;
}

sdword_t pfs_name_server_unlink(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, sm_exc_t *_ev)
{
	return 0;
}

sdword_t pfs_name_server_symlink(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, pfs_name_t symname, sm_exc_t *_ev)
{
	struct dentry dentry;
	struct inode *inode;
	int ret;

	inode = iget(root_super, dir);
	if (!inode)
		return -ENOENT;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);
	
	ret = ext2_symlink(inode, &dentry, symname);

	iput(inode);

	return ret;
}

sdword_t pfs_name_server_link(sm_request_t *request, pfs_fobj_t dir, pfs_fobj_t old_fobj, pfs_name_t name, sm_exc_t *_ev)
{
	struct dentry old_dentry, new_name;
	struct inode *dir_inode;
	int ret;

	ret = -ENOENT;

	old_dentry.d_inode = iget(root_super, old_fobj);
	if (!old_dentry.d_inode)
		return ret;

	dir_inode = iget(root_super, dir);
	if (!dir_inode)
		goto out;

	new_name.d_name.name = name;
	new_name.d_name.len = strlen(name);

	ret = ext2_link(&old_dentry, dir_inode, &new_name);
 out:
	if (old_dentry.d_inode) iput(old_dentry.d_inode);
	if (dir_inode) iput(dir_inode);

	return ret;
}

sdword_t pfs_name_server_rename(sm_request_t *request, pfs_fobj_t old_dir, pfs_name_t old_name, pfs_fobj_t new_dir, pfs_name_t new_name, sm_exc_t *_ev)
{
	struct inode *inode_old, *inode_new;
	struct dentry dentry_old, dentry_new;
	int ret = -ENOENT;

	inode_old = iget(root_super, old_dir);
	inode_new = iget(root_super, new_dir);
	if (!(inode_old && inode_new)) goto out;

	dentry_old.d_name.name = old_name;
	dentry_old.d_name.len = strlen(old_name);
	dentry_new.d_name.name = new_name;
	dentry_new.d_name.len = strlen(new_name);

	ret = ext2_rename(inode_old, &dentry_old, inode_new, &dentry_new);

 out:
	if (inode_old) iput(inode_old);
	if (inode_new) iput(inode_new);
	return ret;
}

sdword_t pfs_name_server_readlink(sm_request_t *request, pfs_fobj_t fobj, pfs_name_t *name, sm_exc_t *_ev)
{
	return 0;
}

sdword_t pfs_name_server_readparam(sm_request_t *request, pfs_fobj_t fobj, sdword_t *ino, sdword_t *flags, sdword_t *ctime, sdword_t *atime, sdword_t *mtime, sdword_t *uid, sdword_t *gid, sdword_t *size, sdword_t *mode, sm_exc_t *_ev)
{
	struct inode* inode = iget(root_super, fobj);
	printk("pfs: Read Inode: %x\n", inode);
	if (inode) {
		*ino = inode->i_ino;
		*flags = inode->i_flags;
		*ctime = inode->i_ctime;
		*atime = inode->i_atime;
		*mtime = inode->i_mtime;
		*uid = inode->i_uid;
		*gid = inode->i_gid;
		*size = inode->i_size;
		*mode = inode->i_mode;
		iput(inode);
		return 1;
	}
	return 0;
}

sdword_t pfs_name_server_writeparams(sm_request_t *request, pfs_fobj_t fobj, sdword_t flags, sdword_t ctime, sdword_t atime, sdword_t mtime, sdword_t uid, sdword_t gid, sdword_t size, sm_exc_t *_ev)
{
	struct inode* inode = iget(root_super, fobj);
	printk("pfs: writeparams %x\n", inode);
	if (inode) {
		inode->i_flags = flags;
		inode->i_ctime = ctime;
		inode->i_atime = atime;
		inode->i_mtime = mtime;
		inode->i_uid = uid;
		inode->i_gid = gid;
		/* inode->i_size = size; */
		ext2_write_inode(inode);
		iput(inode);
		return 1;
	}
	return 0;
}


sdword_t pfs_name_server_get_super(sm_request_t *request, pfs_fobj_t *fobj, sm_exc_t *_ev)
{
	printk("pfs: get_super\n");
	if (root_super->s_root && root_super->s_root->d_inode) {
		*fobj = root_super->s_root->d_inode->i_ino;
		return 1;
	}
	return 0;

}

void pfs_name_server_write_super(sm_request_t *request, sm_exc_t *_ev)
{
	ext2_write_super(root_super);
}

void pfs_name_server_put_super(sm_request_t *request, sm_exc_t *_ev)
{
	ext2_put_super(root_super);
}

sdword_t pfs_name_server_sync(sm_request_t *request, sm_exc_t *_ev)
{
	enter_kdebug("pfs: sync");
	return sys_sync();
}


sdword_t pfs_file_server_write(sm_request_t *request, pfs_fobj_t handle, sdword_t *pos, sdword_t len, l4_strdope_t data, sm_exc_t *_ev)
{
	ssize_t ret;
	struct file * file;
	struct inode * inode;

	lock_kernel();

	ret = -EBADF;

	file = (struct file*)attached_obj_sm_handle((sm_handle_t)handle);
	file->f_pos = *pos;

	printk("pfs: write: client: %x, handle: %x, file: %x, mode: %x\n", 
	       request->client_tid, handle, file, file ? file->f_mode : 0);
	printk("pfs: write: buffer: %x, size: %d\n", 
	       data.rcv_str, data.snd_size);

	if (!file)
		goto bad_file;
	if (!(file->f_mode & FMODE_WRITE))
		goto out;

	inode = file->f_dentry->d_inode;
#if 0
	ret = locks_verify_area(FLOCK_VERIFY_WRITE, inode, file,
				file->f_pos, count);
	if (ret)
		goto out;
#endif
	down(&inode->i_sem);
	ret = ext2_file_write(file, (void*)data.rcv_str, data.snd_size, 
			      &file->f_pos);
	*pos = file->f_pos;
	up(&inode->i_sem);
out:
bad_file:
	unlock_kernel();
	return ret;
}

sdword_t pfs_file_server_read(sm_request_t *request, pfs_fobj_t handle, sdword_t *pos, sdword_t *len, l4_strdope_t *data, sm_exc_t *_ev)
{
	ssize_t ret;
	struct file * file;
	struct inode* dummy;

	lock_kernel();

	data->snd_size = 0;

	ret = -EBADF;

	file = (struct file*)attached_obj_sm_handle((sm_handle_t)handle);

	printk("pfs: read: client: %x, handle: %x, file: %x, mode: %x\n", 
	       request->client_tid, handle, file, file ? file->f_mode : 0);

	if (!file)
		goto bad_file;
	if (!(file->f_mode & FMODE_READ)) 
		goto out;

	file->f_pos = *pos;
#if 0
	/* locks are currently ignored */
	ret = locks_verify_area(FLOCK_VERIFY_READ, file->f_dentry->d_inode,
				file, file->f_pos, count);
	if (!ret) 
#endif
	{
		if (*len > MSG_BUF_SIZE) *len = MSG_BUF_SIZE;
		ret = generic_file_read(file, flick_server_get_local(request), 
					*len, &file->f_pos);


		if (ret > 0) {
			data->snd_str = (dword_t)flick_server_get_local(request);
			data->snd_size = ret;
			data->rcv_size = 0;
		}
		*pos = file->f_pos;
	}
out:
bad_file:
	unlock_kernel();
	return ret;
}

sdword_t pfs_file_server_release(sm_request_t *request, pfs_fobj_t handle, sm_exc_t *_ev)
{
	return 0;
}

sdword_t pfs_file_server_truncate(sm_request_t *request, pfs_fobj_t handle, sm_exc_t *_ev)
{
	return 0;
}

sdword_t pfs_file_server_sync(sm_request_t *request, pfs_fobj_t handle, sm_exc_t *_ev)
{
	struct file * file;
	struct dentry * dentry;
	struct inode * inode;
	int err;

	enter_kdebug("pfs: sync");

	lock_kernel();
	err = -EBADF;
	file = (struct file*)attached_obj_sm_handle((sm_handle_t)handle);
	if (!file)
		goto out;

	dentry = file->f_dentry;
	if (!dentry)
		goto out_putf;

	inode = dentry->d_inode;
	if (!inode)
		goto out_putf;

	/* We need to protect against concurrent writers.. */
	down(&inode->i_sem);
	err = ext2_sync_file(file, dentry);
	up(&inode->i_sem);

out_putf:
out:
	unlock_kernel();
	return err;
	
}


sdword_t pfs_name_server_open(sm_request_t *request, client_t client, pfs_fobj_t fobj, sdword_t flags, sdword_t mode, handle_t *handle, sm_exc_t *_ev)
{
	struct file *filep;
	struct inode* inode;
	int ret;
	sm_handle_t smh;

	inode = iget(root_super, fobj);
	if (!inode)
		return -EEXIST;

	filep = (struct file*) kmalloc(sizeof(struct file), 0);
	filep->f_dentry = (struct dentry*) kmalloc(sizeof(struct dentry), 0);
	filep->f_mode = mode;
	filep->f_flags = flags;
	filep->f_dentry->d_inode = inode;
	
	printk("pfs: open: client: %x, inode: %x, file: %x, mode: %x\n", 
	       client, inode, filep, mode);

	ret = ext2_open_file(inode, filep);

	if (ret)
		goto errout;

	smh = new_sm_handle(client, 0, filep);
	if (!smh) {
		ret = -ENOMEM;
		goto errout;
	}
	*handle = (handle_t)smh;

	return ret;

 errout:
	iput(inode);
	kfree(filep->f_dentry);
	kfree(filep);
	return ret;

}

sdword_t pfs_name_server_create(sm_request_t *request, pfs_fobj_t dir, pfs_name_t name, sdword_t mode, pfs_fobj_t *fobj, sm_exc_t *_ev)
{
	struct inode *inode;
	struct dentry dentry;
	int ret;

	printk("pfs: create %s\n", name);

	inode = iget(root_super, dir);
	if (!inode) 
		return -ENOENT;

	dentry.d_name.name = name;
	dentry.d_name.len = strlen(name);

	ret = ext2_create(inode, &dentry, mode);

	if (!ret) 
		*fobj = dentry.d_inode->i_ino;
	
	iput(inode);
	return ret;
}


sdword_t pfs_name_server_close(sm_request_t *request, handle_t handle, sm_exc_t *_ev)
{
	struct file * filep;
	int ret;
	struct inode* inode;

	filep = (struct file*)attached_obj_sm_handle((sm_handle_t)handle);
	inode = filep->f_dentry->d_inode;
	printk("pfs: close %x (inode: %x)\n", handle, inode);
	free_sm_handle((sm_handle_t)handle);

	ret = ext2_release_file(inode, filep);

	iput(inode);
	kfree(filep->f_dentry);
	kfree(filep);
	return ret;
}


/*
 * New, all-improved, singing, dancing, iBCS2-compliant getdents()
 * interface. 
 */

sdword_t pfs_file_server_getdents(sm_request_t *request, handle_t handle, sdword_t *pos, dword_t count, l4_strdope_t *data, sm_exc_t *_ev)
{
	struct file * file;
	struct dentry * dentry;
	struct inode * inode;
	struct linux_dirent * lastdirent;
	struct getdents_callback buf;
	int error;

	
       
	lock_kernel();
	error = -EBADF;

	file = (struct file*)attached_obj_sm_handle((sm_handle_t)handle);
	if (!file)
		goto out;

	dentry = file->f_dentry;
#if 0
	printk("getdents: dentry: %x\n", dentry);
	enter_kdebug("getdents");
#endif
	if (!dentry)
		goto out_putf;

	inode = dentry->d_inode;
	if (!inode)
		goto out_putf;

	if (count > MSG_BUF_SIZE) 
		count = MSG_BUF_SIZE;

	buf.current_dir = (struct linux_dirent *) flick_server_get_local(request);
	buf.previous = NULL;
	buf.count = count;
	buf.error = 0;

	file->f_pos = *pos;

	/* check for dir entry */
#if 0
	error = -ENOTDIR;
	if (!file->f_op || !file->f_op->readdir)
		goto out_putf;
#endif
	

	/*
	 * Get the inode's semaphore to prevent changes
	 * to the directory while we read it.
	 */
	down(&inode->i_sem);
	error = ext2_readdir(file, &buf, filldir);
	up(&inode->i_sem);
	if (error < 0)
		goto out_putf;
	error = buf.error;
	lastdirent = buf.previous;
	if (lastdirent) {
		error = count - buf.count;
		data->snd_str = flick_server_get_local(request);
		data->snd_size = error;
		*pos = &lastdirent->d_off;
	}
	else
		data->snd_size = 0;
out_putf:
out:
	unlock_kernel();
	return error;
}


