#define sm_service_t SmService
#define sm_request_t SmRequest
#define sm_exc_t     SmExcpt

#include <l4/types.h>
#include <l4/syscalls.h>
#include <l4/ipc.h>
#include <l4/kdebug.h>
#include <l4/librmgr.h>

#include <flick/link/l4.h>
#include <services/server/rm.h>
#include <services/client/genericdm.h>
#include <services/client/decodelf.h>
#include <services/client/rmm.h>
#include <services/client/taskserv.h>

#include <services/lib/utils.h>
#include <services/lib/rm.h>
#include <services/lib/decodelf.h>
#include <services/lib/nameservice.h>
#include "assert.h"

typedef struct regionInfo_s regionInfo_t;
struct regionInfo_s {
  sm_service_t dsmgr_service; /* the dataspace manager for this region */
  unsigned objId; /* id given by dataspace manager */
  unsigned start, size; /* describes region (not dataspace) in our addrspace */
  unsigned dsoffset; /* the region "window"'s position in the dataspace */
  unsigned access;
  regionInfo_t *prev, *next;
};

#define D_OPEN 0
#define D_CLOSE 1
#define D_DSFAULT 2
#define MAX_REGIONS 64
#define PF_X        0x00000001
#define PF_W        0x00000002
#define PF_R        0x00000004

static int isCreatedTask = 0;
static int startup_stack[1024];
static int fwdUnknownPfaults = 1;
static int use_gdb_stub;
int rm_stack[1024];
static regionInfo_t *regionInfoHead, *regionInfoTail;
static regionInfo_t regionInfoStorage[MAX_REGIONS];
static dataspace_t init_ds;
dataspace_t rm_init_ds;
static l4_threadid_t rmmid;
combuf_t combuf;
sm_service_t taskservice;

regionInfo_t *
allocRegionInfoStruct()
{
  int i;
  for (i = 0; i < MAX_REGIONS; i++)
    if (regionInfoStorage[i].prev == NULL
	&& regionInfoStorage[i].next == NULL
        && &regionInfoStorage[i] != regionInfoHead)
      break;

  if (i == MAX_REGIONS) {
    /* we're in trouble :( */
    enter_kdebug ("region mapper: too many regions attached");
    halt();
    return NULL;
  }
  
  regionInfoStorage[i].prev = regionInfoStorage[i].next = NULL;

  return &regionInfoStorage[i];
}

void 
freeRegionInfoStruct(regionInfo_t *info)
{
  /*enter_lock(&regionInfoLock);*/
  info->prev = info->next = NULL;
  /*leave_lock(&regionInfoLock);*/
}

int
addRegion (l4_threadid_t dsmgr, unsigned objId, unsigned start, unsigned size, 
		unsigned dsoffset, unsigned access)
{
  regionInfo_t *newRegion, *tempptr;
  newRegion = allocRegionInfoStruct();
  /*enter_lock(&regionInfoListLock);*/
  if (regionInfoHead == NULL) {
    /* we are putting the first struct into the list */
    regionInfoHead = regionInfoTail = newRegion;
    newRegion->next = newRegion->prev = NULL;
  } else {
    /* find the right spot in the list */
    for (tempptr = regionInfoHead; tempptr != NULL; tempptr = tempptr->next) {
      if (tempptr->start+tempptr->size > start)
	break;
    }
    if (tempptr == NULL) {
      /* this region belongs at the tail of the list */
      regionInfoTail->next = newRegion;
      newRegion->prev = regionInfoTail;
      regionInfoTail = newRegion;
    } else if (tempptr->start < start + size) {
      /* this region overlaps an existing region */
      /*leave_lock(&regionInfoListLock);*/
      freeRegionInfoStruct (newRegion);
      return -1;
    } else {
      /* put this region in the middle of the list */
      newRegion->prev = tempptr->prev;
      newRegion->next = tempptr;
      if (newRegion->prev != NULL)
        newRegion->prev->next = newRegion;
      else
        regionInfoHead = newRegion;
      tempptr->prev = newRegion;
    }
  }
  newRegion->dsmgr_service = dsmgr;
  newRegion->objId = objId;
  newRegion->start = start;
  newRegion->size = size;
  newRegion->dsoffset = dsoffset;
  newRegion->access = access;
  /*leave_lock(&regionInfoListLock);*/
  dbgprintf("rm%d: add region: dsmgr=0x%x,objId=0x%x,start=0x%x,size=0x%x\n",
           isCreatedTask, dsmgr.dw, objId, start, size);
  if (size == 0 && start == 0) {
    enter_kdebug("bad region getting attached");
    halt();
  }
  return 0;
}

void
clearMappings (regionInfo_t *ptr)
{
  /* TODO: reduce the number of system calls by using bigger fpages */
  unsigned i;
  for (i = ptr->start; i < ptr->start + ptr->size; i+=4096) {
    l4_fpage_unmap(l4_fpage(i,12,0,0), L4_FP_ALL_SPACES | L4_FP_FLUSH_PAGE);
  }
}

int
unmapRegion (unsigned start)
{
  regionInfo_t *tempptr;
  for (tempptr = regionInfoHead; tempptr != NULL; tempptr = tempptr->next)
    if (tempptr->start == start) {
      clearMappings(tempptr);
      if (tempptr->prev != NULL)
        tempptr->prev->next = tempptr->next;
      else
        regionInfoHead = tempptr->next;
      if (tempptr->next != NULL)
        tempptr->next->prev = tempptr->prev;
      else
        regionInfoTail = tempptr->prev;
      freeRegionInfoStruct (tempptr);
      return 0;
    }
  /* should also unmap l4 mappings */
  return 1; /* region not found */
}

int
unmapDataspace (l4_threadid_t dsmgr, unsigned objId)
{
  regionInfo_t *tempptr, *nextptr;
  int result = 1;
  for (tempptr = regionInfoHead; tempptr != NULL; tempptr = nextptr) {
    nextptr = tempptr->next;
    if (tempptr->dsmgr_service.dw == dsmgr.dw && tempptr->objId == objId) {
      clearMappings(tempptr);
      if (tempptr->prev != NULL)
        tempptr->prev->next = tempptr->next;
      else
        regionInfoHead = tempptr->next;
      if (tempptr->next != NULL)
        tempptr->next->prev = tempptr->prev;
      else
        regionInfoTail = tempptr->prev;
      freeRegionInfoStruct (tempptr);
      result = 0;
    }
  }
  /* should also unmap l4 mappings */
  return result; 
}

int
lookupRegion (unsigned addr, regionInfo_t *info)
{
  regionInfo_t *tempptr;
  /*enter_lock(&regionInfoListLock);*/
  for (tempptr = regionInfoHead; tempptr != NULL; tempptr = tempptr->next)
    if ((tempptr->start <= addr) && (tempptr->start + tempptr->size > addr))
      break;
  if (tempptr != NULL)
    *info = *tempptr; /* we've gotta copy (or lock individual structs)
		       * due to synchronisation probs */
  /*leave_lock(&regionInfoListLock);*/
  return (tempptr == NULL);
}

void __exit(void);

void 
startup_code ()
{
  segmentInfo_t *infopage;
  int i;

  dbgprintf ("startup_code: alive, init_ds={dsm:%x.%x,dsid:%x}\n", 
           ((l4_threadid_t)init_ds.dsm).id.task,
           ((l4_threadid_t)init_ds.dsm).id.lthread, init_ds.dsid);
  
  addRegion ((l4_threadid_t)init_ds.dsm, init_ds.dsid, 0x180000, 0x1000, 
             0, PF_R);
  
  infopage = (segmentInfo_t *) 0x180000;
  for (i = 0; infopage[i].destbase != -1 && infopage[i].destbase != -2; i++) {
    dbgprintf ("startup: attaching {dsm:%x.%x,dsid:%x} to %x size %x\n",
                ((l4_threadid_t)init_ds.dsm).id.task,
                ((l4_threadid_t)init_ds.dsm).id.lthread, infopage[i].dsid,
                infopage[i].destbase, infopage[i].size);
    addRegion ((l4_threadid_t)init_ds.dsm, infopage[i].dsid, 
               infopage[i].destbase, infopage[i].size, 0, infopage[i].access);
  }

  
  if (infopage[i].destbase == -2) {
    l4_threadid_t myself;
    myself = l4_myself();
    gdbstub_thread_startstop_fn (myself, 1);
    use_gdb_stub = 1;
    asm volatile ("int $3"); /* breakpoint */
  }
  
  dbgprintf ("startup_code: calling main at 0x%x\n", infopage[i].size);

  /* start program */
  i = ((int(*)(int argc, char *argv[])) infopage[i].size) (0, NULL);
  if (use_gdb_stub)
    tell_gdb_byebye(i);
  __exit();
}

extern void _start, _end;
extern int mem_err; /* flag for gdbstub */
extern void (*volatile mem_fault_routine)(void); /* flag in gdbstub */

void
rm_pgr_server_page_fault(sm_request_t *request, sdword_t fault_addr,
                     sdword_t fault_eip/*, l4_snd_fpage_t *map_fpage*/,
                     sm_exc_t *_ev)
{
  int flag_writefault, ret;
  l4_snd_fpage_t rcvfpage;
  sm_exc_t exc;
  regionInfo_t info;

  flag_writefault = fault_addr & 2;
  fault_addr &=~ 3;
  if (isCreatedTask)
    dbgprintf ("rm: got %s fault at addr=0x%x, eip=0x%x\n", 
             flag_writefault ? "write" : "read",
             fault_addr, fault_eip);
/*  if (isCreatedTask)
    enter_kdebug ("rm: got page fault"); */

  ret = lookupRegion (fault_addr, &info);
  if (!ret && ((!flag_writefault) || (info.access & PF_W))) {
    unsigned fault_dsoffset;
    dbgprintf ("rm%d: fwding fault to dsm at %x.%x\n", isCreatedTask,
             info.dsmgr_service.id.task, info.dsmgr_service.id.lthread);
    fault_dsoffset = fault_addr - info.start + info.dsoffset;
    fault_dsoffset &=~ 0xfff;
    fault_addr &=~ 0xfff;
    genericdm_dm_map_page (info.dsmgr_service, l4_fpage (fault_addr, 12, 0, 0),
                           info.objId, fault_dsoffset | (2*!!flag_writefault),
                           &rcvfpage, &exc);
    dbgprintf ("rm: rcvfpage = (base 0x%x, fpage 0x%x) exc=0x%x\n", 
               rcvfpage.snd_base, rcvfpage.fpage.fpage, exc);
    if (exc._type != exc_l4_no_exception) {
      hprintf ("rm: unfulfilled %s fault at addr=0x%x, eip=0x%x thr=%x.%x\n", 
               flag_writefault ? "write" : "read",
               fault_addr, fault_eip, request->client_tid.id.task,
               request->client_tid.id.lthread);
      hprintf ("rm: dsm=%x.%x, dsid=0x%x, offset=0x%x; exc=0x%x\n",
               info.dsmgr_service.id.task, info.dsmgr_service.id.lthread,
               info.objId, fault_dsoffset, exc._type);
      _ev->_type = -3;
      if (use_gdb_stub && 
          (request->client_tid.id.lthread == 3 && mem_fault_routine != NULL)) {
          /* if the gdbstub faulted, and it was prepared for the possibility
             of faulting on this memory access .. */
          unsigned old_eflags, old_eip, old_esp;
          l4_threadid_t pager, preempter;
          mem_err = 1; /* set flag for gdbstub */
          pager.dw = preempter.dw = -1;
          l4_thread_ex_regs (request->client_tid, -1, -1, &preempter, &pager,
                             &old_eflags, &old_eip, &old_esp);
          pager.dw = preempter.dw = -1;
          /* now kick gdbstub alive */
          l4_thread_ex_regs (request->client_tid,
                          /* assume gdbstub faulted in set_char or get_char */
                             old_eip + (flag_writefault?2:3),
                             -1, &preempter, &pager,
                             &old_eflags, &old_eip, &old_esp);
      } else if (use_gdb_stub && request->client_tid.id.lthread != 3) {
          static char errmsgbuf[160];
          hsprintf (errmsgbuf,
                    "*** dataspace %s fault unfulfilled by dsm=%x.%x\n"
                    "    for handle=0x%x, offset=0x%x\n",
                    flag_writefault ? "write" : "read",
                    info.dsmgr_service.id.task, info.dsmgr_service.id.lthread,
                    info.objId, fault_dsoffset);
          make_thread_do_exception_wmesg (request->client_tid, errmsgbuf);
      } else {
        enter_kdebug ("rm: pfault unfulfilled");
        if (isCreatedTask)
          __exit();
        else
          halt();
      }
    }
  } else if ((ret && fwdUnknownPfaults) ||
             (((unsigned)fault_addr < (unsigned)&_end) &&
              (fault_addr >= (unsigned)&_start)) ||
	     /* L4 kernel info page */
	     (fault_addr >= 0x1000 && fault_addr < 0x2000))
  {
    if (flag_writefault)
      *(volatile char *) fault_addr = 0;
    else
      *(volatile char *) fault_addr;
  } else {
    _ev->_type = -3;
    hprintf ("rm: bad %s fault at 0x%x eip=0x%x thread=%x.%x\n",
             flag_writefault ? "write" : "read",
             fault_addr, fault_eip, request->client_tid.id.task,
             request->client_tid.id.lthread);
    if (use_gdb_stub && 
          (request->client_tid.id.lthread == 3 && mem_fault_routine != NULL)) {
        unsigned old_eflags, old_eip, old_esp;
        l4_threadid_t pager, preempter;
        mem_err = 1; /* set flag for gdbstub */
        pager.dw = preempter.dw = -1;
        l4_thread_ex_regs (request->client_tid, -1, -1, &preempter, &pager,
                           &old_eflags, &old_eip, &old_esp);
        pager.dw = preempter.dw = -1;
        /* now kick gdbstub alive */
        l4_thread_ex_regs (request->client_tid, old_eip+3, -1, &preempter,
                           &pager, &old_eflags, &old_eip, &old_esp);
    } else if (use_gdb_stub && request->client_tid.id.lthread != 3)
        make_thread_do_exception(request->client_tid);
    else {
      enter_kdebug ("rm: bad fault");
      if (isCreatedTask)
        __exit();
      else
        halt();
    }
  }
}

void rm_pgr_server_getbuf(sm_request_t *request, sdword_t *bufptr, sm_exc_t *_ev)
{
  dbgprintf ("rm: getbuf\n");
  *bufptr = (sdword_t) &combuf;
}

void rm_pgr_server_attach(sm_request_t *request, sm_exc_t *_ev)
{
  int ret;
  dbgprintf ("rm: attach dsmgr=%x,objid=%x,start=%x,size=%x,dsoffset=%x\n",
           combuf.dsmgr, combuf.objId, combuf.start, combuf.size,
           combuf.dsoffset);
  ret = addRegion (combuf.dsmgr, combuf.objId, combuf.start, combuf.size,
                   combuf.dsoffset, combuf.access);
  if (ret != 0)
    _ev->_type = 42;
}

void rm_pgr_server_detachregion(sm_request_t *request, sm_exc_t *_ev)
{
  int ret;
  dbgprintf("rm: detach start=%x\n", combuf.start);
  ret = unmapRegion (combuf.start);
  if (ret != 0)
    _ev->_type = 42;
}

void rm_pgr_server_detachdataspace(sm_request_t *request, sm_exc_t *_ev)
{
  int ret;
  dbgprintf("rm: detach dsmgr=0x%x, objid=0x%x\n",
          combuf.dsmgr, combuf.objId);
  ret = unmapDataspace (combuf.dsmgr, combuf.objId);
  if (ret != 0)
    _ev->_type = 42;
}

void __die_here (void);

void __exit(void)
{
  sm_exc_t exc;
  l4_threadid_t id;
  id = l4_myself();
  genericdm_dm_close ((sm_service_t)init_ds.dsm, init_ds.dsid, &exc);
  rmm_pgr_stop_paging_rm (rmmid, id.dw, &exc);
  halt();
  enter_kdebug ("__exit: unreachable");
}

void rm_pgr_server_exit(sm_request_t *request, sm_exc_t *_ev)
{
  __exit();
}

void rm_pgr_server_kill(sm_request_t *request, sm_exc_t *_ev)
{
  __exit();
}

void rm_pgr_server_init_thread_debugging (sm_request_t *request,
                                          sdword_t *startstop_fn,
                                          sdword_t getinfo_fn, sm_exc_t *_ev)
{
  if (use_gdb_stub)
    gdbstub_init_thread_debugging(startstop_fn, getinfo_fn);
}

void
rm_main_thread(int fwd_unknown_pfaults)
     /* This thread should be established as the pager for each 
      * non-region-mapper thread  */
{
  l4_threadid_t myid, myPager, myPreempter;
  int dummy;

  sm_request_t request;
  l4_ipc_buffer_t requestbuf;
  l4_msgdope_t msgdope;
  int i;

  fwdUnknownPfaults = fwd_unknown_pfaults;
  myid = l4_myself();
  myPreempter = myPager = L4_INVALID_ID;
  l4_thread_ex_regs(myid, (dword_t)-1, (dword_t)-1, &myPreempter, &myPager,
		    &dummy, &dummy, &dummy);
		    
  dbgprintf ("rm%d alive: threadid = 0x%x, my pager = 0x%x\n",
	   isCreatedTask, myid.dw, myPager.dw);

  rmmid = myPager;

  flick_init_request (&request, &requestbuf);

  msgdope = flick_server_wait(&request);
  if (L4_IPC_IS_ERROR(msgdope)) {
    hprintf("rm: flick_server_wait failed, result=0x%x\n", msgdope);
    enter_kdebug ("rm: wait failed");
  }

  while (3) {
/*    hprintf ("rm: got request from 0x%x, msgdope = 0x%x\n", 
            request.client_tid, request.ipc_status); */
/*    hprintf ("rm: msgbuf: %08x %08x %08x %08x\n",
      request.msgbuf->buffer[0], request.msgbuf->buffer[1],
      request.msgbuf->buffer[2], request.msgbuf->buffer[3]);*/
#if 0
    if (use_gdb_stub && !were_using_gdb_stub) {
      set_debug_traps();
      were_using_gdb_stub = 1;
    }
#endif
    i = rm_pgr_server(&request);

    switch(i) {
    case DISPATCH_ACK_SEND:
      msgdope = flick_server_reply_and_wait (&request);
      break;
    case DISPATCH_NO_REPLY:
      dbgprintf("rm: not replying to 0x%x\n", request.client_tid);
    case DISPATCH_PROPAGATE:
    case FLICK_OPERATION_PROPAGATING:
      msgdope = flick_server_wait(&request);
      break;
    default:
      hprintf("rm: confused by return %d from rm_pgr_server\n", i);
      enter_kdebug("rm: confused by rm_pgr_server");
    }
    /* should check msgdope here */
  }
}

void
task_entry_pt()
{
  l4_threadid_t myid;

  myid = l4_myself();
  hprintf ("new task alive: task 0x%x, version %d\n", 
           myid.id.task, myid.id.version_low);

  isCreatedTask = 2;
  init_ds = rm_init_ds;

  create_thread (8, startup_code, &startup_stack[1022], l4_myself());

  rm_main_thread(0); /* this function shouldn't return */

  enter_kdebug ("elfdm:unreachable");
  halt();

  asm volatile
    (
     /* startup code */
     ".globl __real_entry_pt\n\t"
     "__real_entry_pt:\n\t"
     "lea rm_stack,%%esp\n\t"
     "add $4080,%%esp\n\t"
     "jmp task_entry_pt"
     : /* output regs */
     : /* input regs */
     );
}

