#define sm_service_t SmService
#define sm_request_t SmRequest
#define sm_exc_t     SmExcpt

#include <l4/syscalls.h>
#include <services/server/rmm.h>
#include <services/client/rm.h>
#include <services/lib/rm.h>
#include <l4/librmgr.h>
#include <services/lib/utils.h>
#include <services/lib/cow.h>
#include <services/lib/nameservice.h>
#include <services/client/taskserv.h>

#define MAX_CLIENTS 32
l4_threadid_t clients[MAX_CLIENTS];
sm_service_t taskservice = L4_NIL_ID;

static void
add_client (l4_threadid_t newclient)
{
  int i;
  for (i = 0; i < MAX_CLIENTS; i++)
    if ((clients[i].dw == L4_NIL_ID.dw) ||
        (clients[i].dw == TASKID(newclient).dw))
      break;
  if (i == MAX_CLIENTS)
    enter_kdebug ("rmm: too many clients");
  else
    clients[i] = TASKID(newclient);
}

static void
remove_client (l4_threadid_t oldclient)
{
  int i;
  for (i = 0; i < MAX_CLIENTS; i++)
    if (clients[i].dw == TASKID(oldclient).dw)
      clients[i] = L4_NIL_ID;
  dealloc_copies(TASKID(oldclient).dw);
}

static int
we_page (l4_threadid_t faulter)
{
  int i;
  for (i = 0; i < MAX_CLIENTS; i++)
    if (clients[i].dw == TASKID(faulter).dw)
      return 1;
  return 0;
}

void
RmmMain (void)
{
  sm_request_t request;
  l4_ipc_buffer_t requestbuf;
  l4_threadid_t myid;
  l4_msgdope_t msgdope;
  int i;

  myid = l4_myself();
  hprintf ("rmm: alive, id = 0x%x\n", myid.dw);
  rmgr_init();
  nameservice_init();
  nameservice_register_service("rmm");
  taskservice = nameservice_get_service("taskserv");
  if (taskservice.dw == L4_INVALID_ID.dw) {
    hprintf ("rmm: couldn't find task service\n");
    halt();
  }

  flick_init_request (&request, &requestbuf);

  msgdope = flick_server_wait(&request);
  if (L4_IPC_IS_ERROR(msgdope)) {
    hprintf("rmm: flick_server_wait failed, msgdope=%x\n", msgdope);
    enter_kdebug("wait failed");
  }

  while (3) {
    i = rmm_pgr_server(&request);

    switch(i) {
    case DISPATCH_ACK_SEND:
      msgdope = flick_server_reply_and_wait (&request);
      break;
    case DISPATCH_NO_REPLY:
      msgdope = flick_server_wait(&request);
      break;
    default:
      hprintf("rmm: confused by return from rmm_pgr_server(%d)\n", i);
      enter_kdebug ("rmm: confused by rmm_pgr_server");
    }
    /* should check msgdope here */
  }
}

void rmm_pgr_server_get_rm_start(sm_request_t *request, sdword_t *rm_start_eip,
                             sm_exc_t *_ev)
{
  asm ("leal __real_entry_pt,%0" : "=r"(*rm_start_eip) : );
}

void rmm_pgr_server_stop_paging_rm (sm_request_t *request, threadid_t old_task,
                                    sm_exc_t *_ev)
{
  sm_exc_t exc;
  /* rmgr_delete_task(((l4_threadid_t)old_task).id.task); */
  taskserv_def_delete(taskservice, old_task, &exc);
  remove_client((l4_threadid_t)old_task);
  hprintf ("rmm: killed task 0x%x\n", ((l4_threadid_t)old_task).id.task);
}

void rmm_pgr_server_kill_all_clients (sm_request_t *request, sm_exc_t *_ev)
{
  int i;
  sm_exc_t exc;
  for (i = 0; i < MAX_CLIENTS; i++)
    if (clients[i].dw != L4_NIL_ID.dw) {
      hprintf ("rmm: telling task 0x%x to quit\n", clients[i].id.task);
      rm_pgr_kill ((sm_service_t)clients[i].dw, &exc);
    }
}

extern void __data_start;

void rmm_pgr_server_page_fault(SmRequest *request, sdword_t fault_addr,
                               sdword_t fault_eip, l4_snd_fpage_t *map_fpage,
                               SmExcpt *_ev)
{
  dbgprintf ("rmm: got pfault at %x eip=%x from %x.%x\n", fault_addr,
           fault_eip, request->client_tid.id.task, 
           request->client_tid.id.lthread);

  if (!we_page(request->client_tid)) {
    sm_exc_t exc;
    l4_strdope_t strdope;
    add_client (TASKID(request->client_tid));
    strdope.snd_size = 0;
    strdope.snd_str = 0;
    strdope.rcv_size = sizeof(rm_init_ds);
    strdope.rcv_str = (dword_t) &rm_init_ds;
    taskserv_def_get_pager_info (taskservice, TASKID(request->client_tid).dw,
                                 &strdope, &exc);
    if (exc._type != exc_l4_no_exception) {
      _ev->_type = 42;
      return;
    }
    /* make a copy of the dseg page containing rm_init_ds for this rm */
    get_page_copy ((void *) ((unsigned)&rm_init_ds &~ 4095),
                   TASKID(request->client_tid).dw);
  }

  map_fpage->snd_base = fault_addr &~ 4095;

  if ((fault_addr &~ 4095) == 0x1000) {
    l4_msgdope_t result;
    l4_snd_fpage_t fpage;
    dword_t dummy;
    int error;
    l4_threadid_t pager;
 
    pager.dw = rmgr_id.dw;
    pager.id.lthread = 0;
    error = l4_i386_ipc_call(pager,
                             L4_IPC_SHORT_MSG, 1, 1, 1,
                             L4_IPC_MAPMSG(0x1000, L4_LOG2_PAGESIZE),
                             &fpage.snd_base, &fpage.fpage.fpage, &dummy,
                             L4_IPC_NEVER, &result);
    if (!error && l4_ipc_fpage_received(result)) {
        map_fpage->fpage = l4_fpage(0x1000, 12, 0, 0);
    } else {
        printf("rmm: failed to get kernel info page\n");
    }
  } else if (fault_addr < (unsigned)&__data_start) {
    *(volatile char *) fault_addr; /* ensure page is mapped to us */
    map_fpage->fpage = l4_fpage (fault_addr &~ 4095, 12, 0, 0);
  } else {
    /* copy-on-access data pages */
    char ch;
    void *src_addr;
    src_addr = get_page_copy ((void *) (fault_addr &~ 4095), 
                              TASKID(request->client_tid).dw);
    map_fpage->fpage = l4_fpage ((unsigned) src_addr, 12, 1, 0);
    ch = *(volatile char *) src_addr; /* ensure page is mapped to us .. */
    *(volatile char *) src_addr = ch; /* writeable */
  }
  dbgprintf ("rmm: mapping fpage=0x%x\n", map_fpage->fpage);
}
 
int rmm_stack[1024];

int
main (void)
{
  create_thread (2, RmmMain, &rmm_stack[1022], l4_myself());
  rm_main_thread(1);
  return 0;
}

