//#define ON_LINUX

#ifdef ON_LINUX
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

#include <l4/syscalls.h>

#include <services/server/nameservice.h>

#define MAX_NAME_LEN 100
#define MAX_ENTRIES  100

#define ENTRY_FREE     0
#define ENTRY_VALID    1
#define ENTRY_RESERVED 2

typedef struct name_entry_t {
	char name[MAX_NAME_LEN];
	handle_t handle;
	service_t service;
	int valid;
} name_entry_t;

static name_entry_t names[MAX_ENTRIES];

dword_t main_stack[1024];

name_entry_t* get_free_entry()
{
	int i;
	for (i=0; i<MAX_ENTRIES; i++)
		if (names[i].valid == ENTRY_FREE) {
			names[i].valid = ENTRY_RESERVED;
			return &names[i];
		}
	return NULL;
}

void free_entry(name_entry_t* entry)
{
	entry->valid = 0;
}

name_entry_t* lookup_entry(char* name) 
{
	int i;
	for (i=0; i<MAX_ENTRIES; i++) {
		if (names[i].valid == ENTRY_VALID && 
		    strcmp(name, names[i].name) == 0)
			return &names[i];
	}
	return NULL;
}

void nameservice_thread()
{
    sm_request_t request;
    l4_ipc_buffer_t msgbuf;
    l4_msgdope_t result;
    int ret;
    
    printf("starting thread %x\n", l4_myself());

    flick_init_request(&request, &msgbuf);
    
    while(1) {
	result = flick_server_wait(&request);
	while(!L4_IPC_ERROR(result)) {

	    /* call the server unmarshaller */
	    ret = nameservice_name_server(&request);

	    switch(ret) {
	    case DISPATCH_ACK_SEND:
		result = flick_server_reply_and_wait(&request);
		break;
	    case DISPATCH_PROPAGATE:
		panic("propagation not implemented yet");
		break;
	    case DISPATCH_NO_REPLY:
		result = flick_server_wait(&request);
		break;
	    default:
		panic("something screwed with the server-loop");
		break;
	    }
	}
	printf("nameserver: flick ipc error (%x)", result);
	enter_kdebug("nameserver ipc err");
    }
}

void nameservice_name_server_resolve(sm_request_t *request, nameservice_name_t name, service_t *sm_service, nameservice_name_t *unresolved, handle_t *handle, sm_exc_t *_ev)
{
	name_entry_t *entry = lookup_entry((char*)name);
	if (entry == NULL) {
		flick_throw_exception(exc_nameservice_not_found);
		return;
	}
	*handle = entry->handle;
	*sm_service = entry->service;
	*unresolved = NULL;
}

void nameservice_name_server_register(sm_request_t *request, nameservice_name_t name, handle_t handle, service_t sm_service, sm_exc_t *_ev)
{
	name_entry_t *entry;
	/*
	printf("%x registers %s. handle: %x, service: %x\n", 
	request->client_tid, name, handle, sm_service);*/
	if (lookup_entry(name)) {
		flick_throw_exception(exc_nameservice_exists); 
		return;
	}

	entry = get_free_entry();
	
	if (entry == NULL) {
		flick_throw_exception(exc_nameservice_nomem);
		return;
	}

	strcpy(entry->name, name);
	entry->handle = handle;
	entry->service = sm_service;
	entry->valid = ENTRY_VALID;
}


void nameservice_name_server_unregister(sm_request_t *request, nameservice_name_t name, sm_exc_t *_ev)
{
	name_entry_t *entry;
	entry = lookup_entry(name);
	if (entry == NULL) {
		flick_throw_exception(exc_nameservice_not_found);
		return;
	}
	entry->valid = ENTRY_FREE;
}


int main(void) {
	int fd;
	int i;
	l4_threadid_t tid = l4_myself();
	printf("Sawmill root name-server (%x)\n", tid.dw);
	//enter_kdebug("nameserver init");

#ifdef ON_LINUX
	fd = creat("/tmp/nameserver.id", O_WRONLY);
	if (fd == -1) {
		printf("could not open file\n");
		exit(1);
	}
	write(fd, (void*)&tid, sizeof(tid));
#endif

	for (i = 0; i< MAX_ENTRIES; i++) {
		names[i].valid = ENTRY_FREE;
	}

	nameservice_thread();
}

