/**********************************************************************
 *
 * Volkmar Uhlig, (c) 1999 IBM
 *
 * This is the communication stub to Physical file systems. Instead of
 * having each filesystem in the "linux-way" - we just set up
 * the functions hooks per inode type - to differentiate between
 * directories, files etc.
 *
 **********************************************************************/

#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/sched.h> /* for current */

/* hmm - don't want that really!!! */
#include <linux/ext2_fs.h>

#include <services/client/pfs.h>
#include <l4/kdebug.h>

struct super_block * pfs_read_super (struct super_block * sb, void * data,
				      int silent);


struct inode_operations pfs_dir_inode_operations;
struct inode_operations pfs_symlink_inode_operations;
struct inode_operations pfs_file_inode_operations;
static struct super_operations pfs_sops;



inline sm_service_t get_service(struct file * filp)
{
	return filp->f_dentry->d_inode->i_sb->pfs;
}

static ssize_t pfs_dir_read (struct file * filp, char * buf,
			      size_t count, loff_t *ppos)
{
	return -EISDIR;
}

static struct file_system_type pfs_fs_type = {
	"pfs", 
	FS_REQUIRES_DEV,	/* ibaskets have unresolved bugs */
        pfs_read_super, 
	NULL
};


int init_pfs_fs()
{
        return register_filesystem(&pfs_fs_type);
};

struct linux_dirent {
	unsigned long	d_ino;
	unsigned long	d_off;
	unsigned short	d_reclen;
	char		d_name[1];
};

struct getdents_callback {
	struct linux_dirent * current_dir;
	struct linux_dirent * previous;
	int count;
	int error;
};

struct super_block * pfs_read_super (struct super_block * sb, void * data,
				      int silent)
{
	sm_exc_t exc;
	pfs_fobj_t fobj;
	printk("pfs read superblock (pfs id: %x)\n", sb->pfs.dw);

	if (pfs_name_get_super(sb->pfs, &fobj, &exc)) {
		struct inode* inode;
		sb->fobj  = fobj;
		sb->s_op = &pfs_sops;
		sb->s_root = d_alloc_root(iget(sb, fobj), NULL);
		return sb;
	}
	return 0;
}

static int pfs_readdir(struct file * filp,
		       void * dirent, filldir_t filldir)
{
#if 0
	struct getdents_callback* de = (struct getdents_callback*)dirent;
	int tmp;
	int i;
	int len;
	sm_exc_t exc;
	l4_strdope_t dope;

	dope.snd_str = 0;
	dope.snd_size = 0;
	dope.rcv_str = (dword_t)de->current_dir;
	dope.rcv_size = de->count;
	len = de->count; /* here it is maxlen - when it returns, it contains how much of
			    the buffer is used */

	tmp = pfs_name_readdir(get_service(filp), 
			       filp->f_dentry->d_inode->i_ino, 
			       &filp->f_pos, (dword_t*)&filp->f_version, 
			       &len,&de->error, &dope, &exc);
#if 0
	printk("vfs: returned (%d), len: %x, \nreceive buffer:\n", tmp, len);
	
	for (i=0; i<20; i++) {
		printk("%x ", ((dword_t*)de->current_dir)[i]);
	}
#endif
	// error?
	if (tmp < 0) 
		return tmp;


	
	if (! len) {
	// arg. emulating the linux-grap is a nightmare
	// this is the error case, when we could not read
	// any entry
	// see: fs/readdir.c/sys_getdents(...)
		de->previous = NULL;
		return tmp;
	}

	for (i=0; i < len; i += de->previous->d_reclen) {
		de->previous = de->current_dir; 
		de->current_dir = (struct linux_dirent*)(((char*)(de->current_dir)) + de->current_dir->d_reclen);
		//printk("name: %s, len: %d (%x->%x)\n", de->previous->d_name, de->previous->d_reclen, de->previous, de->current_dir);
	}
		

	//de->previous = de->current_dir /* +  last */;
	//printk("lx: len: %d, last: %d, previous: %x, current: %x\n", rcvbuf.len, rcvbuf.last, de->previous, de->current_dir);
	de->count = de->count - len;

	return tmp;
#else 
	return 0;
#endif
}

void pfs_read_inode (struct inode * inode)
{
	sm_exc_t exc;
	pfs_name_readparam(inode->i_sb->pfs, 
			   inode->i_ino,
			   (sdword_t*)&inode->i_ino, 
			   (sdword_t*)&inode->i_flags,
			   (sdword_t*)&inode->i_ctime, 
			   (sdword_t*)&inode->i_atime, 
			   (sdword_t*)&inode->i_mtime,
			   (sdword_t*)&inode->i_uid, 
			   (sdword_t*)&inode->i_gid, 
			   (sdword_t*)&inode->i_size, 
			   (sdword_t*)&inode->i_mode,
			   &exc);

	if (flick_is_exception(&exc)) {
		printk("exception: %x\n", exc._type);
		enter_kdebug("read_inode exc");
	}
	if (inode->i_ino == EXT2_ACL_IDX_INO ||
	    inode->i_ino == EXT2_ACL_DATA_INO)
		/* Nothing to do */ ;
	else if (S_ISREG(inode->i_mode))
		inode->i_op = &pfs_file_inode_operations;
	else if (S_ISDIR(inode->i_mode))
		inode->i_op = &pfs_dir_inode_operations;
	else if (S_ISLNK(inode->i_mode))
		inode->i_op = &pfs_symlink_inode_operations;
	else 
	{
		printk("Inode mode: %d - should not happen!", inode->i_mode);
		enter_kdebug("vfs: read inode error");
	}
			   
}

void pfs_write_inode (struct inode * inode)
{
	sm_exc_t exc;
	pfs_name_writeparams(inode->i_sb->pfs, 
			     inode->i_ino,
			     (sdword_t)inode->i_flags,
			     (sdword_t)inode->i_ctime, 
			     (sdword_t)inode->i_atime, 
			     (sdword_t)inode->i_mtime,
			     (sdword_t)inode->i_uid, 
			     (sdword_t)inode->i_gid, 
			     (sdword_t)inode->i_size, 
			     &exc);

}

void pfs_put_inode (struct inode * inode)
{

}

/*
 * Called at the last iput() if i_nlink is zero.
 */
void pfs_delete_inode (struct inode * inode)
{

}

void pfs_put_super (struct super_block * sb)
{
	sm_exc_t exc;
	printk("vfs: put_super file %x\n", sb);
	pfs_name_put_super(sb->pfs, &exc);

}

void pfs_write_super (struct super_block * sb)
{
	sm_exc_t exc;
	printk("vfs: write_super file %x\n", sb);
	pfs_name_write_super(sb->pfs, &exc);
}

int pfs_statfs (struct super_block * sb, struct statfs * buf, int bufsiz)
{
	return -EFAULT;
}

int pfs_remount (struct super_block * sb, int * flags, char * data)
{
	return 0;
}

int pfs_ioctl (struct inode * inode, struct file * filp, unsigned int cmd,
	       unsigned long arg)
{
	return -EFAULT;
}


static int pfs_release_file (struct inode * inode, struct file * filp)
{
	sm_exc_t exc;
	int ret;
	printk("vfs: release file %x\n", filp);
	ret = pfs_name_close(inode->i_sb->pfs, filp->handle, &exc);
	return ret;

}

static int pfs_open_file (struct inode * inode, struct file * filp)
{
	handle_t handle;
	sm_exc_t exc;
	int ret;
	printk("vfs: open file\n");
	//enter_kdebug("open");
	ret = pfs_name_open(inode->i_sb->pfs, current->tid.dw, 
			    inode->i_ino, filp->f_flags, 
			    filp->f_mode, &handle, &exc);
	if (!ret) {
		filp->handle = handle;
		filp->service = inode->i_sb->pfs;
		printk("vfs: open: handle: %x, service: %x\n", handle, filp->service.dw);
	}

	return ret;
}

static long long pfs_file_lseek(struct file *file,
				long long offset, int origin)
{
	return offset;
}

static struct dentry * pfs_follow_link(struct dentry * dentry,
				       struct dentry *base,
				       unsigned int follow)
{
	return NULL;
}

static int pfs_readlink (struct dentry * dentry, char * buffer, int buflen)
{
	return 0;
}

int pfs_create (struct inode * dir, struct dentry * dentry, int mode)
{
	sm_exc_t exc;
	int ret;
	int ino;

	printk("vfs: create %s\n", dentry->d_name.name);

	ret = pfs_name_create(dir->i_sb->pfs, dir->i_ino,
			      (char*)dentry->d_name.name,
			      mode, &ino,
			      &exc);

	if (!ret) {
		dentry->d_inode = iget(dir->i_sb, ino);
		printk("vfs: create: get inode %x (%x)\n", 
		       dentry->d_inode, ino);
	}

	return ret;
}

int pfs_mknod (struct inode * dir, struct dentry *dentry, int mode, int rdev)
{
	sm_exc_t exc;
	printk("vfs: mknod %s\n", dentry->d_name.name);

	return pfs_name_mknod(dir->i_sb->pfs, dir->i_ino, 
			      (char*)dentry->d_name.name, 
			      mode, rdev,
			      &exc);
}

int pfs_mkdir(struct inode * dir, struct dentry * dentry, int mode)
{
	sm_exc_t exc;
	printk("vfs: mkdir %s\n", dentry->d_name.name);

	return pfs_name_mkdir(dir->i_sb->pfs, dir->i_ino, (char*)dentry->d_name.name, mode, &exc);
}

int pfs_rmdir (struct inode * dir, struct dentry *dentry)
{
	sm_exc_t exc;
	printk("vfs: rmdir %s\n", dentry->d_name.name);

	return pfs_name_rmdir(dir->i_sb->pfs, dir->i_ino, 
			      (char*)dentry->d_name.name, &exc);
}

int pfs_unlink(struct inode * dir, struct dentry *dentry)
{
	sm_exc_t exc;
	return pfs_name_unlink(dir->i_sb->pfs, dir->i_ino, 
			       (char*)dentry->d_name.name, &exc);
}

int pfs_symlink (struct inode * dir, struct dentry *dentry, const char * symname)
{
	sm_exc_t exc;
	return pfs_name_symlink(dir->i_sb->pfs, dir->i_ino, 
			       (char*)dentry->d_name.name, 
				(char*)symname, 
				&exc);
}

int pfs_link (struct dentry * old_dentry,
		struct inode * dir, struct dentry *dentry)
{
	sm_exc_t exc;
	return pfs_name_link(old_dentry->d_inode->i_sb->pfs,
			     old_dentry->d_inode->i_ino,
			     dir->i_ino,
			     (char*)dentry->d_name.name,
			     &exc);
}

int pfs_rename (struct inode * old_dir, struct dentry *old_dentry,
		 struct inode * new_dir, struct dentry *new_dentry)
{
	sm_exc_t exc;
	return pfs_name_rename(old_dir->i_sb->pfs, 
			       old_dir->i_ino, 
			       (char*)old_dentry->d_name.name,
			       new_dir->i_ino, 
			       (char*)new_dentry->d_name.name,
			       &exc);
			       
}

int pfs_sync_file(struct file * file, struct dentry *dentry)
{
	return 0;
}

int pfs_lookup(struct inode * dir, struct dentry *dentry) 
{
	sm_exc_t exc;
	int tmp;
	int ino;
	int fobj;
	struct inode *inode;

	printk("vfs: pfs_lookup: %s (dir ino: %d)\n", dentry->d_name.name, dir->i_ino);
	tmp = pfs_name_lookup(dir->i_sb->pfs, dir->i_ino, (char*)dentry->d_name.name, &fobj, &ino, &exc);
	if (!tmp) {
		printk("vfs: pfs_lookup inode: %d\n", ino);
		if (ino) {
			inode = iget(dir->i_sb, ino);
			if (!inode) {
				printk("VFS: pfs_lookup: iget returned NULL\n");
				return -EACCES;
			}
		}
		else 
			inode = NULL;

		d_add(dentry, inode);
	}
	return tmp;
}

int pfs_permission (struct inode * inode, int mask)
{
	return 0;
}

void pfs_truncate (struct inode * inode)
{

}

static struct super_operations pfs_sops = {
	pfs_read_inode,
	pfs_write_inode,
	pfs_put_inode,
	pfs_delete_inode,
	NULL,
	pfs_put_super,
	pfs_write_super,
	pfs_statfs,
	pfs_remount
};

static struct file_operations pfs_dir_operations = {
	NULL,			/* lseek - default */
	pfs_dir_read,		/* read */
	NULL,			/* write - bad */
	pfs_readdir,		/* readdir */
	NULL,			/* poll - default */
	pfs_ioctl,		/* ioctl */
	NULL,			/* mmap */
	pfs_open_file,		/* since it is in the pfs - special open */
	NULL,			/* flush */
	pfs_release_file,	/* and also special close */
	pfs_sync_file,		/* fsync */
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};


/*
 * directories can handle most operations...
 */
struct inode_operations pfs_dir_inode_operations = {
	&pfs_dir_operations,	/* default directory file-ops */
	pfs_create,		/* create */
	pfs_lookup,		/* lookup */
	pfs_link,		/* link */
	pfs_unlink,		/* unlink */
	pfs_symlink,		/* symlink */
	pfs_mkdir,		/* mkdir */
	pfs_rmdir,		/* rmdir */
	pfs_mknod,		/* mknod */
	pfs_rename,		/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	pfs_permission,	/* permission */
	NULL			/* smap */
};

static struct file_operations pfs_file_operations = {
	pfs_file_lseek,	        /* lseek */
	pfs_file_read,	        /* read */
	pfs_file_write,	        /* write */
	NULL,			/* readdir - bad */
	NULL,			/* poll - default */
	pfs_ioctl,		/* ioctl */
#if 0
	generic_file_mmap,	/* mmap */
#else
	NULL,                   /* no mmap for the time beeing */
#endif
	pfs_open_file,
	NULL,			/* flush */
	pfs_release_file,	/* release */
	pfs_sync_file,		/* fsync */
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};

struct inode_operations pfs_symlink_inode_operations = {
	NULL,			/* no file-operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	pfs_readlink,		/* readlink */
	pfs_follow_link,	/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL			/* smap */
};

struct inode_operations pfs_file_inode_operations = {
	&pfs_file_operations,/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
#if 0
	generic_readpage,	/* readpage */
#else
	NULL,                   /* for the time beeing... */
#endif
	NULL,			/* writepage */
#if 0
	pfs_bmap,		/* bmap */
#else
	NULL,                   /* no bmap */
#endif
	pfs_truncate,		/* truncate */
	pfs_permission,	        /* permission */
	NULL			/* smap */
};






