#ifndef _LINUX_SCHED_H
#define _LINUX_SCHED_H

extern unsigned long event;


#include <linux/fs.h>
#include <linux/tasks.h>
#include <linux/securebits.h>
#include <linux/capability.h>
#include <l4/types.h>
#include <linux/time.h>

#include <asm/param.h>
#include <asm/current.h>

extern struct timeval xtime;

#define	MAX_SCHEDULE_TIMEOUT	LONG_MAX
#define CURRENT_TIME (xtime.tv_sec)
extern unsigned long volatile jiffies; 
/*
 * Open file table structure
 */
struct files_struct {
	atomic_t count;
	int max_fds;
	struct file ** fd;	/* current fd array */
	fd_set close_on_exec;
	fd_set open_fds;
};

struct fs_struct {
	atomic_t count;
	int umask;
	struct dentry * root, * pwd;
};

struct mm_struct {
};


struct task_struct {
	unsigned long flags;	/* per process flags, defined below */

	pid_t pid;
	pid_t pgrp;
	pid_t tty_old_pgrp;
	pid_t session;

	/* Pointer to task[] array linkage. */
	uid_t uid,euid,suid,fsuid;
	gid_t gid,egid,sgid,fsgid;
	int ngroups;
	gid_t	groups[NGROUPS];

	struct task_struct *next_task, *prev_task;

/* filesystem information */
	struct fs_struct *fs;
	int link_count;

/* open file information */
	struct files_struct *files;

        kernel_cap_t   cap_effective, cap_inheritable, cap_permitted;

#ifdef VFS_SAWMILL
        l4_threadid_t tid;
	atomic_t ref_count;
#endif
        
};

/*
 * Per process flags
 */
#define PF_ALIGNWARN	0x00000001	/* Print alignment warning msgs */
					/* Not implemented yet, only for 486*/
#define PF_STARTING	0x00000002	/* being created */
#define PF_EXITING	0x00000004	/* getting shut down */
#define PF_PTRACED	0x00000010	/* set if ptrace (0) has been called */
#define PF_TRACESYS	0x00000020	/* tracing system calls */
#define PF_FORKNOEXEC	0x00000040	/* forked but didn't exec */
#define PF_SUPERPRIV	0x00000100	/* used super-user privileges */
#define PF_DUMPCORE	0x00000200	/* dumped core */
#define PF_SIGNALED	0x00000400	/* killed by a signal */
#define PF_MEMALLOC	0x00000800	/* Allocating memory */
#define PF_VFORK	0x00001000	/* Wake up parent in mm_release */

#define PF_USEDFPU	0x00100000	/* task used FPU this quantum (SMP) */
#define PF_DTRACE	0x00200000	/* delayed trace (used on m68k, i386) */

extern struct   mm_struct init_mm;
extern struct task_struct *task[NR_TASKS];


/*
 * This has now become a routine instead of a macro, it sets a flag if
 * it returns true (to do BSD-style accounting where the process is flagged
 * if it uses root privs). The implication of this is that you should do
 * normal permissions checks first, and check suser() last.
 *
 * [Dec 1997 -- Chris Evans]
 * For correctness, the above considerations need to be extended to
 * fsuser(). This is done, along with moving fsuser() checks to be
 * last.
 *
 * These will be removed, but in the mean time, when the SECURE_NOROOT 
 * flag is set, uids don't grant privilege.
 */
extern inline int suser(void)
{
	if (!issecure(SECURE_NOROOT) && current->euid == 0) { 
		current->flags |= PF_SUPERPRIV;
		return 1;
	}
	return 0;
}

extern inline int fsuser(void)
{
	if (!issecure(SECURE_NOROOT) && current->fsuid == 0) {
		current->flags |= PF_SUPERPRIV;
		return 1;
	}
	return 0;
}

/*
 * capable() checks for a particular capability.  
 * New privilege checks should use this interface, rather than suser() or
 * fsuser(). See include/linux/capability.h for defined capabilities.
 */

extern inline int capable(int cap)
{
#if 0
#if 1 /* ok now */
	if (cap_raised(current->cap_effective, cap))
#else
	if (cap_is_fs_cap(cap) ? current->fsuid == 0 : current->euid == 0)
#endif
        {
		current->flags |= PF_SUPERPRIV;
		return 1;
	}
	return 0;
#else
	return 1;
#endif
}


/*
 * The wait-queues are circular lists, and you have to be *very* sure
 * to keep them correct. Use only these two functions to add/remove
 * entries in the queues.
 */
extern inline void __add_wait_queue(struct wait_queue ** p, struct wait_queue * wait)
{
	wait->next = *p ? : WAIT_QUEUE_HEAD(p);
	*p = wait;
}

extern rwlock_t waitqueue_lock;

extern inline void add_wait_queue(struct wait_queue ** p, struct wait_queue * wait)
{
	unsigned long flags;

	write_lock_irqsave(&waitqueue_lock, flags);
	__add_wait_queue(p, wait);
	write_unlock_irqrestore(&waitqueue_lock, flags);
}

extern inline void __remove_wait_queue(struct wait_queue ** p, struct wait_queue * wait)
{
	struct wait_queue * next = wait->next;
	struct wait_queue * head = next;
	struct wait_queue * tmp;

	while ((tmp = head->next) != wait) {
		head = tmp;
	}
	head->next = next;
}

extern inline void remove_wait_queue(struct wait_queue ** p, struct wait_queue * wait)
{
	unsigned long flags;

	write_lock_irqsave(&waitqueue_lock, flags);
	__remove_wait_queue(p, wait);
	write_unlock_irqrestore(&waitqueue_lock, flags); 
}
#ifndef VFS_SAWMILL
#define REMOVE_LINKS(p) do { \
	(p)->next_task->prev_task = (p)->prev_task; \
	(p)->prev_task->next_task = (p)->next_task; \
	if ((p)->p_osptr) \
		(p)->p_osptr->p_ysptr = (p)->p_ysptr; \
	if ((p)->p_ysptr) \
		(p)->p_ysptr->p_osptr = (p)->p_osptr; \
	else \
		(p)->p_pptr->p_cptr = (p)->p_osptr; \
	} while (0)

#define SET_LINKS(p) do { \
	(p)->next_task = &init_task; \
	(p)->prev_task = init_task.prev_task; \
	init_task.prev_task->next_task = (p); \
	init_task.prev_task = (p); \
	(p)->p_ysptr = NULL; \
	if (((p)->p_osptr = (p)->p_pptr->p_cptr) != NULL) \
		(p)->p_osptr->p_ysptr = p; \
	(p)->p_pptr->p_cptr = p; \
	} while (0)

#define for_each_task(p) \
	for (p = &init_task ; (p = p->next_task) != &init_task ; )

/* we need this for lookup a task - all tasks are in a queue and 
 * init_task is the head
 */
extern struct task_struct init_task;

#endif

#endif
