#include <l4/types.h>

#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <asm/atomic.h>

#include <vfs/vfs.h>

/* the current-ptr per worker thread */
struct task_struct* __current[WORKER_THREADS];

/* the task-struct per registered client */
struct task_struct* tasks[MAX_TASKS];

#define INIT_FILES { \
        ATOMIC_INIT(1), \
        NR_OPEN, \
        &__init_fd[0], \
        { { 0, } }, \
        { { 0, } } \
}


struct file * __init_fd[NR_OPEN] = { NULL, };
struct files_struct __init_files = INIT_FILES;
struct fs_struct __init_fs;
struct task_struct __init_task;

struct task_struct * get_task(l4_threadid_t tid)
{
	return tasks[tid.id.task];
}


struct task_struct * allocate_taskstruct(l4_threadid_t tid)
{
	struct task_struct *task = get_task(tid);
	if (!task) {
		task = kmalloc(sizeof(struct task_struct), 0);
		task->fs = kmalloc(sizeof(struct fs_struct), 0);
		task->files = kmalloc(sizeof(struct files_struct), 0);
		task->tid = tid;

		atomic_set(&task->files->count, 1);
		task->files->max_fds = NR_OPEN;
		task->files->fd = kmalloc(sizeof(struct file *) * NR_OPEN, 0);
		task->files->fd[0] = NULL;
		task->files->close_on_exec.fds_bits[0] = 0;
		task->files->open_fds.fds_bits[0] = 0;

		task->fs->root = __init_task.fs->root;
		task->fs->pwd = __init_task.fs->pwd;

		tasks[tid.id.task] = task;
	};
	return task;
}

int free_task_struct(l4_threadid_t tid) {
	if (!get_task(tid)) return 0;
	printk("have to free open taskstruct!\n");
	enter_kdebug("vfs: free");
	return 1;
}


void init_task_structs()
{
	int i;

	for (i = 0; i<MAX_TASKS; i++) 
		tasks[i]=NULL;
	
	printk("init task structs\n");

	__current[l4_myself().id.lthread] = &__init_task;
	__init_task.fs = &__init_fs;
	__init_task.files = &__init_files;

	printk("get_current: %x\n", get_current());

	/* for testing purposes I allow myself to use the interface */
	//allocate_taskstruct(l4_myself()); 
}


/* see also linux/asm/current.h */
int set_current(sm_request_t *request)
{
	struct task_struct *task;
	l4_threadid_t me = l4_myself();
#if 1
	task = get_task(request->client_tid);
#else
	task = allocate_taskstruct(request->client_tid);
#endif
	if (!task) return 0;

	/* reference counter */
	atomic_inc(&task->ref_count);

	request->local_data = task;
	__current[me.id.lthread] = task;

	printk("set current to: %x (tid: %x)\n", task, task->tid);

	return 1;
}


