/*
 * Copyright (c) 1995, 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/compiler.h>
#include <mom/c/libcast.h>

#define CHECK_OPT_STRING(str) assert(str != 0);
#define CHECK_STRING(str) assert(str != 0); assert(strlen(str) > 0);

void cast_check_primitive_type(cast_primitive_kind kind, cast_primitive_modifier mod)
{
	if (mod & CAST_MOD_SIGNED) assert(!(mod & CAST_MOD_UNSIGNED));
	if (mod & CAST_MOD_SHORT) assert(!(mod & CAST_MOD_LONG));
	if (mod & CAST_MOD_LONG_LONG) assert(!(mod & CAST_MOD_LONG));
	switch (kind)
	{
		case CAST_PRIM_CHAR:
			assert(!(mod & ~(CAST_MOD_SIGNED | CAST_MOD_UNSIGNED)));
			break;
		case CAST_PRIM_INT:
			break;
		case CAST_PRIM_FLOAT:
			assert(!mod);
			break;
		case CAST_PRIM_DOUBLE:
			assert(!(mod & ~(CAST_MOD_LONG | CAST_MOD_LONG_LONG)));
			break;
		default:
			panic("cast_check_lit_prim: unknown cast_primitive_kind %d", kind);
	}
}

void cast_check_enum(cast_enum_type *et)
{
	int i;

	CHECK_OPT_STRING(et->name);
	assert(et->slots.slots_val);

	for(i = 0; i< (signed int)et->slots.slots_len; i++) {
		CHECK_STRING(et->slots.slots_val[i].name);
		cast_check_expr(et->slots.slots_val[i].val);
	}
}

void cast_check_struct_type(cast_struct_type *stype)
{
	int i;

	CHECK_OPT_STRING(stype->name);
	
	if (stype->slots.slots_len)
		assert(stype->slots.slots_val);
	else
		assert(!stype->slots.slots_val);
	
	for (i = 0; i < (signed int)stype->slots.slots_len; i++)
	{
		CHECK_STRING(stype->slots.slots_val[i].name);
		cast_check_type(stype->slots.slots_val[i].type);
	}
}

void cast_check_union_type(cast_union_type *utype)
{
	int i;

	CHECK_OPT_STRING(utype->name);

	for (i = 0; i < (signed int)utype->cases.cases_len; i++)
	{
		assert(utype->cases.cases_val);
		CHECK_STRING(utype->cases.cases_val[i].name);
		cast_check_type(utype->cases.cases_val[i].type);
	}
}

void cast_check_func_type(cast_func_type *ftype)
{
	int i;

	for (i = 0; i < (signed int)ftype->params.params_len; i++)
	{
		assert(ftype->params.params_val);
		CHECK_STRING(ftype->params.params_val[i].name);
		cast_check_type(ftype->params.params_val[i].type);
	}

	cast_check_type(ftype->return_type);
}

void cast_check_type(cast_type type)
{
	assert(type);
	switch (type->kind)
	{
		case CAST_TYPE_NAME:
		case CAST_TYPE_STRUCT_NAME:
		case CAST_TYPE_UNION_NAME:
		case CAST_TYPE_ENUM_NAME:
			CHECK_STRING(type->cast_type_u_u.name);
			break;
		case CAST_TYPE_PRIMITIVE:
			cast_check_primitive_type(type->cast_type_u_u.primitive_type.kind,
						  type->cast_type_u_u.primitive_type.mod);
			break;
		case CAST_TYPE_STRUCT:
			cast_check_struct_type(&type->cast_type_u_u.struct_type);
			break;
		case CAST_TYPE_UNION:
			cast_check_union_type(&type->cast_type_u_u.union_type);
			break;
		case CAST_TYPE_POINTER:
			cast_check_type(type->cast_type_u_u.pointer_type.target);
			break;
		case CAST_TYPE_ARRAY:
			cast_check_expr(type->cast_type_u_u.array_type.length);
			cast_check_type(type->cast_type_u_u.array_type.element_type);
			break;
		case CAST_TYPE_FUNCTION:
			cast_check_func_type(&type->cast_type_u_u.func_type);
			break;
		case CAST_TYPE_VOID:
			break;
  	        case CAST_TYPE_FPAGE:
		        break;
  	        case CAST_TYPE_L4STRING:
		        break;
		case CAST_TYPE_QUALIFIED:
			cast_check_type(type->cast_type_u_u.qualified.actual);
			break;
		case CAST_TYPE_ENUM:
			cast_check_enum(&type->cast_type_u_u.enum_type);
			break;
		default:
			panic("cast_check_type: unknown cast_type_kind %d", type->kind);
	}
}

void cast_check_expr(cast_expr expr)
{
	assert(expr);
	switch (expr->kind)
	{
		case CAST_EXPR_NAME:
			CHECK_STRING(expr->cast_expr_u_u.name);
			break;
		case CAST_EXPR_LIT_PRIM:
			cast_check_primitive_type(expr->cast_expr_u_u.lit_prim.u.kind,
						  expr->cast_expr_u_u.lit_prim.mod);
			break;
		case CAST_EXPR_LIT_STRING:
			CHECK_OPT_STRING(expr->cast_expr_u_u.lit_string);
			break;
		case CAST_EXPR_CALL:
		{
			cast_expr_call *c = &expr->cast_expr_u_u.call;
			int i;

			cast_check_expr(c->func);
			for (i = 0; i < (signed int)c->params.params_len; i++)
			{
				assert(c->params.params_val);
				cast_check_expr(c->params.params_val[i]);
			}
			break;
		}
		case CAST_EXPR_SEL:
			cast_check_expr(expr->cast_expr_u_u.sel.var);
			CHECK_STRING(expr->cast_expr_u_u.sel.member);
			break;
		case CAST_EXPR_UNARY:
			assert(expr->cast_expr_u_u.unary.op);
			cast_check_expr(expr->cast_expr_u_u.unary.expr);
			break;
		case CAST_EXPR_CAST:
			cast_check_expr(expr->cast_expr_u_u.cast.expr);
			cast_check_type(expr->cast_expr_u_u.cast.type);
			break;
		case CAST_EXPR_SIZEOF_EXPR:
			cast_check_expr(expr->cast_expr_u_u.sizeof_expr);
			break;
		case CAST_EXPR_SIZEOF_TYPE:
			cast_check_type(expr->cast_expr_u_u.sizeof_type);
			break;
		case CAST_EXPR_BINARY:
		case CAST_EXPR_OP_ASSIGN:
			assert(expr->cast_expr_u_u.binary.op);
			cast_check_expr(expr->cast_expr_u_u.binary.expr[0]);
			cast_check_expr(expr->cast_expr_u_u.binary.expr[1]);
			break;
		case CAST_EXPR_COND:
			cast_check_expr(expr->cast_expr_u_u.cond.test);
			cast_check_expr(expr->cast_expr_u_u.cond.true_expr);
			cast_check_expr(expr->cast_expr_u_u.cond.false_expr);
			break;
		default:
			panic("cast_check_expr: unknown cast_expr_kind %d", expr->kind);
	}
}

void cast_check_block(cast_block *block)
{
	int i;

	cast_check(&block->scope);

	for (i = 0; i < (signed int)block->stmts.stmts_len; i++)
	{
		assert(block->stmts.stmts_val);
		cast_check_stmt(block->stmts.stmts_val[i]);
	}
}

void cast_check_stmt(cast_stmt stmt)
{
	assert(stmt);
	switch (stmt->kind)
	{
		case CAST_STMT_EXPR:
			cast_check_expr(stmt->cast_stmt_u_u.expr);
			break;
		case CAST_STMT_BLOCK:
			cast_check_block(&stmt->cast_stmt_u_u.block);
			break;
		case CAST_STMT_IF:
			cast_check_expr(stmt->cast_stmt_u_u.s_if.test);
			cast_check_stmt(stmt->cast_stmt_u_u.s_if.true_stmt);
			cast_check_stmt(stmt->cast_stmt_u_u.s_if.false_stmt);
			break;
		case CAST_STMT_WHILE:
			cast_check_expr(stmt->cast_stmt_u_u.s_while.test);
			cast_check_stmt(stmt->cast_stmt_u_u.s_while.stmt);
			break;
		case CAST_STMT_DO_WHILE:
			cast_check_expr(stmt->cast_stmt_u_u.s_do_while.test);
			cast_check_stmt(stmt->cast_stmt_u_u.s_do_while.stmt);
			break;
		case CAST_STMT_FOR:
			if (stmt->cast_stmt_u_u.s_for.init)
				cast_check_expr(stmt->cast_stmt_u_u.s_for.init);
			if (stmt->cast_stmt_u_u.s_for.test)
				cast_check_expr(stmt->cast_stmt_u_u.s_for.test);
			if (stmt->cast_stmt_u_u.s_for.iter)
				cast_check_expr(stmt->cast_stmt_u_u.s_for.iter);
			cast_check_stmt(stmt->cast_stmt_u_u.s_for.stmt);
			break;
		case CAST_STMT_SWITCH:
			cast_check_expr(stmt->cast_stmt_u_u.s_switch.test);
			cast_check_stmt(stmt->cast_stmt_u_u.s_switch.stmt);
			break;
		case CAST_STMT_BREAK:
			break;
		case CAST_STMT_CONTINUE:
			break;
		case CAST_STMT_GOTO:
			CHECK_STRING(stmt->cast_stmt_u_u.goto_label);
			break;
		case CAST_STMT_LABEL:
			CHECK_STRING(stmt->cast_stmt_u_u.s_label.label);
			cast_check_stmt(stmt->cast_stmt_u_u.s_label.stmt);
			break;
		case CAST_STMT_CASE:
			cast_check_expr(stmt->cast_stmt_u_u.s_case.label);
			cast_check_stmt(stmt->cast_stmt_u_u.s_case.stmt);
			break;
		case CAST_STMT_DEFAULT:
			cast_check_stmt(stmt->cast_stmt_u_u.default_stmt);
			break;
		case CAST_STMT_RETURN:
			cast_check_expr(stmt->cast_stmt_u_u.return_expr);
			break;
		case CAST_STMT_NULL:
			break;
		default:
			panic("cast_check_stmt: unknown cast_stmt_kind %d", stmt->kind);
	}
}

void cast_check_init(cast_init init)
{
	assert(init);
	switch (init->kind)
	{
		case CAST_INIT_EXPR:
		{
			cast_check_expr(init->cast_init_u_u.expr);
			break;
		}
		case CAST_INIT_AGGREGATE:
		{
			int i;
			for (i = 0; i < (signed int)init->cast_init_u_u.subs.subs_len; i++)
			{
				assert(init->cast_init_u_u.subs.subs_val);
				cast_check_init(init->cast_init_u_u.subs.subs_val[i]);
			}
			break;
		}
		default:
			panic("cast_check_init: unknown cast_init_kind %d", init->kind);
	}
}

void cast_check_def(cast_def *def)
{
	assert(def);
	CHECK_OPT_STRING(def->name);
	switch (def->u.kind)
	{
		case CAST_TYPEDEF:
			cast_check_type(def->u.cast_def_u_u.typedef_type);
			break;
		case CAST_STRUCT:
			cast_check_struct_type(&def->u.cast_def_u_u.struct_type);
			break;
		case CAST_UNION:
			cast_check_union_type(&def->u.cast_def_u_u.union_type);
			break;
		case CAST_FUNC_DECL:
			cast_check_func_type(&def->u.cast_def_u_u.func_type);
			break;
		case CAST_FUNC_DEF:
			cast_check_func_type(&def->u.cast_def_u_u.func_def.type);
			cast_check_block(&def->u.cast_def_u_u.func_def.block);
			break;
		case CAST_VAR_DECL:
			assert(def->sc == CAST_SC_EXTERN);
			cast_check_type(def->u.cast_def_u_u.var_type);
			break;
		case CAST_VAR_DEF:
			assert(def->sc != CAST_SC_EXTERN);
			cast_check_type(def->u.cast_def_u_u.var_def.type);
			if (def->u.cast_def_u_u.var_def.init)
				cast_check_init(def->u.cast_def_u_u.var_def.init);
			break;
		case CAST_DEFINE:
			cast_check_expr(def->u.cast_def_u_u.define_as);
			break;
		case CAST_INCLUDE:
			CHECK_STRING(def->u.cast_def_u_u.include.filename);
			break;
		case CAST_DIRECT_CODE:
			CHECK_STRING(def->u.cast_def_u_u.direct.code_string);
			break;
		default:
			panic("cast_check_def: unknown cast_def_kind %d", def->u.kind);
	}
}

void cast_check(cast_scope *scope)
{
	int i;

	for (i = 0; i < (signed int)scope->cast_scope_len; i++)
	{
		assert(scope->cast_scope_val);
		cast_check_def(&scope->cast_scope_val[i]);
	}
}

