/*
 * Copyright (c) 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <string.h>

#include <mom/c/libcast.h>

/*
 * ``Dereference'' a named type by locating and returning its definition.
 * If no definition can be found, return 0.
 *
 * XXX --- This function does not detect mutually-referential type names.
 */
cast_type cast_find_typedef_type(cast_scope *scope, cast_type name_ctype)
{
	cast_type current_ctype = name_ctype;
	
	while (current_ctype
	       && (current_ctype->kind == CAST_TYPE_NAME)) {
		/*
		 * Locate the definition of `current_ctype'.
		 */
		char *current_ctype_name = current_ctype->cast_type_u_u.name;
		unsigned int i;
		cast_type typedef_ctype = 0;
		
		/*****/
		
		for (i = 0; i < scope->cast_scope_len; ++i) {
			cast_def *this_def = &(scope->cast_scope_val[i]);
			
			if ((this_def->u.kind == CAST_TYPEDEF)
			    && !strcmp(this_def->name, current_ctype_name)) {
				typedef_ctype = this_def->u.cast_def_u_u.
						typedef_type;
				break;
			}
		}
		/* Guard against `typedef Foo Foo' insanity. */
		if (typedef_ctype != current_ctype)
			current_ctype = typedef_ctype;
		else
			current_ctype = 0;
	}
	
	return current_ctype;
}

/******************************************************************************
 *
 * The next three functions --- `cast_find_{enum,struct,union}_type' --- are
 * essentially identical, except that they work on different kinds of names.
 * Maybe I should define all three functions using a macro template.
 *
 *****************************************************************************/

/*
 * Return the `cast_enum_type' corresponding to the given C type.  If there is
 * no underlying enumeration definition, return 0.
 */
cast_enum_type *cast_find_enum_type(cast_scope *scope, cast_type ctype)
{
	cast_enum_type *enum_type;
	
	/*****/
	
	/* If `ctype' is a named type, resolve it. */
	ctype = cast_find_typedef_type(scope, ctype);
	
	if (!ctype) {
		/* We failed to resolve the named type. */
		enum_type = 0;
		
	} else if (ctype->kind == CAST_TYPE_ENUM) {
		/* Easy case: we have a real C `enum' type. */
		enum_type = &(ctype->cast_type_u_u.enum_type);
		
	} else if (ctype->kind == CAST_TYPE_ENUM_NAME) {
		/*
		 * Harder case: we have a named enumeration type like `enum
		 * foo'.  The actual enumeration type may be found in a typedef
		 * or in a bare enumeration definition.
		 */
		char *enum_name = ctype->cast_type_u_u.enum_name;
		unsigned int i;
		
		for (i = 0, enum_type = 0;
		     (i < scope->cast_scope_len) && !enum_type;
		     ++i) {
			cast_def *this_def = &(scope->cast_scope_val[i]);
			cast_type typedef_ctype;
			
			switch (this_def->u.kind) {
			case CAST_TYPEDEF:
				typedef_ctype = this_def->u.cast_def_u_u.
						typedef_type;
				if ((typedef_ctype->kind == CAST_TYPE_ENUM)
				    && !strcmp((typedef_ctype->cast_type_u_u.
						enum_type.name),
					       enum_name))
					enum_type
						= &(typedef_ctype->
						    cast_type_u_u.enum_type);
				break;
				
			case CAST_ENUM:
				if (!strcmp((this_def->u.cast_def_u_u.
					     enum_type.name),
					    enum_name))
					enum_type
						= &(this_def->u.cast_def_u_u.
						    enum_type);
				break;
				
			default:
				break;
			}
		}
		
	} else {
		/* The `ctype' is not any kind of enumeration type! */
		enum_type = 0;
	}
	
	return enum_type;
}

/*****************************************************************************/

/*
 * Return the `cast_struct_type' corresponding to the given C type.  If there
 * is no underlying structure definition, return 0.
 */
cast_struct_type *cast_find_struct_type(cast_scope *scope, cast_type ctype)
{
	cast_struct_type *struct_type;
	
	/*****/
	
	/* If `ctype' is a named type, resolve it. */
	ctype = cast_find_typedef_type(scope, ctype);
	
	if (!ctype) {
		/* We failed to resolve the named type. */
		struct_type = 0;
		
	} else if (ctype->kind == CAST_TYPE_STRUCT) {
		/* Easy case: we have a real C `struct' type. */
		struct_type = &(ctype->cast_type_u_u.struct_type);
		
	} else if (ctype->kind == CAST_TYPE_STRUCT_NAME) {
		/*
		 * Harder case: we have a named struct type like `struct foo'.
		 * The actual structure type may be found in a typedef or in a
		 * bare structure definition.
		 */
		char *struct_name = ctype->cast_type_u_u.struct_name;
		unsigned int i;
		
		for (i = 0, struct_type = 0;
		     (i < scope->cast_scope_len) && !struct_type;
		     ++i) {
			cast_def *this_def = &(scope->cast_scope_val[i]);
			cast_type typedef_ctype;
			
			switch (this_def->u.kind) {
			case CAST_TYPEDEF:
				typedef_ctype = this_def->u.cast_def_u_u.
						typedef_type;
				if ((typedef_ctype->kind == CAST_TYPE_STRUCT)
				    && !strcmp((typedef_ctype->cast_type_u_u.
						struct_type.name),
					       struct_name))
					struct_type
						= &(typedef_ctype->
						    cast_type_u_u.struct_type);
				break;
				
			case CAST_STRUCT:
				if (!strcmp((this_def->u.cast_def_u_u.
					     struct_type.name),
					    struct_name))
					struct_type
						= &(this_def->u.cast_def_u_u.
						    struct_type);
				break;
				
			default:
				break;
			}
		}
		
	} else {
		/* The `ctype' is not any kind of structure type! */
		struct_type = 0;
	}
	
	return struct_type;
}

/*****************************************************************************/

/*
 * Return the `cast_union_type' corresponding to the given C type.  If there
 * is no underlying union definition, return 0.
 */
cast_union_type *cast_find_union_type(cast_scope *scope, cast_type ctype)
{
	cast_union_type *union_type;
	
	/*****/
	
	/* If `ctype' is a named type, resolve it. */
	ctype = cast_find_typedef_type(scope, ctype);
	
	if (!ctype) {
		/* We failed to resolve the named type. */
		union_type = 0;
		
	} else if (ctype->kind == CAST_TYPE_UNION) {
		/* Easy case: we have a real C `union' type. */
		union_type = &(ctype->cast_type_u_u.union_type);
		
	} else if (ctype->kind == CAST_TYPE_UNION_NAME) {
		/*
		 * Harder case: we have a named union type like `union foo'.
		 * The actual union type may be found in a typedef or in a bare
		 * union definition.
		 */
		char *union_name = ctype->cast_type_u_u.union_name;
		unsigned int i;
		
		for (i = 0, union_type = 0;
		     (i < scope->cast_scope_len) && !union_type;
		     ++i) {
			cast_def *this_def = &(scope->cast_scope_val[i]);
			cast_type typedef_ctype;
			
			switch (this_def->u.kind) {
			case CAST_TYPEDEF:
				typedef_ctype = this_def->u.cast_def_u_u.
						typedef_type;
				if ((typedef_ctype->kind == CAST_TYPE_UNION)
				    && !strcmp((typedef_ctype->cast_type_u_u.
						union_type.name),
					       union_name))
					union_type
						= &(typedef_ctype->
						    cast_type_u_u.union_type);
				break;
				
			case CAST_UNION:
				if (!strcmp((this_def->u.cast_def_u_u.
					     union_type.name),
					    union_name))
					union_type
						= &(this_def->u.cast_def_u_u.
						    union_type);
				break;
				
			default:
				break;
			}
		}
		
	} else {
		/* The `ctype' is not any kind of union type! */
		union_type = 0;
	}
	
	return union_type;
}

/* End of file. */

