/*
 * Copyright (c) 1995, 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <mom/compiler.h>
#include <mom/c/libcast.h>
#include <stdlib.h>

/* Quick access to definitions in a scope */
#define c(i) (scope->cast_scope_val[i])

static char **types = 0;
static int typesize = 0, typecount = 0;
static char **funcs = 0;
static int funcsize = 0, funccount = 0;

void ClearDefs();
static int DefType(char *t);
static int DefFunc(char *t);

void ClearDefs() {
  int i;
  for (i = 0; i < typecount; i++)
    free(types[i]);
  free(types);
  types = 0;
  typesize = typecount = 0;

  for (i = 0; i < funccount; i++)
    free(funcs[i]);
  free(types);
  funcs = 0;
  funcsize = funccount = 0;
}

static int DefType(char *t) {
  int i;
  for (i = 0; i < typecount; i++)
    if (!strcmp(types[i],t))
      return 1;
  
  if (!typesize) {
    types = (char **)mustcalloc(8 * sizeof(char *));
    typesize = 8;
  }
  if (typesize == typecount) {
    typesize += 8;
    types = (char **)mustrealloc(types, typesize * sizeof(char *));
  }
  types[typecount] = (char *)mustcalloc((strlen(t)+1) * sizeof(char));
  strcpy(types[typecount++],t);
  return 0;
}

static int DefFunc(char *t) {
  int i;
  for (i = 0; i < funccount; i++)
    if (!strcmp(funcs[i],t))
      return 1;
  
  if (!funcsize) {
    funcs = (char **)mustcalloc(8 * sizeof(char *));
    funcsize = 8;
  }
  if (funcsize == funccount) {
    funcsize += 8;
    funcs = (char **)mustrealloc(funcs, funcsize * sizeof(char *));
  }
  funcs[funccount] = (char *)mustcalloc((strlen(t)+1) * sizeof(char));
  strcpy(funcs[funccount++],t);
  return 0;
}

void cast_w_storage_class(cast_storage_class sc) 
{
  switch (sc) {
  case CAST_SC_STATIC:
    w_printf("static ");
    break;
  case CAST_SC_EXTERN:
    w_printf("extern ");
    break;
  case CAST_SC_REGISTER:    
    w_printf("register ");
    break;
  case CAST_SC_NONE:
  case CAST_SC_AUTO:
  default:
    break;
  }
}

  

void cast_w_def(cast_scope *scope, int i, int indent)
{
	switch (c(i).u.kind)
	{
		case CAST_TYPEDEF:
			if (!DefType(c(i).name)) {
				w_i_printf(indent, "#ifndef _typedef___%s\n", c(i).name);
				w_i_printf(indent, "#define _typedef___%s\n", c(i).name);
				w_i_printf(indent, "typedef ");
				cast_w_type(c(i).name, c(i).u.cast_def_u_u.typedef_type, indent);
				w_printf(";\n");
				w_i_printf(indent, "#endif /* _typedef___%s */\n", c(i).name);
			}
			break;
		case CAST_DEFINE:
			w_i_printf(indent, "#define %s (", c(i).name);
			cast_w_expr(c(i).u.cast_def_u_u.define_as,0);
			w_printf(")\n");
			break;
			
		case CAST_INCLUDE:
			w_printf("#include ");
			if (c(i).u.cast_def_u_u.include.system_only)
				w_printf("<%s>\n",
					 c(i).u.cast_def_u_u.include.filename);
			else
				w_printf("\"%s\"\n",
					 c(i).u.cast_def_u_u.include.filename);

			break;

		case CAST_DIRECT_CODE:
			w_printf(c(i).u.cast_def_u_u.direct.code_string);
			break;
			
		case CAST_FUNC_DECL:
			if(!DefFunc(c(i).name)) {
				w_indent(indent);
				cast_w_func_type(c(i).name, &c(i).u.cast_def_u_u.func_type, indent);
				w_printf(";\n");
			}
			break;
		case CAST_FUNC_DEF:
			w_indent(indent);
			cast_w_storage_class(c(i).sc);
			cast_w_func_type(c(i).name, &c(i).u.cast_def_u_u.func_def.type, indent);
			cast_w_block(&c(i).u.cast_def_u_u.func_def.block, indent);
			break;
		case CAST_VAR_DECL:
			w_indent(indent);
			cast_w_storage_class(c(i).sc);
			cast_w_type(c(i).name, c(i).u.cast_def_u_u.var_type, indent);
			w_printf(";\n");
			break;
		case CAST_VAR_DEF:
			w_indent(indent);
			cast_w_storage_class(c(i).sc);
			cast_w_type(c(i).name, c(i).u.cast_def_u_u.var_def.type, indent);
			if (c(i).u.cast_def_u_u.var_def.init)
			{
				w_printf(" = ");
				cast_w_init(c(i).u.cast_def_u_u.var_def.init, indent);
			}
			w_printf(";\n");
			break;
		default:
			panic("unknown cast_def_kind %d", c(i).u.kind);
	}
}

