/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <stdio.h>
#include <mom/compiler.h>
#include <mom/c/libcast.h>

typedef struct qual_node
{
	cast_qualified_type *this_qual;
	struct qual_node *next;
} qual_node;

typedef struct type_node
{
	struct type_node *next;
	void (*print_prefix)(struct type_node *n, int indent);
	void (*print_suffix)(struct type_node *n, int indent);
	int precedence; /* 0=*, 1=(),[] */
	qual_node *quals;
	void *print_func_data;
} type_node;

/*****/

static void type(char *name,
		 cast_type type, type_node *on, qual_node *quals,
		 int indent);


static void print_quals(qual_node *quals)
{
	if (quals) {
		print_quals(quals->next);
		if (quals->this_qual->qual & CAST_TQ_CONST)
			w_printf("const ");
		if (quals->this_qual->qual & CAST_TQ_VOLATILE)
			w_printf("volatile ");
	}
}

static void type_list(char *name, type_node *n, int last_prec, int indent)
{
	if (n) {
		if (n->precedence < last_prec)
			w_putc('(');
		if (n->print_prefix)
			(*n->print_prefix)(n, indent);
		print_quals(n->quals);
		type_list(name, n->next, n->precedence, indent);
		if (n->print_suffix)
			(*n->print_suffix)(n, indent);
		if (n->precedence < last_prec)
			w_putc(')');
	}
	else
	{
		if (name)
			w_printf("%s", name);
	}
}

static void func_type_print_suffix(type_node *n, int indent)
{
	cast_func_type *fd = n->print_func_data;
	
	w_putc('(');
	if (fd->params.params_len > 0)
	{
		int i;

		for (i = 0; i < (signed int)fd->params.params_len; i++)
		{
			if (i > 0) w_printf(", ");
			type(fd->params.params_val[i].name,
			     fd->params.params_val[i].type,
			     0, 0,
			     indent);
		}
	}
	else
		w_printf("void");
	w_putc(')');
}

static void func_type(char *name,
		      cast_func_type *fd, type_node *on, qual_node *quals,
		      int indent)
{
	type_node nn = {on, 0, func_type_print_suffix, 1, quals, fd};

	assert(fd->return_type);
	type(name, fd->return_type, &nn, 0, indent);
}

static void array_type_print_suffix(type_node *n, int indent)
{
	w_putc('[');
	cast_w_expr(n->print_func_data, indent);
	w_putc(']');
}

static void pointer_type_print_prefix(type_node *n, int indent)
{
	w_putc('*');
}


static void type(char *name,
		 cast_type d, type_node *on, qual_node *quals,
		 int indent)
{
	assert(d != 0);
	switch (d->kind)
	{
		case CAST_TYPE_NAME:
		case CAST_TYPE_STRUCT_NAME:
		case CAST_TYPE_UNION_NAME:
		case CAST_TYPE_ENUM_NAME:
		{
			print_quals(quals);
			w_printf("%s%s ",
				 ((d->kind == CAST_TYPE_NAME) ?
				  "" :
				  (d->kind == CAST_TYPE_STRUCT_NAME) ?
				  "struct " :
				  (d->kind == CAST_TYPE_UNION_NAME) ?
				  "union " :
				  (d->kind == CAST_TYPE_ENUM_NAME) ?
				  "enum " :
				  (panic("Unrecognized type name in `type'"),
				   "")),
				 d->cast_type_u_u.name);
			type_list(name, on, 0, indent);
			break;
		}
		case CAST_TYPE_PRIMITIVE:
		{
			cast_primitive_type *pd = &d->cast_type_u_u.primitive_type;
			print_quals(quals);
			if (!pd->name)
				w_printf("%s%s%s ",
					 pd->mod & CAST_MOD_SIGNED ? "signed "
					 : pd->mod & CAST_MOD_UNSIGNED ? "unsigned "
					 : "",
					 pd->mod & CAST_MOD_LONG_LONG ? "long long "
					 : pd->mod & CAST_MOD_LONG ? "long "
					 : pd->mod & CAST_MOD_SHORT ? "short "
					 : "",
					 pd->kind == CAST_PRIM_CHAR ? "char"
					 : pd->kind == CAST_PRIM_INT ? "int"
					 : pd->kind == CAST_PRIM_FLOAT ? "float"
					 : pd->kind == CAST_PRIM_DOUBLE ? "double"
					 : "???");
			else
				w_printf("%s ", *pd->name);
			type_list(name, on, 0, indent);
			break;
		}
		case CAST_TYPE_STRUCT:
		{
			cast_struct_type *sd = &d->cast_type_u_u.struct_type;
			int i;

			print_quals(quals);
			if ((sd->name) && (sd->name[0]))
				w_printf("struct %s {\n", sd->name);
			else
				w_printf("struct {\n");
			for (i = 0; i < (signed int)sd->slots.slots_len; i++)
			{
				w_indent(indent+1);
				type(sd->slots.slots_val[i].name,
				     sd->slots.slots_val[i].type,
				     0, 0,
				     indent+1);
				w_printf(";\n");
			}
			w_indent(indent);
			w_printf("} ");
			type_list(name, on, 0, indent);
			break;
		}
		case CAST_TYPE_UNION:
		{
			cast_union_type *ud = &d->cast_type_u_u.union_type;
			int i;

			print_quals(quals);
			if ((ud->name) && (ud->name[0]))
				w_printf("union %s {\n", ud->name);
			else
				w_printf("union {\n");
			for (i = 0; i < (signed int)ud->cases.cases_len; i++)
			{
				w_indent(indent+1);
				type(ud->cases.cases_val[i].name,
				     ud->cases.cases_val[i].type,
				     0, 0,
				     indent+1);
				w_printf(";\n");
			}
			w_indent(indent);
			w_printf("} ");
			type_list(name, on, 0, indent);
			break;
		}
		case CAST_TYPE_ARRAY:
		{
			cast_array_type *ad = &d->cast_type_u_u.array_type;
			type_node nn = {on, 0, array_type_print_suffix, 1,
					quals, ad->length};
			
			type(name, ad->element_type, &nn, 0, indent);
			break;
		}
		case CAST_TYPE_POINTER:
		{
			type_node nn = {on, pointer_type_print_prefix, 0, 0,
					quals, 0};
			
			type(name, d->cast_type_u_u.pointer_type.target, &nn,
			     0, indent);
			break;
		}
		case CAST_TYPE_FUNCTION:
			func_type(name, &d->cast_type_u_u.func_type, on, quals,
				  indent);
			break;
		case CAST_TYPE_ENUM:
		{
			cast_enum_type *ed = &d->cast_type_u_u.enum_type;
			int i;
			
			print_quals(quals);
			if ((ed->name) && (ed->name[0]))
				w_printf("enum %s {\n", ed->name);
			else
				w_printf("enum {\n");
			for (i = 0; i < (signed int)ed->slots.slots_len; i++) {
				w_i_printf(indent+1,"%s = ",ed->slots.slots_val[i].name);
				cast_w_expr(ed->slots.slots_val[i].val, indent+1);
				if(i == (signed int)(ed->slots.slots_len-1)) {
					w_printf("\n");
					w_i_printf(indent,"} ");
				} else
					w_printf(",\n");
			}
			type_list(name, on, 0, indent);
			break;
		}
		case CAST_TYPE_VOID:
		{
			w_printf("void ");
			type_list(name, on, 0, indent);
			break;
		}
	        case CAST_TYPE_FPAGE:
                {
		        w_printf("l4_snd_fpage_t ");
			type_list(name, on, 0, indent);
			break;
		}
  	        case CAST_TYPE_L4STRING:
                {
		        w_printf("l4_strdope_t ");
			type_list(name, on, 0, indent);
			break;
		}
  	        case CAST_TYPE_INTRFC:
                {
		        w_printf("intrfc_t ");
			type_list(name, on, 0, indent);
			break;
		}
	case CAST_TYPE_QUALIFIED:
	{
		cast_qualified_type *qd = &(d->cast_type_u_u.qualified);
		qual_node qn = { qd, quals };
		
		type(name, qd->actual, on, &qn, indent);
		break;
	}
	
		default:
			panic("unknown type kind %d\n", d->kind);
	}
}

void cast_w_type(char *name, cast_type d, int indent)
{
	type(name, d, 0, 0, indent);
}

void cast_w_func_type(char *name, cast_func_type *fd, int indent)
{
	func_type(name, fd, 0, 0, indent);
}

/* End of file. */

