/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <mom/compiler.h>
#include <mom/libmint.h>
#include <mom/libaoi.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#define CHECK_MINT_REF(ref)					\
	assert((ref) >= 0);					\
	assert((ref) < (signed int) pres->mint.defs.defs_len);

#define CHECK_CAST_REF(ref)					\
	assert((ref) >= 0);					\
	assert((ref) < (signed int) pres->cast.cast_scope_len);

#define CHECK_OPT_STRING(str) assert((str) != 0);

#define CHECK_STRING(str) assert((str) != 0); assert(strlen((str)) > 0);

struct inline_state
{
	int isfunc;
	union {
		struct cast_struct_type *stype;
		struct cast_func_type *ftype;
	} u;
};

static void check_mapping(pres_c_1 *pres, pres_c_mapping map,
			  cast_type ctype, mint_ref itype);

static cast_type *is_type(struct inline_state *is, int slot)
{
	if (is->isfunc)
	{
		if (slot >= 0)
		{
			assert(slot < (signed int)is->u.ftype->params.params_len);
			assert(is->u.ftype->params.params_val);
			return &is->u.ftype->params.params_val[slot].type;
		}
		else
		{
			assert(slot == pres_c_func_return_index);
			return &is->u.ftype->return_type;
		}
	}
	else
	{
		assert(slot >= 0); assert(slot < (signed int)is->u.stype->slots.slots_len);
		assert(is->u.stype->slots.slots_val);
		return &is->u.stype->slots.slots_val[slot].type;
	}
}

static cast_type is_get_type(struct inline_state *is, int slot)
{
	cast_type ctype = *is_type(is, slot);
	assert(ctype != 0);
	return ctype;
}

static void check_alloc_flags(pres_c_alloc_flags flags)
{
}

static void check_alloc(pres_c_allocation *alloc)
{
	check_alloc_flags(alloc->flags);
	if (alloc->flags & (PRES_C_ALLOC_EVER | PRES_C_DEALLOC_EVER))
	{
		CHECK_STRING(alloc->allocator);
	}
	else
	{
		CHECK_OPT_STRING(alloc->allocator);
	}
}

static void check_stub_op_flags(pres_c_stub_op_flags op_flags)
{
	u_int valid_bits = (  PRES_C_STUB_OP_FLAG_ONEWAY
			    | PRES_C_STUB_OP_FLAG_IDEMPOTENT
			    | PRES_C_STUB_OP_FLAG_PROPAGATE
			   );
	
	/* Assert that no invalid flag bits are set. */
	assert((op_flags & ~valid_bits) == 0);
}

static void check_inline(pres_c_1 *pres, pres_c_inline inl,
			 struct inline_state *is, mint_ref itype)
{
	mint_def *idef;

	assert(inl);
	CHECK_MINT_REF(itype);
	idef = &(pres->mint.defs.defs_val[itype]);
	
	switch (inl->kind) {
	case PRES_C_INLINE_ATOM: {
		cast_type ctype
			= is_get_type(is, inl->pres_c_inline_u_u.atom.index);
		
		check_mapping(pres,
			      inl->pres_c_inline_u_u.atom.mapping,
			      ctype,
			      itype);
		break;
	}
	
	case PRES_C_INLINE_STRUCT: {
		pres_c_inline_struct *sinl
			= &(inl->pres_c_inline_u_u.struct_i);
		mint_struct_def *istruct
			= &(idef->mint_def_u.struct_def);
		
		int		slot_index;
		int		pres_c_slot_found;
		mint_ref	this_mint_ref;
		
		int		i, j;
		
		/*
		 * We require that the associated MINT node be a struct type,
		 * and that every slot in that MINT be managed by exactly one
		 * PRES_C_INLINE_STRUCT slot.
		 *
		 * We may have more PRES_C_INLINE_STRUCT slots than MINT_STRUCT
		 * slots; in that case, the ``extra'' PRES_C slots will
		 * necessarily correspond to no MINT representation.
		 */
		assert(idef->kind == MINT_STRUCT);
		
		for (i = 0; i < (signed int) istruct->slots.slots_len; ++i) {
			/*
			 * Find the PRES_C slot for the i'th MINT slot.
			 */
			pres_c_slot_found = 0;
			
			for (j = 0;
			     j < (signed int) sinl->slots.slots_len;
			     ++j) {
				/*
				 * Verify that the j'th PRES_C slot refers to a
				 * valid MINT struct slot, or to no slot.
				 */
				slot_index = (sinl->slots.slots_val[j].
					      mint_struct_slot_index);
				assert(((slot_index >= 0)
					&& (slot_index
					    < ((signed int)
					       istruct->slots.slots_len)))
				       ||
				       (slot_index == mint_slot_index_null));
				
				if (slot_index == i) {
					/*
					 * The j'th PRES_C slot corresponds to
					 * the i'th MINT slot.  Assert that we
					 * haven't yet made a connection to the
					 * i'th MINT slot, and then set the
					 * flag (so that we'll die if we find
					 * another association).
					 */
					assert(!pres_c_slot_found);
					pres_c_slot_found = 1;
				}
			}
			/*
			 * Assert that we found a (single) PRES_C <--> MINT
			 * slot correspondence.
			 */
			assert(pres_c_slot_found);
		}
		
		/*
		 * Now that we know the PRES_C <--> MINT slot associations are
		 * OK, we can descend and check the associated PRES_C inlines.
		 */
		for (i = 0; i < (signed int) sinl->slots.slots_len; ++i) {
			assert(sinl->slots.slots_val[i].inl);
			if (sinl->slots.slots_val[i].mint_struct_slot_index
			    != mint_slot_index_null)
				/*
				 * Default: The PRES_C slot refers to a MINT
				 * structure slot.
				 */
				this_mint_ref
					= (istruct->
					   slots.slots_val[
						   sinl->
						   slots.slots_val[i].
						   mint_struct_slot_index]);
			else
				/*
				 * Special case: The PRES_C slot refers to no
				 * underlying MINT structure slot.  We'll check
				 * the PRES_C inline anyway, by passing down a
				 * reference to our basic MINT_VOID node.
				 */
				this_mint_ref = pres->
						mint.standard_refs.void_ref;
			
			check_inline(pres,
				     sinl->slots.slots_val[i].inl,
				     is,
				     this_mint_ref);
		}
		
		break;
	}
	
	case PRES_C_INLINE_FUNC_PARAMS_STRUCT: {
		pres_c_inline_func_params_struct *fpinl
			= &(inl->pres_c_inline_u_u.func_params_i);
		mint_struct_def *istruct
			= &(idef->mint_def_u.struct_def);
		
		int		slot_index;
		int		pres_c_slot_found;
		mint_ref	this_mint_ref;
		
		int		i, j;
		
		/*
		 * We require that the associated MINT node be a struct type,
		 * and that every slot in that MINT be managed by exactly one
		 * PRES_C_INLINE_FUNC_PARAMS_STRUCT slot.
		 *
		 * We may have more PRES_C_INLINE_FUNC_PARAMS_STRUCT slots than
		 * MINT_STRUCT slots; in that case, the ``extra'' PRES_C slots
		 * will necessarily correspond to no MINT representation.
		 */
		assert(idef->kind == MINT_STRUCT);
		
		for (i = 0; i < (signed int) istruct->slots.slots_len; ++i) {
			/*
			 * Find the PRES_C slot for the i'th MINT slot.
			 */
			pres_c_slot_found = 0;
			
			for (j = 0;
			     j < (signed int) fpinl->slots.slots_len;
			     ++j) {
				/*
				 * Verify that the j'th PRES_C slot refers to a
				 * valid MINT struct slot, or to no slot.
				 */
				slot_index = (fpinl->slots.slots_val[j].
					      mint_struct_slot_index);
				assert(((slot_index >= 0)
					&& (slot_index
					    < ((signed int)
					       istruct->slots.slots_len)))
				       ||
				       (slot_index == mint_slot_index_null));
				
				if (slot_index == i) {
					/*
					 * The j'th PRES_C slot corresponds to
					 * the i'th MINT slot.  Assert that we
					 * haven't yet made a connection to the
					 * i'th MINT slot, and then set the
					 * flag (so that we'll die if we find
					 * another association).
					 */
					assert(!pres_c_slot_found);
					pres_c_slot_found = 1;
				}
			}
			
			/*
			 * PRES_C_INLINE_FUNC_PARAMS_STRUCTs are more compli-
			 * cated than ordinary PRES_C_INLINE_STRUCTs, because
			 * the former has a retun slot that must be checked in
			 * addition to the ``ordinary'' slots.
			 */
			if (fpinl->return_slot) {
				/*
				 * Verify that the `return_slot' refers to a
				 * valid MINT struct slot, or to no slot.
				 */
				slot_index = (fpinl->return_slot->
					      mint_struct_slot_index);
				assert(((slot_index >= 0)
					&& (slot_index
					    < ((signed int)
					       istruct->slots.slots_len)))
				       ||
				       (slot_index == mint_slot_index_null));
				
				if (slot_index == i) {
					/*
					 * The `return_slot' corresponds to the
					 * i'th MINT slot.  Assert that we
					 * haven't yet made a connection to the
					 * i'th MINT slot, and then set the
					 * flag (so that we'll die if we find
					 * another association).
					 */
					assert(!pres_c_slot_found);
					pres_c_slot_found = 1;
				}
			}
			
			/*
			 * Assert that we found a (single) PRES_C <--> MINT
			 * slot correspondence.
			 */
			assert(pres_c_slot_found);
		}
		
		/*
		 * Now that we know the PRES_C <--> MINT slot associations are
		 * OK, we can descend and check the associated PRES_C inlines.
		 */
		for (i = 0; i < (signed int) fpinl->slots.slots_len; ++i) {
			assert(fpinl->slots.slots_val[i].inl);
			if (fpinl->slots.slots_val[i].mint_struct_slot_index
			    != mint_slot_index_null)
				/*
				 * Default: The PRES_C slot refers to a MINT
				 * structure slot.
				 */
				this_mint_ref
					= (istruct->
					   slots.slots_val[
						   fpinl->
						   slots.slots_val[i].
						   mint_struct_slot_index]);
			else
				/*
				 * Special case: The PRES_C slot refers to no
				 * underlying MINT structure slot.  We'll check
				 * the PRES_C inline anyway, by passing down a
				 * reference to our basic MINT_VOID node.
				 */
				this_mint_ref = pres->
						mint.standard_refs.void_ref;
			
			check_inline(pres,
				     fpinl->slots.slots_val[i].inl,
				     is,
				     this_mint_ref);
		}
		if (fpinl->return_slot) {
			assert(fpinl->return_slot->inl);
			if (fpinl->return_slot->mint_struct_slot_index
			    != mint_slot_index_null)
				/*
				 * Default: The PRES_C slot refers to a MINT
				 * structure slot.
				 */
				this_mint_ref
					= (istruct->
					   slots.slots_val[
						   fpinl->return_slot->
						   mint_struct_slot_index]);
			else
				/*
				 * Special case: The PRES_C slot refers to no
				 * underlying MINT structure slot.
				 */
				this_mint_ref = pres->
						mint.standard_refs.void_ref;
			
			check_inline(pres,
				     fpinl->return_slot->inl,
				     is,
				     this_mint_ref);
		}
		
		break;
	}
	
	case PRES_C_INLINE_VIRTUAL_UNION: {
		pres_c_inline_virtual_union *vuinl =
			&(inl->pres_c_inline_u_u.virtual_union);
		mint_union_def *iunion = &idef->mint_def_u.union_def;
		int i;
		assert(idef->kind == MINT_UNION);
		assert(vuinl->cases.cases_len == iunion->cases.cases_len);
		
		for (i = 0; i < (signed int)vuinl->cases.cases_len; i++) {
			if (strcmp(vuinl->cases.cases_val[i].name, ";;;reply"))
			    check_mapping(pres,
					  vuinl->cases.cases_val[i].map,
					  vuinl->cases.cases_val[i].ctype,
					  iunion->cases.cases_val[i].var);
		}
		break;
	}				     
	
	case PRES_C_INLINE_STRUCT_UNION: {
		pres_c_inline_struct_union *suinl =
			&(inl->pres_c_inline_u_u.struct_union);
		
		int mint_union_cases_len;
		cast_type discrim_ctype, union_ctype;
		int case_union_ctype_index;
		
		int i;
		
		/* This mapping must correspond to a MINT_UNION. */
		assert(idef->kind == MINT_UNION);
		
		/* Check the discriminator. */
		discrim_ctype = is_get_type(is, suinl->discrim.index);
		check_mapping(pres,
			      suinl->discrim.mapping,
			      discrim_ctype,
			      idef->mint_def_u.union_def.discrim);
		
		/* Check the number of cases.  Note that the CAST union def may
		   have fewer cases, because some ot the variants of the MINT
		   union may be void. */
		mint_union_cases_len = idef->
				       mint_def_u.union_def.cases.cases_len;
		assert(mint_union_cases_len == (signed int)suinl->cases.cases_len);
		
		/* Check each case. */
		union_ctype = is_get_type(is, suinl->union_index);
		for (i = 0; i < mint_union_cases_len; ++i) {
			/* Check the MINT constant for this case. */
			mint_const_check(idef->mint_def_u.union_def.cases.
					 cases_val[i].val);
			
			/* Can't check the CAST constant - it could be nil */
			
			/* Get the index into the CAST union for this case. */
			case_union_ctype_index = suinl->
						 cases.cases_val[i].index;
			
			/* Check that the index into the CAST union is OK. */
			assert((case_union_ctype_index == -1) ||
			       ((case_union_ctype_index >= 0) &&
				(case_union_ctype_index <
				 (signed int)(union_ctype->
				  cast_type_u_u.union_type.cases.cases_len))
				       ));
			
			/* Check the mapping for this case. */
			if (case_union_ctype_index == -1) {
				/* Do nothing.  Perhaps we should check that
				   the MINT type for this case is void.  But
				   even if it's not, we won't generate bad
				   code. */
			} else
				check_mapping(pres,
					      (suinl->
					       cases.cases_val[i].mapping),
					      (union_ctype->
					       cast_type_u_u.union_type.cases.
					       cases_val
					       [case_union_ctype_index].type),
					      (idef->
					       mint_def_u.union_def.cases.
					       cases_val[i].var)
					);
		}
		
		/* Check the default case, too. */
		if (suinl->dfault) {
			/* Assert that we have a default case in the MINT. */
			assert(idef->mint_def_u.union_def.dfault != -1);
			
			/* Get the index into the CAST union for this case. */
			case_union_ctype_index = suinl->dfault->index;
			
			/* Check that the index into the CAST union is OK. */
			assert((case_union_ctype_index == -1) ||
			       ((case_union_ctype_index >= 0) &&
				(case_union_ctype_index <
				 (signed int)(union_ctype->
				  cast_type_u_u.union_type.cases.cases_len))
				       ));
			
			/* Check the mapping for this case. */
			if (case_union_ctype_index == -1) {
				/* Do nothing.  Perhaps we should check that
				   the MINT type for this case is void.  But
				   even if it's not, we won't generate bad
				   code. */
			} else
				check_mapping(pres,
					      suinl->dfault->mapping,
					      (union_ctype->
					       cast_type_u_u.union_type.cases.
					       cases_val
					       [case_union_ctype_index].type),
					      idef->mint_def_u.union_def.dfault
					);
		} else
			/* No default case in the inline, so assert that there
			   is no default case in the MINT either. */
			assert(idef->mint_def_u.union_def.dfault == -1);
		
		break;
	}
	
	case PRES_C_INLINE_COLLAPSED_UNION: {
		pres_c_inline_collapsed_union *cuinl =
			&inl->pres_c_inline_u_u.collapsed_union;
		mint_ref case_r;
		
		assert(idef->kind == MINT_UNION);
		mint_const_check(cuinl->discrim_val);
		case_r = mint_find_union_case(&pres->mint, itype,
					      cuinl->discrim_val);
		CHECK_MINT_REF(case_r);
		check_inline(pres, cuinl->selected_case, is, case_r);
		break;
	}
	
	case PRES_C_INLINE_TYPED: {
		pres_c_inline_typed *tinl =
			&inl->pres_c_inline_u_u.typed;
		mint_typed_def *ityped = &idef->mint_def_u.typed_def;
		
		assert(idef->kind == MINT_TYPED);
		
		/* check the tag and the data's inlines */
		check_inline(pres, tinl->tag, is, ityped->tag);
		check_inline(pres, tinl->inl, is, ityped->ref);
		break;
	}
	
	case PRES_C_INLINE_COUNTED_ARRAY: {
		pres_c_inline_counted_array *cainl =
			&inl->pres_c_inline_u_u.counted_array;
		mint_array_def *iarray;
		cast_type ctype;
		pres_c_mapping map;
		int i;
		
		assert(idef->kind == MINT_ARRAY
		       || idef->kind == MINT_VOID);
		if (idef->kind == MINT_ARRAY) {
			iarray = &idef->mint_def_u.array_def;
			check_inline(pres, cainl->len, is, iarray->length_type);
			if (cainl->max)
				check_inline(pres, cainl->max, is,
					     iarray->length_type);
			
		} else {
			iarray = 0;
		}
		
		/* Descend through all the levels of pointer indirection
		   represented by this inline node.  */
		ctype = is_get_type(is, cainl->ptr.index);
		map = cainl->ptr.mapping;

		if (map->kind == PRES_C_MAPPING_DIRECTION)
			map = map->pres_c_mapping_u_u.direction.mapping;

		for (i = 0; i <= cainl->ptr_levels; i++) {
			ctype = cast_find_typedef_type(&(pres->cast), ctype);
			assert(ctype);
			/* last pointer (the ptr to data) can be either an
			   array or a pointer. All others must be pointers */
			if (i == cainl->ptr_levels)
				assert(ctype->kind == CAST_TYPE_POINTER ||
				       ctype->kind == CAST_TYPE_ARRAY);
			else
				assert(ctype->kind == CAST_TYPE_POINTER);
			
			if (ctype->kind == CAST_TYPE_ARRAY)
				ctype = ctype->cast_type_u_u.array_type.element_type;
			else
				ctype = ctype->cast_type_u_u.pointer_type.target;
			
			assert(map->kind == PRES_C_MAPPING_POINTER
			       || (map->kind == PRES_C_MAPPING_FIXED_ARRAY));
			check_alloc(&map->pres_c_mapping_u_u.pointer.alloc);
			map = map->pres_c_mapping_u_u.pointer.target;
		}
		
		/* Check the individual element types and mapping.  */
		if (iarray)
		check_mapping(pres, map, ctype, iarray->element_type);
		break;
	}
	
	case PRES_C_INLINE_TERMINATED_ARRAY: {
		pres_c_inline_terminated_array *tainl =
			&inl->pres_c_inline_u_u.terminated_array;
		mint_array_def *iarray = &idef->mint_def_u.array_def;
		cast_type ctype;
		
		assert(idef->kind == MINT_ARRAY);
		if (tainl->max)
			check_inline(pres, tainl->max, is,
				     iarray->length_type);
		ctype = is_get_type(is, tainl->ptr.index);
		assert(ctype->kind == CAST_TYPE_POINTER);
		check_mapping(pres, tainl->ptr.mapping,
			      ctype->cast_type_u_u.pointer_type.target,
			      iarray->element_type);
		check_alloc(&tainl->alloc);
		break;
	}
	
	case PRES_C_INLINE_XLATE: {
		pres_c_inline_xlate *xinl = &inl->pres_c_inline_u_u.xlate;
		cast_type *ctypep = is_type(is, xinl->index);
		cast_type octype = *ctypep;
		
		assert(xinl->internal_ctype);
		*ctypep = xinl->internal_ctype;
		check_inline(pres, xinl->sub, is, itype);
		*ctypep = octype;
		
#if 0
		assert(!(xinl->alloc_flags & ~(PRES_C_DEALLOC_ALWAYS)));
#endif
		check_alloc_flags(xinl->alloc_flags);
		
		CHECK_OPT_STRING(xinl->translator);
		CHECK_OPT_STRING(xinl->destructor);
		break;
	}
	
	case PRES_C_INLINE_ASSIGN: {
		pres_c_inline_assign *ainl = &inl->pres_c_inline_u_u.assign;
		cast_type ctype;
		
		check_inline(pres, ainl->sub, is, itype);
		
		ctype = is_get_type(is, ainl->index);
		assert((ctype->kind == CAST_TYPE_PRIMITIVE)
		       || (ctype->kind == CAST_TYPE_NAME));
		
		cast_check_expr(ainl->value);
		break;
	}
	
	case PRES_C_INLINE_COND: {
		pres_c_inline_cond *cinl = &inl->pres_c_inline_u_u.cond;
		cast_type ctype;
		
		ctype = is_get_type(is, cinl->index);
		assert((ctype->kind == CAST_TYPE_PRIMITIVE)
		       || (ctype->kind == CAST_TYPE_NAME));
		
		check_inline(pres, cinl->true_inl, is, itype);
		check_inline(pres, cinl->false_inl, is, itype);
		
		break;
	}
	
	case PRES_C_INLINE_MESSAGE_ATTRIBUTE: {
		pres_c_inline_message_attribute *msg_attr =
			&(inl->pres_c_inline_u_u.msg_attr);
		assert((msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_FLAGS)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_TIMEOUT)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_SEQUENCE_RECEIVED)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_CLIENT_REFERENCE)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_SERVERCOPY));
		break;
	}
	
	default:
		panic("check_inline: unknown pres_c_inline_kind %d",
		      inl->kind);
	}
}

static void check_mapping(pres_c_1 *pres, pres_c_mapping map,
			  cast_type ctype, mint_ref itype)
{
	mint_def *idef;
	
	assert(map);
	assert(ctype);
	CHECK_MINT_REF(itype);
	idef = &(pres->mint.defs.defs_val[itype]);
	
	/* Dig down through the type qualifiers. */
	while (ctype->kind == CAST_TYPE_QUALIFIED)
		ctype = ctype->cast_type_u_u.qualified.actual;
	
	switch (map->kind) {
	case PRES_C_MAPPING_DIRECT:
		if (idef->kind == MINT_VOID) {
			assert(ctype->kind == CAST_TYPE_VOID);
			break;
		}
		if (idef->kind == MINT_FPAGE) {
		        assert(ctype->kind == CAST_TYPE_FPAGE);
			break;
		}
		if (idef->kind == MINT_L4STRING) {
		        assert(ctype->kind == CAST_TYPE_L4STRING);
			break;
		}
		if (idef->kind == MINT_INTRFC) {
		        assert(ctype->kind == CAST_TYPE_INTRFC);
			break;
		}
		assert((idef->kind == MINT_INTEGER) ||
		       (idef->kind == MINT_FLOAT) ||
		       (idef->kind == MINT_CHAR) ||
		       (idef->kind == MINT_SCALAR));
		assert(ctype->kind == CAST_TYPE_PRIMITIVE ||
		       ctype->kind == CAST_TYPE_NAME);
		break;
		
	case PRES_C_MAPPING_IGNORE:
		assert(idef->kind == MINT_VOID);
		/*
		 * We used to only print a warning if a non-void MINT type was
		 * being ignored, but it is really wrong to ignore non-void
		 * message data.  We need to marshal/unmarshal the data even if
		 * it is not going to be presented.  If/when we need this, we
		 * will have to invent PRES_C_MAPPING_UNPRESENTED.
		 */
		break;
		
	case PRES_C_MAPPING_POINTER: {
		pres_c_mapping_pointer *pmap =
			&map->pres_c_mapping_u_u.pointer;
		ctype = cast_find_typedef_type(&(pres->cast), ctype);
		assert(ctype);
		
		assert(ctype->kind == CAST_TYPE_POINTER ||
		       ctype->kind == CAST_TYPE_ARRAY);
		if (ctype->kind == CAST_TYPE_ARRAY)
			check_mapping(pres, pmap->target,
				      ctype->cast_type_u_u.array_type.element_type, itype);
		else
			check_mapping(pres, pmap->target,
				      ctype->cast_type_u_u.pointer_type.target, itype);
		check_alloc(&pmap->alloc);
		break;
	}
	
	case PRES_C_MAPPING_FIXED_ARRAY: {
		pres_c_mapping_fixed_array *amap =
			&(map->pres_c_mapping_u_u.fixed_array);
		cast_type element_ctype;
		mint_ref element_itype;
		
		ctype = cast_find_typedef_type(&(pres->cast), ctype);
		assert(ctype);
		
		assert((ctype->kind == CAST_TYPE_ARRAY)
		       || (ctype->kind == CAST_TYPE_POINTER));
		
		assert(/* Generally, the MINT type must be an array. */
			(idef->kind == MINT_ARRAY)
			||
			/*
			 * But we can be associated with a MINT_VOID iff our
			 * `element_mapping' is MAPPING_IGNORE.  This kind of
			 * PRES_C is created, for instance, in order to get a
			 * server dispatch function to allocate space for `out'
			 * arrays.
			 *
			 * XXX --- Should we be checking our `element_mapping'
			 * in this case?  Presumably, if it's wrong, then
			 * `check-mapping' will signal an appropriate error.
			 */
		       ((idef->kind == MINT_VOID)
			&& amap->element_mapping
			&& (amap->element_mapping->kind
			    == PRES_C_MAPPING_IGNORE))
			);
		
		/* assert(amap->length >= 0); --- length is unsigned. */
		/* XXX --- Check mapping length against CAST array length? */
		check_alloc(&(amap->alloc));
		
		switch (ctype->kind) {
		case CAST_TYPE_ARRAY:
			element_ctype = ctype->cast_type_u_u.array_type.
					element_type;
			break;
		case CAST_TYPE_POINTER:
			element_ctype = ctype->cast_type_u_u.pointer_type.
					target;
			break;
		default:
			panic("check_mapping: Invalid CAST type for "
			      "PRES_C_MAPPING_FIXED_ARRAY.");
			break;
		}
		switch (idef->kind) {
		case MINT_ARRAY:
			element_itype = idef->mint_def_u.array_def.
					element_type;
			break;
		case MINT_VOID:
			element_itype = itype;
			break;
		default:
			panic("check_mapping: Invalid MINT type for "
			      "PRES_C_MAPPING_FIXED_ARRAY.");
			break;
		}
		check_mapping(pres,
			      amap->element_mapping,
			      element_ctype,
			      element_itype);
		break;
	}
	
	case PRES_C_MAPPING_TERMINATED_ARRAY: {
		pres_c_mapping_terminated_array *amap =
			&(map->pres_c_mapping_u_u.terminated_array);
		mint_ref element_itype;
		
		ctype = cast_find_typedef_type(&(pres->cast), ctype);
		assert(ctype);
		
		assert(ctype->kind == CAST_TYPE_POINTER ||
		       ctype->kind == CAST_TYPE_ARRAY);
		assert(/* Generally, the MINT type must be an array. */
			(idef->kind == MINT_ARRAY)
			||
			/*
			 * But we can be associated with a MINT_VOID iff our
			 * `element_mapping' is MAPPING_IGNORE.  This kind of
			 * PRES_C is created, for instance, in order to get a
			 * server dispatch function to allocate space for `out'
			 * arrays.
			 *
			 * XXX --- Should we be checking our `element_mapping'
			 * in this case?  Presumably, if it's wrong, then
			 * `check-mapping' will signal an appropriate error.
			 */
		       ((idef->kind == MINT_VOID)
			&& amap->element_mapping
			&& (amap->element_mapping->kind
			    == PRES_C_MAPPING_IGNORE))
			);
		check_alloc(&(map->pres_c_mapping_u_u.
			      terminated_array.alloc));
		switch (idef->kind) {
		case MINT_ARRAY:
			element_itype = idef->mint_def_u.array_def.
					element_type;
			break;
		case MINT_VOID:
			element_itype = itype;
			break;
		default:
			panic("check_mapping: Invalid MINT type for "
			      "PRES_C_MAPPING_TERMINATED_ARRAY.");
			break;
		}
		check_mapping(pres,
			      (map->pres_c_mapping_u_u.terminated_array.
			       element_mapping),
			      ((ctype->kind == CAST_TYPE_POINTER)?
			       ctype->cast_type_u_u.pointer_type.target:
			       ctype->cast_type_u_u.array_type.element_type),
			      element_itype);
		
		break;
	}
	
	case PRES_C_MAPPING_STRUCT: {
		struct inline_state is;
		
		assert(ctype->kind == CAST_TYPE_STRUCT);
		is.isfunc = 0;
		is.u.stype = &ctype->cast_type_u_u.struct_type;
		check_inline(pres, map->pres_c_mapping_u_u.struct_i, &is,
			     itype);
		break;
	}
	
	case PRES_C_MAPPING_XLATE: {
		pres_c_mapping_xlate *xmap = &map->pres_c_mapping_u_u.xlate;
		
		check_mapping(pres, xmap->internal_mapping,
			      xmap->internal_ctype, itype);
		
		assert(!(xmap->alloc_flags & ~(PRES_C_DEALLOC_ALWAYS)));
		check_alloc_flags(xmap->alloc_flags);
		
		CHECK_OPT_STRING(xmap->translator);
		CHECK_OPT_STRING(xmap->destructor);
		break;
	}
	
	case PRES_C_MAPPING_REFERENCE: {
		assert(idef->kind == MINT_INTERFACE);
		
		/*
		 * XXX --- I (Eric Eide) think that requiring a named type at
		 * this point is unnecessary.  Why can't object references be
		 * essentially anything: pointers, integers (indices),
		 * structures, ...?
		 */
		/* assert(ctype->kind == CAST_TYPE_NAME); */
		assert(ctype->cast_type_u_u.name != 0);
		switch (map->pres_c_mapping_u_u.ref.kind) {
		case PRES_C_REFERENCE_COPY:
		case PRES_C_REFERENCE_MOVE:
		case PRES_C_REFERENCE_COPY_AND_CONVERT:
			break;
		default:
			panic("pres_c_check: Invalid mapping reference!");
		}
		/* this means how many am I sending/receiving - can't be 0 */
		assert(map->pres_c_mapping_u_u.ref.ref_count > 0);
		break;
	}
	
	case PRES_C_MAPPING_TYPE_TAG: {
		assert(idef->kind == MINT_TYPE_TAG);
		break;
	}
	
	case PRES_C_MAPPING_STUB: {
		assert((ctype->kind == CAST_TYPE_NAME) ||
		       (ctype->kind == CAST_TYPE_STRUCT_NAME) ||
		       (ctype->kind == CAST_TYPE_UNION_NAME) ||
		       (ctype->kind == CAST_TYPE_ENUM_NAME));
		break;
	}
	
	case PRES_C_MAPPING_OPTIONAL_POINTER: {
		pres_c_mapping_optional_pointer *pmap =
			&(map->pres_c_mapping_u_u.optional_pointer);
		mint_array_def *iarray = &(idef->mint_def_u.array_def);
		mint_def *length_def;
		
		/* On the CAST side, this mapping corresponds to a pointer. */
		assert(ctype->kind == CAST_TYPE_POINTER);
		/* On the MINT side, we have a MINT array (a counted array
		   containing zero or one elements). */
		assert(idef->kind == MINT_ARRAY);
		
		/* Check that the length of the MINT array is an integer that
		   may be zero or one. */
		CHECK_MINT_REF(iarray->element_type);
		length_def = &(pres->mint.defs.defs_val[iarray->length_type]);
		assert(length_def->kind == MINT_INTEGER);
		assert(length_def->mint_def_u.integer_def.min == 0);
		assert(length_def->mint_def_u.integer_def.range == 1);
		
		/* Check the mapping of this pointer's target type. */
		check_mapping(pres,
			      pmap->target,
			      ctype->cast_type_u_u.pointer_type.target,
			      iarray->element_type);
		/* Check the allocation of the pointer itself. */
		check_alloc(&(pmap->alloc));
		break;
	}
	
	case PRES_C_MAPPING_SYSTEM_EXCEPTION:
		assert(idef->kind == MINT_SYSTEM_EXCEPTION);
		break;
		
	case PRES_C_MAPPING_DIRECTION: {
		pres_c_mapping_direction *dir_map =
			&(map->pres_c_mapping_u_u.direction);
		
		assert((dir_map->dir == PRES_C_DIRECTION_UNKNOWN)
		       || (dir_map->dir == PRES_C_DIRECTION_IN)
		       || (dir_map->dir == PRES_C_DIRECTION_INOUT)
		       || (dir_map->dir == PRES_C_DIRECTION_OUT)
		       || (dir_map->dir == PRES_C_DIRECTION_RETURN));
		
		check_mapping(pres,
			      dir_map->mapping,
			      ctype,
			      itype);
		break;
	}
	
	case PRES_C_MAPPING_SID:
		/*
		 * The `ctype' corresponding to a SID could be anything, so we
		 * can't really check anything about it here.
		 *
		 * SIDs are not part of an IDL-defined message and therefore
		 * don't have MINT representations.  So we can't say anything
		 * about the `itype' either.
		 */
		break;

	case PRES_C_MAPPING_ARGUMENT:
		assert(map->pres_c_mapping_u_u.argument.arg_name);
		if (map->pres_c_mapping_u_u.argument.map)
			check_mapping(pres, map->pres_c_mapping_u_u.argument.map, ctype, itype);
		break;
		
	case PRES_C_MAPPING_MESSAGE_ATTRIBUTE: {
		pres_c_mapping_message_attribute *msg_attr =
			&(map->pres_c_mapping_u_u.message_attribute);
		assert((msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_FLAGS)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_TIMEOUT)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_SEQUENCE_RECEIVED)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_CLIENT_REFERENCE)
		       || (msg_attr->kind == PRES_C_MESSAGE_ATTRIBUTE_SERVERCOPY));
		/*
		 * The `ctype' corresponding to an attribute could be
		 * anything, so we can't really check anything about it
		 * here.
		 *
		 * attributes are not part of an IDL-defined message and
		 * therefore don't have MINT representations.  So we can't
		 * say anything about the `itype' either.
		 */
		break;
	}
	
	default:
		panic("check_mapping: unknown pres_c_mapping_kind %d",
		      map->kind);
	}
}

static void check_stub(pres_c_1 *pres, int n)
{
	pres_c_stub *stub = &(pres->stubs.stubs_val[n]);
	cast_def *cdef = NULL;
	
	assert(n >= 0);
	assert(n < (signed int)pres->stubs.stubs_len);
	
	switch (stub->kind)
	{
		case PRES_C_MARSHAL_STUB:
		{
			pres_c_marshal_stub *mstub = &stub->pres_c_stub_u.mstub;
			struct inline_state is;
			cdef = &pres->cast.cast_scope_val[mstub->c_func];

			CHECK_CAST_REF(mstub->c_func);
			assert(cdef->u.kind == CAST_FUNC_DECL);
			is.isfunc = 1;
			is.u.ftype = &cdef->u.cast_def_u_u.func_type;

			CHECK_MINT_REF(mstub->itype);

			check_inline(pres, mstub->i, &is, mstub->itype);
			/*
			 * XXX --- We can't check the target inline because we
			 * don't have any MINT representation of the target
			 * object.  (m/u stubs don't *have* target objects!)
			 *
			 * check_inline(pres, mstub->target_i, &is, XXX);
			 */
			break;
		}
		case PRES_C_UNMARSHAL_STUB:
		{
			pres_c_marshal_stub *ustub = &stub->pres_c_stub_u.ustub;
			struct inline_state is;
			cdef = &pres->cast.cast_scope_val[ustub->c_func];

			CHECK_CAST_REF(ustub->c_func);
			assert(cdef->u.kind == CAST_FUNC_DECL);
			is.isfunc = 1;
			is.u.ftype = &cdef->u.cast_def_u_u.func_type;

			CHECK_MINT_REF(ustub->itype);

			check_inline(pres, ustub->i, &is, ustub->itype);
			/*
			 * XXX --- We can't check the target inline because we
			 * don't have any MINT representation of the target
			 * object.  (m/u stubs don't *have* target objects!)
			 *
			 * check_inline(pres, ustub->target_i, &is, XXX);
			 */
			break;
		}
		case PRES_C_CLIENT_STUB:
		{
			pres_c_client_stub *cstub = &stub->pres_c_stub_u.cstub;
			struct inline_state is;
			cdef = &pres->cast.cast_scope_val[cstub->c_func];

			CHECK_CAST_REF(cstub->c_func);
			assert(cdef->u.kind == CAST_FUNC_DECL);
			is.isfunc = 1;
			is.u.ftype = &cdef->u.cast_def_u_u.func_type;
			
			check_stub_op_flags(cstub->op_flags);
			
			CHECK_MINT_REF(cstub->request_itype);
			CHECK_MINT_REF(cstub->reply_itype);

			check_inline(pres, cstub->request_i, &is,
				     cstub->request_itype);
			check_inline(pres, cstub->reply_i, &is,
				     cstub->reply_itype);

			check_inline(pres, cstub->target_i, &is,
				     cstub->target_itype);
			if (cstub->client_i) {
				check_inline(pres, cstub->client_i, &is,
					     cstub->client_itype);
			} else {
				assert(cstub->client_itype == mint_ref_null);
			}
			break;
		}
		case PRES_C_SERVER_SKEL:
		{
			pres_c_server_skel *sskel = &stub->pres_c_stub_u.sskel;
			int i;
			cdef = &pres->cast.cast_scope_val[sskel->c_def];

			CHECK_CAST_REF(sskel->c_def);
			assert((cdef->u.kind == CAST_VAR_DECL)
			       || (cdef->u.kind == CAST_FUNC_DECL));
			
			CHECK_MINT_REF(sskel->request_itype);
			CHECK_MINT_REF(sskel->reply_itype);

			for (i = 0; i < (signed int)sskel->funcs.funcs_len; i++)
			{
				struct pres_c_server_func *sfunc = &sskel->funcs.funcs_val[i];
				struct inline_state is;

				assert(sskel->funcs.funcs_val);

				CHECK_CAST_REF(sfunc->c_func);
				cdef = &pres->cast.cast_scope_val[sfunc->c_func];
				assert(cdef->u.kind == CAST_FUNC_DECL);
				is.isfunc = 1;
				is.u.ftype = &cdef->u.cast_def_u_u.func_type;

				check_stub_op_flags(sfunc->op_flags);
				
				check_inline(pres, sfunc->request_i, &is,
					     sskel->request_itype);
				check_inline(pres, sfunc->reply_i, &is,
					     sskel->reply_itype);
				check_inline(pres, sfunc->target_i, &is,
					     sfunc->target_itype);
				if (sfunc->client_i) {
					check_inline(pres, sfunc->client_i,
						     &is, sfunc->client_itype);
				} else {
					assert(sfunc->client_itype
					       == mint_ref_null);
				}
			}
			break;
		}
		case PRES_C_SEND_STUB:
		{
			pres_c_send_stub *send_stub = &stub->pres_c_stub_u.send_stub;
			struct inline_state is;
			cdef = &pres->cast.cast_scope_val[send_stub->c_func];

			CHECK_CAST_REF(send_stub->c_func);
			assert(cdef->u.kind == CAST_FUNC_DECL);
			is.isfunc = 1;
			is.u.ftype = &cdef->u.cast_def_u_u.func_type;

			CHECK_MINT_REF(send_stub->msg_itype);

			check_inline(pres, send_stub->msg_i, &is,
				     send_stub->msg_itype);
			check_inline(pres, send_stub->target_i, &is,
				     send_stub->target_itype);
			break;
		}
		default:
			panic("check_stub: unknown pres_c_stub_kind %d",
			      stub->kind);
	}
	
	assert(cdef &&
	       ((cdef->included == NOT_INCLUDED)
		|| (cdef->included == INCLUDED)
		|| (cdef->included == IMPLIED))
		);
}

void pres_c_1_check(pres_c_1 *pres)
{
	int i;
	
	/* Check the integrity of the embedded MINT and CAST individually. */
	mint_1_check(&pres->mint);
	cast_check(&pres->cast);
	
	/* Now check the integrity of the stubs
	   and their relationships to the MINT and CAST.  */
	for (i = 0; i < (signed int) pres->stubs.stubs_len; i++) {
		assert(pres->stubs.stubs_val);
		check_stub(pres, i);
	}
};

/* End of file. */

