/*
 * Copyright (c) 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/compiler.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include "l4.h"

#define IS_PROPAGATION (cstub->op_flags & PRES_C_STUB_OP_FLAG_PROPAGATE)
#define IS_ONEWAY (cstub->op_flags & PRES_C_STUB_OP_FLAG_ONEWAY)
/*
 *
 */

void w_client_stub(pres_c_1 *pres, int stub_idx)
{
	assert(pres);
	
	pres_c_stub        *stub   = &pres->stubs.stubs_val[stub_idx];
	pres_c_client_stub *cstub  = &stub->pres_c_stub_u.cstub;
	cast_def           *cfunc  = &pres->cast.cast_scope_val[cstub->c_func];
	cast_func_type     *cfunct = &cfunc->u.cast_def_u_u.func_type;
	
	mint_ref            operation_itype;
	pres_c_inline       operation_inline;
	
	int                 assumptions = RPCM_TRUSTED; /* XXX */
	char		   *emergency_return_string;
	
	cast_stmt           rpc_cast_stmt;
	
	/*****/
	
	if (cfunct->return_type->kind != CAST_TYPE_VOID)
		emergency_return_string = "return _return";
	else
		emergency_return_string = "return";
	emergency_return_value = cast_new_expr_name(emergency_return_string);
	
	/*
	 * Figure out which of the stub function parameters is the object
	 * reference and create a CAST expression that names it.
	 *
	 * We do this by using a `l4_target_mu_state' to ``marshal'' the
	 * target object reference.  This process doesn't create any marshaling
	 * code, however.  The `l4_taregt_mu_state' knows that when it would
	 * ordinarily marshal an object reference (to the target object), it
	 * should instead simply store the CAST name of the object in its
	 * `target_cast_expr' slot.  Clever, eh?
	 */
	l4_target_mu_state must_target(pres,
					  (MUST_ENCODE),
					  assumptions, "client");
	
	must_target.mu_func_params(cstub->c_func, cstub->target_itype,
				   cstub->target_i);
	must_target.mu_end();
	
	/*
	 * Build the request marshaling/deallocation code.  We strip away the
	 * ``collapsed union'' goo that encodes the IDL and interface IDs.  We
	 * don't need this information for L4 IPC because it is manifest in
	 * the object references.
	 */
	l4_mu_state must_in(pres,
			       (MUST_ENCODE | MUST_DEALLOCATE),
			       assumptions, "client");
	remove_idl_and_interface_ids(pres,
				     cstub->request_itype, cstub->request_i,
				     &operation_itype, &operation_inline);
	must_in.mu_func_params(cstub->c_func,
			       operation_itype, operation_inline);
	must_in.mu_end();
	
	/*
	 * Build the reply unmarshaling/allocation code.  Again, we strip away
	 * all of the ``collapsed union'' IDL and interface goo.  This time we
	 * strip away the operation reply identifier, too.  (Note that this is
	 * just a fixed number indicating what operation is being replied to.
	 * Any success/failure code is a separate message component!)
	 */
	l4_mu_state must_out(pres,
				(MUST_DECODE | MUST_ALLOCATE),
				assumptions, "client");
	remove_idl_and_interface_ids(pres,
				     cstub->reply_itype, cstub->reply_i,
				     &operation_itype, &operation_inline);
	remove_operation_id(pres,
			    operation_itype, operation_inline,
			    &operation_itype, &operation_inline);
	must_out.mu_func_params(cstub->c_func,
				operation_itype, operation_inline);
	must_out.mu_end();
	
	/* Output the client stub. */
	
	cast_w_func_type(cfunc->name, cfunct, 0);
	w_printf("\n{\n");
	w_i_printf(1, "char *_buf_current, *_buf_start;\n");
	if (!IS_PROPAGATION) {
		w_i_printf(1, "l4_msgdope_t _rc;\n");
		w_i_printf(1, "l4_ipc_buffer_t __msgbuf; /* temporary send buffer */\n");
		w_i_printf(1, "l4_ipc_buffer_t* _msgbuf = &__msgbuf;\n");
	}
	else {
	        w_i_printf(1, "l4_ipc_buffer_t* _msgbuf;\n");
	}

	/* for propagation - the return-type is always void - 
	 * so don't care... */
	if (cfunct->return_type->kind != CAST_TYPE_VOID) {
		/*
		 * Initialize `_return' to be zero.  This eliminates compiler
		 * warnings about returning uninitialized data.
		 */
		cast_expr init_expr
			= cast_new_expr_assign_to_zero(
				cast_new_expr_name("_return"),
				cfunct->return_type,
				&(pres->cast)
				);
		
		w_indent(1);
		cast_w_type("_return", cfunct->return_type, 1);
		w_printf(";\n");
		
		w_printf("\n");
		cast_w_stmt(cast_new_stmt_expr(init_expr), 1);
	} else
		w_printf("\n");


	w_i_printf(1,
		   "flick_%s_%s_start_encode();\n",
		   must_in.get_be_name(),
		   IS_PROPAGATION ? "propagation" : "client");

	/* if there is an rcv-fpage parameter - we have to configure
	 * the msg-buffer - we have to do it after the "start_encode"
	 */
	for (int i=0; i<cfunct->params.params_len;i++) {
	    if (strcmp(cfunct->params.params_val[i].name, "rcvfpage") == 0) {
		w_i_printf(1, "flick_%s_client_set_rcvfpage();\n",
			 must_in.get_be_name());
		break;
	    }
	}

		
	/* Output the request marshaling code. */
	cast_w_stmt(must_in.c_block, 1);

	w_i_printf(1,
		   "flick_%s_%s_end_encode();\n",
		   must_in.get_be_name(),
		   IS_PROPAGATION ? "propagation" : "client");

	if (IS_PROPAGATION) {
		w_printf("}\n\n");
		return;
	}

	/*
	 * Make the RPC.  If the client stub parametr list includes SIDs, use
	 * the secure version of the RPC function.
	 */
	/*
	 * Until December 1997, secure Flask stubs used server SID parameters.
	 * But now, server SIDs are not presented by the MOM/L4 PG and are
	 * not supported by the MOM-on-L4 runtime.  Warn if the server SIDs
	 * parameters are present; the old assertions are `#if 0'ed out in the
	 * code below.
	 */
	if (must_in.server_sid_cexpr || must_out.server_sid_cexpr) {
		warn("the L4 transport does not support server SID "
		     "arguments to stubs.");
		warn("ignoring server SID parameters to client stub `%s'.",
		     cfunc->name);
	}
	
	if (!must_in.client_sid_cexpr) {
		/*
		 * The client stub does not include the client SID in its
		 * parameter list.  Make sure that it doesn't include any other
		 * SIDs either, and make the normal RPC call.
		 */
#if 0
		/* See above; server SID params are no longer used in Flask. */
		assert(!must_in.server_sid_cexpr
		       && !must_out.server_sid_cexpr);
#endif
		rpc_cast_stmt =
			cast_new_stmt_expr(
				cast_new_expr_assign(
					cast_new_expr_name("_rc"),
					cast_new_expr_call_2(
						cast_new_expr_name(
							IS_ONEWAY ? 
							"flick_client_send" :
							"flick_client_call"
							),
						/* arg 0 */
						must_target.target_cast_expr,
						cast_new_expr_name("_msgbuf")
						)));

	} else {
		/*
		 * The client stub does include the client SID in its
		 * parameter list.  Make the secure RPC call.
		 */
#if 0
		/* See above; server SID params are no longer used in Flask. */
		assert(must_in.server_sid_cexpr
		       && must_out.server_sid_cexpr);
#endif
		
		rpc_cast_stmt =
			cast_new_stmt_expr(
				cast_new_expr_assign(
					cast_new_expr_name("_rc"),
					cast_new_expr_call_3(
						cast_new_expr_name(
							"flick_client_send_"
							"request_get_reply_"
							"secure"
							),
						/* arg 0 */
						must_target.target_cast_expr,
						/* arg 1 */
						cast_new_expr_name("_"),
						/* arg 2 */
						must_in.client_sid_cexpr
#if 0
						/* Not used; see above. */
						/* arg 3 */
						must_in.server_sid_cexpr,
						/* arg 4 */
						must_out.server_sid_cexpr
#endif
						)));
	}
	cast_w_stmt(rpc_cast_stmt, 1);
	
	/* Check for errors from the IPC operation. */
	w_i_printf(1,
		   "if (L4_IPC_IS_ERROR(_rc))\n");
	w_i_printf(2,
		   "flick_%s_decode_client_error"
		   "(%s, FLICK_ERROR_COMMUNICATION, %s, %s);\n",
		   must_in.pres->pres_context,
		   emergency_return_string,
		   must_in.get_encode_name(),
		   must_in.get_be_name());
	
	/* Output the reply unmarshaling code. */
	if (IS_ONEWAY) {
		w_printf("}\n\n");
		return;
	}
	
	w_i_printf(1,
		   "flick_%s_client_start_decode();\n",
		   must_in.get_be_name());
	cast_w_stmt(must_out.c_block, 1);
	w_i_printf(1,
		   "flick_%s_client_end_decode();\n",
		   must_in.get_be_name());
	
	if (cfunct->return_type->kind != CAST_TYPE_VOID)
		w_i_printf(1, "return _return;\n");
	w_printf("}\n\n");
}

/* End of file. */

