/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>

#include <mom/c/libcast.h>

#include <mom/c/pbe.hh>

#include "l4.h"
/*
 * XXX --- The following function is a hack to change the names of the server
 * work function parameters and their types (to remove CAST_TYPE_QUALIFIERs,
 * and to change array types into pointer-to-element types).
 *
 * + Name changes are good because they help us avoid name clashes in the
 *   generated code (e.g., between the name of a type and the name of a local
 *   variable).
 *
 * + The removal of type qualifiers is necessary so that we don't try to
 *   allocate locals with `const' types!
 *
 * + Changing array types into pointer types helps `mu_server_func' avoid
 *   declaring unnecessary array locals.  Using a pointer helps the back end be
 *   smart, e.g., point into the message buffer when the encoded array is
 *   bitwise identical to the presented array.  This transformation works only
 *   because of the ``array slice hack'' that was implemented in `mu_state::
 *   mu_mapping_stub_inline', which allows us to associate pointer-to-element
 *   CAST types with the marshal/unmarshal stubs for the corresponding array
 *   CAST types.
 *
 * But really, we should do all of these things in a more principled way.
 */

#define LOCAL_PREFIX "_local_"

static cast_func_type *whack_on_server_work_cfunct(cast_func_type *cfunct,
						   pres_c_1 *pres)
{
	cast_func_type *new_cfunct;
	cast_type new_ctype;
	char *new_name;
	
	int is_array;
	cast_type internal_ctype;
	
	unsigned int i;
	
	/*
	 * Allocate a new `cast_func_type' with the appropriate number of
	 * parameters.
	 */
	new_cfunct = (cast_func_type *) mustmalloc(sizeof(cast_func_type));
	
	new_cfunct->return_type = cfunct->return_type;
	
	new_cfunct->params.params_len = cfunct->params.params_len;
	new_cfunct->params.params_val = (cast_func_param *)
					mustmalloc(cfunct->params.params_len
						   * sizeof(cast_func_param));
	
	/* Now determine the new name and type of each parameter. */
	for (i = 0; i < cfunct->params.params_len; i++) {
		/*
		 * Determine the new name.  Ultimately, this name will be used
		 * to declare a local variable in the server dispatch function.
		 */
		if (cfunct->params.params_val[i].name[0] == '_') {
			/*
			 * Don't munge the name if it starts with an `_'.  Some
			 * PG/BE code thinks that it knows the names of these
			 * parameters (e.g., that `_ev' is the environment.).
			 */
			new_name = cfunct->params.params_val[i].name;
		} else {
			new_name  = (char *)
				    mustmalloc(sizeof(LOCAL_PREFIX)
					       + strlen(cfunct->
							params.params_val[i].
							name));
			strcpy(new_name, LOCAL_PREFIX);
			strcat(new_name, cfunct->params.params_val[i].name);
		}
		
		/*
		 * Determine the new type.
		 *
		 * We strip away qualifiers and transform array types into
		 * pointer-to-element types.  By transforming array types into
		 * pointer types, we make it easier for `mu_server_func' to be
		 * intelligent.  (We don't want `mu_server_func' to declare a
		 * local array if it's not necessary!)
		 */
		new_ctype = cfunct->params.params_val[i].type;
		
		while (new_ctype->kind == CAST_TYPE_QUALIFIED)
			new_ctype = new_ctype->cast_type_u_u.qualified.actual;
		
		if ((new_ctype->kind == CAST_TYPE_POINTER)
		    && (new_ctype->cast_type_u_u.pointer_type.target->kind
			== CAST_TYPE_QUALIFIED)
			) {
			/*
			 * Strip away modifiers from the target type, too!
			 */
			cast_type target_ctype = new_ctype->cast_type_u_u.
						 pointer_type.target;
			
			while (target_ctype->kind == CAST_TYPE_QUALIFIED)
				target_ctype = target_ctype->cast_type_u_u.
					       qualified.actual;
			
			new_ctype = cast_new_pointer_type(target_ctype);
		}
		
		/* Transform array types into pointer-to-element types. */
		switch (new_ctype->kind) {
		default:
			/*
			 * General case: the parameter has a type that is not
			 * an array type.
			 */
			is_array = 0;
			break;
			
		case CAST_TYPE_ARRAY:
			/*
			 * Easy case: the parameter type is an explicit array
			 * type.
			 */
			is_array = 1;
			new_ctype =
				cast_new_pointer_type(new_ctype->
						      cast_type_u_u.array_type.
						      element_type);
			break;
			
		case CAST_TYPE_NAME:
			/*
			 * Difficult case: the parameter type is a named type
			 * that we must resolve in order to see if it is an
			 * array type.
			 */
			internal_ctype = cast_find_typedef_type(&(pres->cast),
								new_ctype);
			
			if (!internal_ctype) {
				/* We hit an unresolvable name.  Punt. */
				is_array = 0;
				
			} else if (internal_ctype->kind != CAST_TYPE_ARRAY) {
				/* What we finally found wasn't an array. */
				is_array = 0;
				
			} else {
				/* We resolved to an array! */
				is_array = 1;
				new_ctype
					= cast_new_pointer_type(
						internal_ctype->
						cast_type_u_u.array_type.
						element_type);
			}
			break;
		}
		
		/*
		 * Save the new name and type in `new_cfunct's parameters.
		 */
		new_cfunct->params.params_val[i].name = new_name;
		new_cfunct->params.params_val[i].type = new_ctype;
	}
	
	return new_cfunct;
}

/*
 * Another auxiliary method for creating a local varibale to act as a ``root''
 * for one of the parameters of the server functions we wish to call.
 */
void l4_mu_state::mu_server_func_alloc_param(cast_func_param *param)
{
	int def_index = cast_add_def(&(c_block->cast_stmt_u_u.block.scope));
	cast_def *def = &(c_block->
			  cast_stmt_u_u.block.scope.cast_scope_val[def_index]);
	
	def->name = param->name;
	def->u.kind = CAST_VAR_DEF;
	def->u.cast_def_u_u.var_def.type = param->type;
	def->u.cast_def_u_u.var_def.init = 0;
}


/* This routine generates the code to call a single server work function.
   It is generally called (indirectly) from within mu_decode_switch(),
   once the possible choices have been narrowed down to a single server work
   function.
   
   The parameters `inl' and `tn_r' describe the "remaining" part of the itype
   and ctype
   of the incoming request message,
   after mu_decode_switch has "eaten" the part it needed to select a work
   function.
   This routine will generate code to do the rest of the message decoding
   before spitting out the call to the actual C work function.
   
   After the call to the work function,
   this routine generates the necessary code to marshal the reply message,
   using the _entire_ reply itype and ctype extracted directly from `sstub' and
   `sfunc',
   respectively.
   (Reply messages need no decoding on the server side.)
   
   Finally, it calls the mu_server_func_reply() method,
   which more-specific code must provide,
   which does whatever is necessary to finish building the reply message
   and, if appropriate, send it off.
   In some implementations, the reply will be sent off automatically
   after the server stub returns to the runtime support code or to the kernel,
   so nothing special would need to be done here in that case.
   */
void l4_mu_state::mu_server_func(pres_c_inline inl, mint_ref tn_r,
				 pres_c_server_func *sfunc,
				 pres_c_server_skel *sstub)
{
	int i;
	
	cast_def *cfunc = &pres->cast.cast_scope_val[sfunc->c_func];
	cast_func_type *cfunct = &cfunc->u.cast_def_u_u.func_type;
	int cfunct_returns_nonvoid;
	
	/*
	 * XXX --- Here we ``whack'' the C function type into a form that
	 * allows us to avoid name clashes and type problems.
	 */
	cast_func_type orig_sfunc_type;
	
	orig_sfunc_type = *cfunct;
	cfunct = whack_on_server_work_cfunct(cfunct, pres);
	cfunc->u.cast_def_u_u.func_type = *cfunct;
	
	/*
	 * Create a new scope containing the variables passed to this server
	 * function.
	 */
	cast_stmt old_c_block = c_block;
	c_block = cast_new_block(0, 0);
	
	/*
	 * Declare local variables to act the the ``roots'' of our function's
	 * parameters and return value.
	 */
	for (i = 0; i < (signed int) cfunct->params.params_len; i++)
		mu_server_func_alloc_param(&(cfunct->params.params_val[i]));
	
	cfunct_returns_nonvoid = ((cfunct->return_type != 0) &&
				  (cfunct->return_type->kind != CAST_TYPE_VOID)
				  );
	
	if (cfunct_returns_nonvoid) {
		int d_index = cast_add_def(&(c_block->
					     cast_stmt_u_u.block.scope));
		cast_def *d = &(c_block->cast_stmt_u_u.block.scope.
				cast_scope_val[d_index]);
		
		d->name = "_return";
		d->u.kind = CAST_VAR_DEF;
		d->u.cast_def_u_u.var_def.type = cfunct->return_type;
		d->u.cast_def_u_u.var_def.init = 0;
		
		/*
		 * XXX --- If the return type is integer-like or a pointer,
		 * initialize `_return' to zero.  Until we get real error
		 * handling this is better than nothing.
		 */
		switch (cfunct->return_type->kind) {
		case CAST_TYPE_PRIMITIVE:
			d->u.cast_def_u_u.var_def.init =
				(cast_init) mustmalloc(sizeof(cast_init_u));
			d->u.cast_def_u_u.var_def.init->kind = CAST_INIT_EXPR;
			switch (cfunct->return_type->
				cast_type_u_u.primitive_type.kind) {
			case CAST_PRIM_CHAR:
				d->u.cast_def_u_u.var_def.init->
					cast_init_u_u.expr =
					cast_new_expr_lit_char(0, 0);
				break;
			case CAST_PRIM_INT:
				d->u.cast_def_u_u.var_def.init->
					cast_init_u_u.expr =
					cast_new_expr_lit_int(0, 0);
				break;
			case CAST_PRIM_FLOAT:
				d->u.cast_def_u_u.var_def.init->
					cast_init_u_u.expr =
					cast_new_expr_lit_float(0.0);
				break;
			case CAST_PRIM_DOUBLE:
				d->u.cast_def_u_u.var_def.init->
					cast_init_u_u.expr =
					cast_new_expr_lit_double(0.0, 0);
				break;
			default:
				panic("In `mu_server_func', unrecognized CAST "
				      "primitive type.");
				break;
			}
			break;
			
		case CAST_TYPE_POINTER:
			d->u.cast_def_u_u.var_def.init =
				(cast_init) mustmalloc(sizeof(cast_init_u));
			d->u.cast_def_u_u.var_def.init->kind = CAST_INIT_EXPR;
			d->u.cast_def_u_u.var_def.init->cast_init_u_u.expr =
				cast_new_expr_lit_int(0, 0);
			break;
			
		default:
			/* Do not initialize the return value. */
			break;
		}
	}
	
	/*
	 * Build the code to locate the target object.
	 */
	mu_server_func_target(sfunc);
	/*
	 * ... and the client object, if present.
	 */
	mu_server_func_client(sfunc);
	
	/*
	 * Build the code to unmarshal the request.  Additionally, this will
	 * also allocate any extra storage required for `out' parameters.
	 */
	mu_func_params(sfunc->c_func, tn_r, inl);
	mu_end();
	
	/*
	 * Signal that we are finished with the decode phase.  The matching
	 * `flick_*_server_start_encode' is output by `w_server_skel', which is
	 * defined individually by each back end.
	 */
	add_stmt(
		cast_new_stmt_expr(
			cast_new_expr_call(
				cast_new_expr_name(
					flick_asprintf(
						"flick_%s_server_end_decode",
						get_be_name())),
				0)));
	
	/*
	 * Initialize the environment.
	 *
	 * XXX --- This should be part of the PRES_C for unmarshaling the
	 * request --- i.e., to have a PRES_C_MAPPING_INIT or somesuch that
	 * sticks a constant value into the environment.
	 */
	add_stmt(
		cast_new_stmt_expr(
			cast_new_expr_call(
				cast_new_expr_name(
					flick_asprintf(
						"flick_%s_init_environment",
						pres->pres_context)),
				0)));

	
	/*
	 * Build the function call to the server work function.
	 */
	cast_expr fexpr = cast_new_expr_call(
		cast_new_expr_name(cfunc->name),
		cfunct->params.params_len);
	cast_expr_call *fcall = &fexpr->cast_expr_u_u.call;
	
	for (i = 0; i < (signed int) fcall->params.params_len; i++) {
		fcall->params.params_val[i] =
			cast_new_expr_name(cfunct->params.params_val[i].name);
	}
	cast_expr rexpr;
	if (cfunct_returns_nonvoid)
		rexpr = cast_new_expr_assign(cast_new_expr_name("_return"),
					     fexpr);
	else
		rexpr = fexpr;
	add_stmt(cast_new_stmt_expr(rexpr));

	
	/* volkmar: we return here, if it is a propagation function */
	add_stmt(
		cast_new_stmt_expr(
			cast_new_expr_call(
				cast_new_expr_name(
					flick_asprintf(
						"flick_%s_return_if_propagating",
						get_be_name())),
				0)));

	/*
	 * Build the code to marshal the reply.  Additionally, this will free
	 * storage allocated for `in' parameters.
	 *
	 * XXX --- The freeing of `in' parameters isn't really implemented yet.
	 * But it makes no difference, since we force all `in' parameters to be
	 * `auto' allocated.  See the `auto_alloc' hack in `get_allocator'.
	 *
	 * XXX --- We still need to free storage for `oneway' operations, so we
	 * can't do the following `if' here.
	 */
	if (!(sfunc->op_flags & PRES_C_STUB_OP_FLAG_ONEWAY))
		mu_server_func_reply(sfunc, sstub);
	else
		add_stmt(
			cast_new_return(
				cast_new_expr_name(
					"DISPATCH_NO_REPLY"
					)));
	
	/* End the new scope we created. */
	cast_stmt new_c_block = c_block;
	c_block = old_c_block;
	add_stmt(new_c_block);
	
	/* XXX --- Here we ``unwhack'' the C function type. */
	cfunc->u.cast_def_u_u.func_type = orig_sfunc_type;
}

/* End of file. */





