/*
 * Copyright (c) 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

extern "C" {
#include <mom/compiler.h>
}
#include <mom/c/pbe.hh>

extern char *progname;

/*
 * `BE_FLAGS' is the number of flags ``hardcoded'' in this file --- yuck!
 * These flags include:
 *
 * `-h' / `--header'
 *		to specify the name of the output `.h' file
 * `-p' / `--prefix'
 *		to specify the path prefix for #includes that refer to the
 *		generated header file
 * `-s' / `--system_header'
 *		to specify that the generated header file should be #included
 *		as a system header (i.e., using <>'s)
 *
 * `--no_timestamp'
 *		to suppress the timestamp information that is otherwise output
 *		into the generated `.h' and `.c' files; useful when doing
 *		regression testing on the generated output
 *
 * `-n' / `--no_mu_stubs'
 *		to suppress the definitions of marahsla/unmarshal stubs
 * `-i' / `--no_included_interfaces'
 *		to suppress client stubs and server funcs that were generated
 *		from IDL files that were #included by the root IDL file
 */
#define BE_FLAGS (6)

be_flags be_args(int argc, char **argv, const be_flags &def_flags, char *info)
{
	flags_in in[STD_FLAGS + BE_FLAGS];
	flags_out out;
	be_flags res;
	
	int flags_index;
	
	/* Initialize the array of command line options. */
	
	set_def_flags(in);
	flags_index = STD_FLAGS;
	
	in[flags_index].sng = 'h';
	in[flags_index].dbl = "header";
	in[flags_index].which = fk_STRING;
	in[flags_index].dfault.string = def_flags.header;
	in[flags_index].explain
		= "Set the name of the output header file to <string>";
	++flags_index;
	
	in[flags_index].sng = 'p';
	in[flags_index].dbl = "prefix";
	in[flags_index].which = fk_STRING;
	in[flags_index].dfault.string = def_flags.prefix;
	in[flags_index].explain
		= "Set the prefix for the path to the header to <string>";
	++flags_index;
	
	in[flags_index].sng = 's';
	in[flags_index].dbl = "system_header";
	in[flags_index].which = fk_FLAG;
	in[flags_index].dfault.flag = def_flags.system_header;
	in[flags_index].explain
		= "When including the header, use <>, not \"\"";
	++flags_index;
	
	in[flags_index].sng = 0;
	in[flags_index].dbl = "no_timestamp";
	in[flags_index].which = fk_FLAG;
	in[flags_index].dfault.flag = def_flags.no_timestamp;
	in[flags_index].explain
		= ("Don't output the timestamp comment block into generated "
		   "files");
	++flags_index;
	
	in[flags_index].sng = 'n';
	in[flags_index].dbl = "no_mu_stubs";
	in[flags_index].which = fk_FLAG;
	in[flags_index].dfault.flag = def_flags.no_mu_stubs;
	in[flags_index].explain
		= ("Don't define the marshal/unmarshal stubs (they will still "
		   "be declared)");
	++flags_index;
	
	in[flags_index].sng = 'i';
	in[flags_index].dbl = "no_included_interfaces";
	in[flags_index].which = fk_FLAG;
	in[flags_index].dfault.flag = def_flags.no_includes;
	in[flags_index].explain
		= ("Neither declare nor define stubs for interfaces that were "
		   "#included");
	++flags_index;
	
	assert(flags_index == (STD_FLAGS + BE_FLAGS));
	
	/* Parse the actual command line arguments. */
	
	out = parse_args(argc, argv, (STD_FLAGS + BE_FLAGS), in);
	std_handler(out,
		    (STD_FLAGS + BE_FLAGS), in,
		    "<optional input filename>", info);
	
	res.output = out.flgs[OUTPUT_FILE_FLAG].string;
	if (out.other_count == 1)
		res.input = *out.other;
	else if (out.other_count) {
		print_args_usage(out.progname,
				 (STD_FLAGS + BE_FLAGS), in,
				 "<optional input filename>", info);
		exit(1);
	} else
		res.input = 0;
	progname = out.progname;
	
	/* These had better line up with the list above! */
	flags_index = STD_FLAGS;
	
	res.header		= out.flgs[flags_index++].string;
	res.prefix		= out.flgs[flags_index++].string;
	res.system_header	= out.flgs[flags_index++].flag;
	res.no_timestamp	= out.flgs[flags_index++].flag;
	res.no_mu_stubs		= out.flgs[flags_index++].flag;
	res.no_includes		= out.flgs[flags_index++].flag;
	
	assert(flags_index == (STD_FLAGS + BE_FLAGS));
	
	return res;
}

/* End of file. */

