/*
 * Copyright (c) 1995, 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/compiler.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>

/* This is the primary accessor method for on inline_state.
   Given a slot index number (a pres_c_inline_index)
   into the "current" C function or structure,
   returns in `out_expr' a C expression
   that can be used in generated code to access that slot at runtime.
   Also returns, in `out_type', the C type of that slot.

   If `slot' is >= 0,
   then it indicates a structure member or function parameter,
   depending on whether this inline_state
   represents a function or a structure.
   Slot number -1 represents a function's return value;
   in that case the inline_state had better represent a function
   rather than a structure.
*/
void struct_inline_state::slot_access(int slot, cast_expr *out_expr, cast_type *out_type)
{
	cast_type type;
	cast_expr expr;

	assert(struct_type != 0);
	assert(var_expr != 0);

	assert(slot >= 0);
	assert(slot < (signed int)struct_type->slots.slots_len);
	assert(struct_type->slots.slots_val[slot].name);

	/* `slot' selects a member of a structure,
	   and `var_expr' is an expression
	   referring to the structure as a whole.
	   Build an expression to access that member.  */
	type = struct_type->slots.slots_val[slot].type;
	expr = cast_new_expr_sel(var_expr, struct_type->slots.slots_val[slot].name);

	assert(expr != 0);
	assert(type != 0);

	*out_expr = expr;
	*out_type = type;
}

void func_inline_state::slot_access(int slot, cast_expr *out_expr, cast_type *out_type)
{
	cast_type type;
	cast_expr expr;

	assert(func_type != 0);

	if (slot >= 0)
	{
		assert(slot < (signed int)func_type->params.params_len);
		assert(func_type->params.params_val[slot].name);

		type = func_type->params.params_val[slot].type;
		expr = cast_new_expr_name(func_type->params.params_val[slot].name);
	}
	else
	{
		/* Select the function return value.
		   It is always kept in a magic variable named `_return'.  */

		assert(slot == -1);

		type = func_type->return_type;
		expr = cast_new_expr_name("_return");
	}

	assert(expr != 0);
	assert(type != 0);

	*out_expr = expr;
	*out_type = type;
}

