/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <stdio.h>
#include <string.h>

#ifndef __CYGWIN32__ /* Is this the correct test for a non-Windows platform? */
#define SYS_INFO_AVAILABLE
/* used for w_version_info */ 
#include <sys/types.h>
#include <pwd.h>

#ifndef MAXHOSTNAMELEN
#include <sys/param.h>
#ifndef MAXHOSTNAMELEN
#include <netdb.h>
#endif
#endif 

#include <unistd.h>
#endif /* __CYGWIN32__ */

#include <time.h>
#include <mom/compiler.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <mom/c/pbe.hh>

static pres_c_1 pres;

FILE *header_file;
FILE *c_file;
FILE *input_file;
cast_expr emergency_return_value;

char *progname;

/* Write the function prototypes, and then generate and write
   the stub code. 
*/

static void f_close(FILE *fh)
{
	if (fclose(fh))
		perror(NULL);
}

void w_version_info(char *outname, char *inname, int suppress_version_info)
{
	/*
	 * Produce an identifying comment block like the following:
	 *
	 *      * DO NOT EDIT!  THIS FILE IS MACHINE-GENERATED! *
	 *
	 *      <filename.c>
	 *      created by <user>@<host> on <date>
	 *      from file <filename.prc>
	 *      with Flick version <version> (compiled on <date>)
	 *	send bug reports to <flick-bugs@cs.utah.edu>
	 */
	time_t t;
#ifdef SYS_INFO_AVAILABLE
	char hostname[MAXHOSTNAMELEN];
	
	gethostname(hostname, MAXHOSTNAMELEN);
#endif
	if (suppress_version_info)
		return;
	
	t = time(NULL);
	w_printf("/*\n");
	w_printf(" *\t* DO NOT EDIT!  THIS FILE IS MACHINE-GENERATED! *\n");
	w_printf(" *\n");
	w_printf(" *\t%s\n", outname);
	w_printf(" *\tcreated ");
#ifdef SYS_INFO_AVAILABLE
	w_printf("by %s@%s ",
		 getpwuid(getuid())->pw_name,
		 hostname);
#endif
	w_printf("on %s", /* `ctime' emits a \n */ ctime(&t));
	w_printf(" *\tfrom file %s\n", inname);
	w_printf(" *\twith Flick version " FLICK_VERSION);
	w_printf(" (compiled on "
		 __DATE__ " " __TIME__
		 ")");
	w_printf("\n");
	w_printf(" *\tsend bug reports to <flick-bugs@cs.utah.edu>\n");
	w_printf(" */\n\n");
}

int main(int argc, char **argv)
{
	unsigned int i;
	be_flags res;
	res = get_default_be_flags();
	res = be_args(argc, argv, res);
	
	if (res.input && !res.output)
		res.output = resuffix(res.input, ".c");
	if (res.output && !res.header)
		res.header = resuffix(res.output, ".h");
	if (res.input && !res.header)
		res.header = resuffix(res.input, ".h");
	
	/* Open the input files. */
	input_file = res.input ? fopen(res.input, "rb") : stdin;
	if (!input_file)
		panic("Unable to open input file `%s'",
		      (res.input ? res.input : "<stdin>"));
	
	/* Open the output files. */
	c_file = res.output ? fopen(res.output, "w") : stdout;
	if (!c_file)
		panic("Unable to open output file `%s'",
		      (res.output ? res.output : "<stdout>"));
	
	header_file = (res.header ? fopen(res.header, "w") :
				    fopen("/dev/null", "w"));
	if (!header_file)
		panic("Unable to open header file `%s'",
		      (res.header ? res.header : "/dev/null"));
	
	/* Read our input data. */
	pres_c_1_readfh(&pres, input_file);
	
	/* Print the header file out */
	w_set_fh(header_file);
	w_version_info(res.header, res.input, res.no_timestamp);
	w_header_prologue(&res, &pres);
	w_header_includes(&pres);
	
	if (res.no_mu_stubs) {
		/*
		 * Mark all marshal and unmarshal stubs as IMPLIED.  This will
		 * cause them not to be output.
		 */
		for (unsigned int j = 0; j < pres.stubs.stubs_len; j++) {
			if ((pres.stubs.stubs_val[j].kind
			     == PRES_C_MARSHAL_STUB)
			    || (pres.stubs.stubs_val[j].kind
				== PRES_C_UNMARSHAL_STUB)) {
				int cast_index = pres.stubs.stubs_val[j].
						 pres_c_stub_u.mstub.c_func;
				
				pres.cast.cast_scope_val[cast_index].included
					= IMPLIED;
			}
		}
	}
	
	for (i = 0; i < pres.cast.cast_scope_len; i++) {
		if (pres.cast.cast_scope_val[i].included == IMPLIED)
			continue;
		
		if ((pres.cast.cast_scope_val[i].included == INCLUDED)
		    && res.no_includes) {
			/* continue; */
			/*
			 * XXX --- If `B.idl' includes `A.idl', then `B.h'
			 * should include `A.h'.  But we can't do that yet,
			 * because we don't keep track of the exact files in
			 * which constructs are defined.  Until we do so, we
			 * must emit `A.h' typedefs into `B.h', or else `B.c'
			 * won't have any access to those types at all!
			 *
			 * So for now we weed out only functions declarations
			 * and definitions.  This is really too strong, since
			 * we may weed out marshal/unmarshal stubs that we will
			 * need later on.  But it's not a problem in practice
			 * right now because we inline stubs so aggressively.
			 */
			if ((pres.cast.cast_scope_val[i].u.kind
			     == CAST_FUNC_DECL) ||
			    (pres.cast.cast_scope_val[i].u.kind
			     == CAST_FUNC_DEF))
				continue;
		}
		
		cast_w_def(&pres.cast, i, 0);
		w_printf("\n");
	}
	
	w_header_epilogue(&res, &pres);
	if (res.header)
		f_close(header_file);
	
	/* Print the C file out. */
	w_set_fh(c_file);
	w_version_info(res.output, res.input, res.no_timestamp);
		
	/*
	 * Include an entire copy of the header file in with the source file,
	 * if the header file has gone to /dev/null...
	 * If it has been output,
	 * use the selected location to build the #include "asdf" statement
	 */
	if (!res.header) {
		w_header_includes(&pres);
		for (i = 0; i < pres.cast.cast_scope_len; i++) {
			if (pres.cast.cast_scope_val[i].included == IMPLIED)
				continue;
			
			if ((pres.cast.cast_scope_val[i].included == INCLUDED)
			    && res.no_includes) {
				/* continue; */
				/*
				 * XXX --- See comment above as to why we only
				 * weed out functions.
				 */
				if ((pres.cast.cast_scope_val[i].u.kind
				     == CAST_FUNC_DECL) ||
				    (pres.cast.cast_scope_val[i].u.kind
				     == CAST_FUNC_DEF))
					continue;
			}
			
			cast_w_def(&pres.cast, i, 0);
			w_printf("\n");
		}
	} else
		w_source_includes(res.header, res.prefix, res.system_header);
	w_printf("\n");
	
	do_main_output(&pres);
	
	/* Print the stubs out. */
	for (i = 0; i < pres.stubs.stubs_len; i++) {
		inclusion included = NOT_INCLUDED;
		
		switch (pres.stubs.stubs_val[i].kind) {
		case PRES_C_MARSHAL_STUB:
		case PRES_C_UNMARSHAL_STUB:
			included = pres.cast.cast_scope_val[
				pres.stubs.stubs_val[i].
				pres_c_stub_u.mstub.c_func].included;
			break;
			
		case PRES_C_CLIENT_STUB:
			included = pres.cast.cast_scope_val[
				pres.stubs.stubs_val[i].
				pres_c_stub_u.cstub.c_func].included;
			break;
			
		case PRES_C_SERVER_SKEL:
			included = pres.cast.cast_scope_val[
				pres.stubs.stubs_val[i].
				pres_c_stub_u.sskel.c_def].included;
			break;
			
		case PRES_C_SEND_STUB:
			included = pres.cast.cast_scope_val[
				pres.stubs.stubs_val[i].
				pres_c_stub_u.send_stub.c_func].included;
			break;
			
		case PRES_C_RECEIVE_FUNC:
			panic("Unknown PRES_C_STUB_KIND!");
			break;
			
		case PRES_C_REPLY_STUB:
			included = pres.cast.cast_scope_val[
				pres.stubs.stubs_val[i].
				pres_c_stub_u.reply_stub.c_func].included;
			break;
		}
		
		if (included == NOT_INCLUDED ||
		    (included == INCLUDED && !res.no_includes))
			w_stub(&pres, i);
	}
}

/* End of file. */

