/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>

/* This function contains generic array processing code
   for all varieties and presentations of arrays;
   it is called by specific array-handling routines
   such as mu_mapping_fixed_array and mu_inline_counted_array.

   When this function is called, the array length must already be known:
   we're ready to generate the code to m/u the actual array elements.
   If unmarshaling, the memory block into which to unmarshal the array
   will be allocated, if necessary.  Likewise, when marshaling, the
   memory block from which the array is marshaled will be deallocated.

   `array_expr' is a C expression evaluating (at run-time)
   to a pointer to the first element of the array buffer
   (i.e. the buffer that is "presented to" or gotten from the user program).
   From this expression will be built the actual expressions
   used to marshal/unmarshal the elements of the array.

   `array_alloc' is the allocation information indicating when and how
   allocation should be done.

   `elem_ctype' is the C (presentation) type of an element of the array.
   (The C type of the `array_expr' is a pointer to this type.)

   `elem_itype' is the MINT type of an element of the array.

   `elem_map' is the pres_c_mapping node
   representing the mapping of each element of the array
   (_not_ the mapping of the array as a whole).

   `len_expr' is a C expression evaluating to the length of the array.
   For fixed-length arrays, this is generally a constant expression;
   for variable-length arrays it will presumably refer to some variable
   containing the actual length of the array
   as determined dynamically, at runtime.
   (Again, any array length determination code necessary
   has already been produced by the time this function is called;
   `len_expr' simply encapsulates that length
   for use in the rest of the array processing code.)

   `len_ctype' specifies the C type of the array length -
   it is always some kind of integer,
   but may vary in size according to the maximum length of the array
   or other factors.
   The array processing code uses this C type
   when declaring variables to hold array indexes and counts;
   in particular, it is used for the array iterator variable
   that this routine declares
   in order to iterate over the elements of the array.

   `len_min' and `len_max' indicate the array length minimum and
   maximum values.  These are equal if it is a fixed array.

   */

void mu_state::mu_array(
	cast_expr array_expr,
	cast_type array_ctype,
	pres_c_allocation *array_alloc,
	cast_type elem_ctype,
	mint_ref elem_itype,
	pres_c_mapping elem_map,
	cast_expr len_expr, cast_type len_ctype,
	unsigned long len_min, unsigned long len_max)
{
	/* Allocate space */
	if ((op & MUST_ALLOCATE) &&
	    (array_ctype->kind == CAST_TYPE_POINTER))
		mu_pointer_alloc(array_expr, elem_ctype,
				 len_expr, array_alloc);

	/* Check for simple allocation/deallocation
	 * (elem_itype is a MINT_VOID).
	 * Note: We don't let this case go unchecked because we would
	 * end up generating an array iteration variable, and
	 * potentially a useless (empty) loop body.
	 */
	assert(elem_itype >= 0
	       && (unsigned) elem_itype < pres->mint.defs.defs_len);
	if (pres->mint.defs.defs_val[elem_itype].kind == MINT_VOID) {
		/* Deallocate space */
		if ((op & MUST_DEALLOCATE) &&
		    (array_ctype->kind == CAST_TYPE_POINTER))
			mu_pointer_free(array_expr, elem_ctype,
					len_expr, array_alloc);
		/* done! */
		return;
	}
	
	if (len_max > 1) {
		assert(len_ctype);
		/* Declare an iterator variable.  */
		cast_expr iter_var_expr = add_temp_var("array_iter", len_ctype);
		
		/* From the array-as-a-whole pointer expression,
		   produce a current-element-of-the-array expression,
		   by adding the current array index to the pointer
		   and dereferencing it.
		   (The combination is equivalent to C's [] operator.)  */
		cast_expr elem_expr =
			cast_new_unary_expr(
				CAST_UNARY_DEREF,
				cast_new_binary_expr(CAST_BINARY_ADD,
						     array_expr, iter_var_expr));
		
		/* Build a for statement to iterate through the array.  */
		/* XXX on decode, ensure that the incoming array doesn't have too many
		   elements - throw the rest away or throw an exception?  */
		cast_stmt for_stmt = cast_new_for(
			cast_new_expr_assign(iter_var_expr,
					     cast_new_expr_lit_int(0, 0)),
			cast_new_binary_expr(CAST_BINARY_LT, iter_var_expr, len_expr),
			cast_new_unary_expr(CAST_UNARY_POST_INC, iter_var_expr),
			0 /* The body will be built incrementally below. */ );
		
		cast_stmt old_c_block = c_block;
		c_block = 0;
		mu_array_elem(elem_expr, elem_ctype, elem_itype, elem_map,
			      len_min, len_max);
		
		for_stmt->cast_stmt_u_u.s_for.stmt = c_block;
		c_block = old_c_block;
		
		/* add only if there's something to add */
		if (for_stmt->cast_stmt_u_u.s_for.stmt)
			add_stmt(for_stmt);
	} else if (len_max > 0) {
		/* There is only one element to m/u, thus a simple
		   dereference of the pointer will produce the correct
		   element expression.
		   (This is equivalent to ptr[0] or *ptr in C) */
		cast_expr elem_expr =
			cast_new_unary_expr(
				CAST_UNARY_DEREF,
				array_expr);
		
		mu_array_elem(elem_expr, elem_ctype, elem_itype, elem_map,
			      len_min, len_max);
	}
	
	/* Deallocate space */
	if ((op & MUST_DEALLOCATE) &&
	    (array_ctype->kind == CAST_TYPE_POINTER))
		mu_pointer_free(array_expr, elem_ctype,
				len_expr, array_alloc);
}

/* End of file. */

