/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>
#include <mom/compiler.h>

static void ptrlevel(mu_state *must, 
		     cast_expr pexpr, cast_type pctype, pres_c_mapping map,
		     mint_ref elem_itype,
		     int ptr_levels, cast_expr lexpr, cast_type lctype,
		     unsigned long len_min, unsigned long len_max)
{
	pres_c_direction old_dir = must->current_param_dir;
	
	/*
	 * If the CAST type of our array is a named type, locate the actual
	 * array type.  (Note: The MIG FE/PG makes these kinds of direct
	 * associations between named C types and INLINE_COUNTED_ARRAY
	 * nodes.  Other PG's generally indirect through a MAPPING_STUB, and
	 * the named type is derefenced to an array type there.)
	 */
	pctype = cast_find_typedef_type(&(must->pres->cast), pctype);
	if (!pctype)
		panic("In `mu_state::mu_inline_counted_array', "
		      "can't determine the C type of the array elements!");
	
	/* Find the type of the array elements.  */
	cast_type elem_ctype;
	assert(pctype->kind == CAST_TYPE_POINTER ||
	       pctype->kind == CAST_TYPE_ARRAY);
	if (pctype->kind == CAST_TYPE_ARRAY)
		elem_ctype = pctype->cast_type_u_u.array_type.element_type;
	else
		elem_ctype = pctype->cast_type_u_u.pointer_type.target;
	
	/* Spit out memory allocation code if necessary.  */
	if (map->kind == PRES_C_MAPPING_DIRECTION) {
		old_dir = must->current_param_dir;
		must->current_param_dir = map->pres_c_mapping_u_u.direction.dir;
		map = map->pres_c_mapping_u_u.direction.mapping;
	}
	assert(map->kind == PRES_C_MAPPING_POINTER);
	pres_c_mapping_pointer *pmap = &map->pres_c_mapping_u_u.pointer;

	if (ptr_levels == 0)
	{
		/* Marshal/unmarshal the array itself.  */
		must->mu_array(pexpr, pctype, &(pmap->alloc),
			       elem_ctype, elem_itype, pmap->target,
			       lexpr, lctype,
			       len_min, len_max);
	}
	else
	{
		cast_expr len_expr = cast_new_expr_lit_int(1, 0);
		/* This is only an intermediate pointer level,
		   so only allocate enough memory for a single item.  */
		must->mu_pointer_alloc(pexpr, elem_ctype,
				       len_expr, &pmap->alloc);

		/* To access the stuff this pointer points to, just dereference the pointer.  */
		cast_expr to_expr = cast_new_unary_expr(CAST_UNARY_DEREF, pexpr);

		/* Recursively descend through the pointer mapping.  */
		ptrlevel(must, to_expr, elem_ctype, pmap->target,
			 elem_itype, ptr_levels-1, lexpr, lctype,
			 len_min, len_max);

		/* Spit out deallocation code for this pointer.  */
		must->mu_pointer_free(pexpr, elem_ctype,
				      len_expr, &pmap->alloc);
	}
	
	must->current_param_dir = old_dir;
}

/* This routine handles counted-array presentations.
   First marshal/umarshal the count,
   then use the generic mu_array() method to handle the array itself.
   This array presentation always uses a pointer variable 
   to hold the address of the first element of the array,
   so also call mu_pointer_alloc() and mu_pointer_free()
   to handle allocation/deallocation semantics for that pointer.

   XXX currently assumes array is variable-length.
*/
void mu_state::mu_inline_counted_array(inline_state *ist,
				       mint_ref itype, pres_c_inline inl)
{
	assert(itype >= 0);
	assert(itype < (signed int) pres->mint.defs.defs_len);
	
	mint_def *def = &pres->mint.defs.defs_val[itype];
	assert(def->kind == MINT_ARRAY
	       || def->kind == MINT_VOID);

	mint_array_def *adef;
	unsigned len_min, len_max;

	if (def->kind == MINT_ARRAY) {
		adef = &def->mint_def_u.array_def;
		
		mint_get_array_len(&pres->mint, itype,
				   &len_min, &len_max);
	} else {
		adef = 0;
		len_min = 0;
		len_max = ~0L;
	}
	
	pres_c_inline_counted_array *cainl
		= &(inl->pres_c_inline_u_u.counted_array);
	assert(cainl->ptr.mapping);
	
	mu_state_arglist *savelist = arglist;
	arglist = new mu_state_arglist;
	arglist->add("length");  /* a.k.a. the # of items in the array */
	arglist->add("maximum"); /* a.k.a. the allocated length */
	
	cast_expr len_cexpr = 0;
	cast_type len_ctype = 0;
	
	/* Marshal/unmarshal the array length. */
	assert(cainl->len);
	if (adef) {
		array_def = adef;
		mu_inline(ist, adef->length_type, cainl->len);
	} else {
		array_def = (mint_array_def *)1;
		mu_inline(ist, pres->mint.standard_refs.unsigned32_ref, cainl->len);
	}
	array_def = 0;
	
	/* Find the (required) length expression and C type. */
	arglist->getargs("length", &len_cexpr, &len_ctype);
	assert(len_cexpr);
	assert(len_ctype);
	
	/*
	 * Optionally marshal/unmarshal the ``maximum'' array length --- i.e.,
	 * the number of slots in the allocated vector.  This is expected to
	 * result in side-effects only; in general, no actual data will be
	 * marshaled or unmarshaled.
	 */
	if (adef && cainl->max) {
		mu_inline(ist, adef->length_type, cainl->max);
		
		/* Find the (optional) maximum length expression and C type. */
		cast_expr max_cexpr;
		cast_type max_ctype;
		
		arglist->getargs("maximum", &max_cexpr, &max_ctype);
		if (max_cexpr) {
			/*
			 * When unmarshaling, set the maximum (allocated)
			 * length to be the same as the data (#-of-elements)
			 * length.
			 *
			 * XXX --- This isn't really ``right.''  The function
			 * that allocates the buffer (`mu_pointer_alloc')
			 * should set the maximum field, since presumably that
			 * function may perform some kind of optimization on
			 * the length.
			 */
			if (op & MUST_DECODE)
				add_stmt(cast_new_stmt_expr(
					cast_new_expr_assign(max_cexpr,
							     len_cexpr)
					));
		}
	}
	
	/* Find the first-element-pointer expression and C type. */
	cast_expr pexpr; cast_type pctype;
	ist->slot_access(cainl->ptr.index, &pexpr, &pctype);
	
	/*
	 * Consume all the levels of pointer indirection indicated by
	 * `ptr_levels' and eventually handle the array elements.
	 */
	ptrlevel(this, pexpr, pctype, cainl->ptr.mapping,
		 ((adef)? adef->element_type
		  : pres->mint.standard_refs.void_ref),
		 cainl->ptr_levels, len_cexpr, len_ctype, len_min, len_max);
	
	/* free(arglist); */
	arglist = savelist;
}

/* End of file. */

