/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <mom/c/pbe.hh>

/*
 * This function handles fixed-array presentations, which map to C array or
 * pointer data types.
 *
 * The current MINT type must be a MINT_ARRAY (or MINT_VOID --- see below), but
 * it doesn't necessarily have to be a fixed-length array.  If it's a variable-
 * length array, then on decode (receive), the incoming array is unmarshaled
 * into the first `n' elements of the C vector, where `n' is the length
 * specified in the mapping.  The actual number of elements received is simply
 * lost.
 *
 * XXX ---- Haven't decided what to do when too many elements are received.
 * More generally, this code may not yet handle cases in which the MINT array,
 * the mapping, and the presented C vector don't all have the same length.
 *
 * On encode, exactly `n' elements are always sent.
 *
 * As a special case, if the current MINT type is a MINT_VOID, that indicates
 * that the array should be allocated/freed, but that there is no message data
 * to be marshaled or unmarshaled.  This is how we convince a server dispatch
 * function to allocate space for `out' arrays as part of its handling of a
 * request.
 */
void mu_state::mu_mapping_fixed_array(
	cast_expr expr,
	cast_type ctype,
	mint_ref itype,
	pres_c_mapping_fixed_array *array_mapping)
{
	assert((itype >= 0)
	       && (itype < (signed int) pres->mint.defs.defs_len));
	
	mint_def *def = &(pres->mint.defs.defs_val[itype]);
	mint_array_def *adef;
	unsigned len_min, len_max;
	
	/*********************************************************************/
	
	assert((def->kind == MINT_ARRAY) || (def->kind == MINT_VOID));
	
	if (def->kind == MINT_ARRAY) {
		/*
		 * We are going to marshal or unmarshal message data as
		 * described by the current MINT_ARRAY.
		 */
		adef = &(def->mint_def_u.array_def);
		mint_get_array_len(&pres->mint, itype,
				   &len_min, &len_max);
	} else {
		/*
		 * We are just going to allocate space for the C presentation
		 * of an array.  There is no actual message data to process.
		 */
		adef = 0;
		len_min = len_max = 1;
	}
	
	/*
	 * If the CAST type of our array is a named type, locate the actual
	 * array/pointer type.  (Note: The MIG FE/PG makes these kinds of
	 * direct associations between named C types and MAPPING_FIXED_ARRAY
	 * nodes.  Other PG's generally indirect through a MAPPING_STUB, and
	 * the named type is derefenced to an array/pointer type there.)
	 */
	if (ctype->kind == CAST_TYPE_NAME) {
		ctype = cast_find_typedef_type(&(pres->cast), ctype);
		if (!ctype)
			panic("In `mu_state::mu_mapping_fixed_array', "
			      "can't locate `typedef' for a named type.");
	}
	
	/*
	 * Determine the CAST types of the array length and the array elements.
	 */
	cast_type length_ctype;
	cast_type element_ctype;
	
	if (adef)
		length_ctype = mint_to_ctype(&(pres->mint), adef->length_type);
	else
		/*
		 * Is this always safe?  I think so; the length is going to be
		 * a literal integer anyway.
		 */
		length_ctype = cast_new_prim_type(CAST_PRIM_INT, 0);
	
	switch (ctype->kind) {
	case CAST_TYPE_ARRAY:
		element_ctype = ctype->cast_type_u_u.array_type.element_type;
		break;
		
	case CAST_TYPE_POINTER:
		element_ctype = ctype->cast_type_u_u.pointer_type.target;
		break;
		
	default:
		panic("In `mu_state::mu_mapping_fixed_array', "
		      "can't determine the C type of the array elements.");
		break;
	}
	
	/*
	 * If we have a MINT_ARRAY (i.e., are really processing message data),
	 * marshal/unmarshal the array length into/out of a temporary variable
	 * so that we can marshal/unmarshal from/to it as usual.  The C
	 * compiler should be able to optimize away the variable easily.
	 *
	 * Note that it's valid for a variable-length array to have a
	 * fixed-array presentation: in that case, the actual length is lost on
	 * receive, and on send the length is always set to the fixed array
	 * mapping's length.
	 *
	 * Of course, if we're not actually marshaling/unmarshaling anything
	 * (i.e., our current MINT type is MINT_VOID), then we aren't going to
	 * marshal/unmarshal a length, either.
	 */
	cast_expr length_cexpr;
	
	length_cexpr = cast_new_expr_lit_int(array_mapping->length, 0);
	
	/*
	 * Marshal/unmarshal the array itself.
	 *
	 * XXX --- On decode, ensure that the incoming array doesn't have too
	 * many elements.  Throw the rest away or throw an exception?
	 */
	
	/* allocation is done by mu_array */
	mu_array(expr, ctype, &(array_mapping->alloc),
		 element_ctype, (adef ? adef->element_type : itype),
		 array_mapping->element_mapping,
		 length_cexpr, length_ctype,
		 len_min, len_max);
}

/* End of file. */

