/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>
#include <stdlib.h>

#include <mom/compiler.h>
#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include <mom/c/pbe_mem.hh>

/*
 * It is sometimes desirable to allocate storage for string data before the
 * actual length of the string is known.  For example, if we know that the
 * string length will always be between 0 and 255 characters, it may be useful
 * to preallocate an array of 256 characters (as `static' or `auto' storage) to
 * hold the string data.  The potential for wasted space is outweighed by the
 * potential for increased code execution speed.
 *
 * We decide when to use this preallocation technique based on the range in the
 * length of the string to be marshaled/unmarshaled.
 *
 * `string_prealloc_range_limit' is the maximum allowable range in the string
 * length for which we will generate code to use preallocated string space
 * storage (based on the maximum length of the string).  Essentially, this
 * value is the amount of space we are willing to ``waste'' in order to get a
 * potential speed benefit.
 *
 * For strings that have ranges greater than `string_prealloc_range_limit', we
 * generate code that counts the number of characters in the string and
 * dynamically allocates storage based on the actual string length.
 *
 * XXX --- This value should be stored in a slot in the `mu_state' object,
 * settable with a command line switch.
 *
 * XXX --- This technique should be generalized for all kinds of arrays.
 */
const unsigned int string_prealloc_range_limit = 255;

/*
 *
 */
void mem_mu_state::mu_mapping_string(cast_expr ptr,
				     cast_type ptr_ctype,
				     mint_array_def *arr,
				     pres_c_allocation *mem_alloc)
{
	char *macro_name;
	
	unsigned int string_length_range;
	unsigned int max_string_length;
	cast_expr max_string_length_cexpr;
	
	cast_expr mem_alloc_func_cexpr;
	cast_type ptr_target_ctype;
	
	cast_expr be_expr;
	
	cast_expr cex;
	
	int size, align_bits;
	mint_ref itype = arr->element_type;
	
	/*********************************************************************/
	
	if (op & (MUST_DECODE | MUST_ENCODE)) {
		// break_chunk();
		// new_chunk_align(2);
		/* Find the marshaling parameters for this primitive type.  */
		get_prim_params(itype, &size, &align_bits, &macro_name);
		itype = arr->length_type;
		
		assert(pres->mint.defs.defs_val[itype].kind == MINT_INTEGER);
		
		string_length_range = pres->mint.defs.defs_val[itype].
				      mint_def_u.integer_def.range;
		/* XXX --- should handle overflow, etc. */
		max_string_length = (pres->mint.defs.defs_val[itype].
				     mint_def_u.integer_def.min
				     + string_length_range);
		max_string_length_cexpr =
			cast_new_expr_lit_int(max_string_length, 0);
		
		/* Determine how we should allocate/deallocate memory. */
		if (op & MUST_ALLOCATE)
			mem_alloc_func_cexpr =
				cast_new_expr_name(get_allocator(mem_alloc));
		else if (op & MUST_DEALLOCATE)
			mem_alloc_func_cexpr =
				cast_new_expr_name(get_deallocator(mem_alloc));
		else
			/* XXX? */
			mem_alloc_func_cexpr =
				cast_new_expr_name("null_flick_free");
		
		be_expr = cast_new_expr_name(get_be_name());
		
		/*
		 * Decide which marshal/unmarshal method to use, and output the
		 * necessary CAST for each case.
		 */
		if ((max_string_length >= max_glob_size)
		    || ((op & MUST_ALLOCATE)
			/*
			 * XXX --- The `auto_alloc' hack.  See `mu_state::get_
			 * allocator' in `mu_pointer_alloc.cc'.
			 */
			&& (mem_alloc->flags & PRES_C_ALLOC_ALWAYS)
			&& !strcmp(get_which_stub(), "server")
			&& (current_param_dir == PRES_C_DIRECTION_IN)
			&& ((mem_alloc->flags & PRES_C_DEALLOC_EVER)
			    == PRES_C_DEALLOC_ALWAYS)
			    )) {
			/*
			 * The `longstring' macro can handle strings that are
			 * larger than a single glob.  It allocates/deallocates
			 * space for the string and moves the bytes.
			 */
			if ((op & MUST_ALLOCATE)
			    /* XXX --- The `auto_alloc' hack. */
			    && (mem_alloc->flags & PRES_C_ALLOC_ALWAYS)
			    && !strcmp(get_which_stub(), "server")
			    && (current_param_dir == PRES_C_DIRECTION_IN)
			    && ((mem_alloc->flags & PRES_C_DEALLOC_EVER)
				== PRES_C_DEALLOC_ALWAYS)
				)
				macro_name
					= flick_asprintf(
						"flick_%s_%s_auto_string",
						get_encode_name(),
						get_buf_name());
			else
				macro_name
					= flick_asprintf(
						"flick_%s_%s_longstring",
						get_encode_name(),
						get_buf_name());
			cex = cast_new_expr_call(
				cast_new_expr_name(macro_name),
				3);
			cex->cast_expr_u_u.call.params.params_val[0]
				= ptr;
			cex->cast_expr_u_u.call.params.params_val[1]
				= mem_alloc_func_cexpr;
			cex->cast_expr_u_u.call.params.params_val[2]
				= be_expr;
			
			break_glob();
			add_stmt(cast_new_stmt_expr(cex));
			
		} else if ((!strcmp(get_which_stub(), "client")
			    && ((mem_alloc->flags & PRES_C_ALLOC_EVER)
				== PRES_C_ALLOC_NEVER))
			   || ((string_length_range <= string_prealloc_range_limit)
			       && ((!strcmp(mem_alloc->allocator, "static")
				    || (!strcmp(mem_alloc->allocator, "auto"))
				/*
				 * XXX --- The `auto_alloc' hack, which forces
				 * deep auto allocation for `in' parameters.
				 * The following condition is the test that
				 * `get_allocator' uses to force auto
				 * allocation.
				 */
				/*
				 * XXX --- We do *not* do this because the
				 * `auto' allocator doesn't work right when
				 * we're within a union, or a sequence, or...
				 * In these cases, declaring a local is *not*
				 * the same as calling `alloca'!
				 *
				 * || (auto_alloc
				 *     && (mem_alloc->flags &
				 *         PRES_C_ALLOC_ALWAYS))
				 */
				       ))
				   )) {
			/*
			 * The `alloced_string' macro moves data into a
			 * separately allocated buffer.
			 */
			macro_name
				= flick_asprintf("flick_%s_%s_alloced_string",
						 get_encode_name(),
						 get_buf_name());
			cex = cast_new_expr_call(
				cast_new_expr_name(macro_name),
				3);
			cex->cast_expr_u_u.call.params.params_val[0]
				= ptr;
			cex->cast_expr_u_u.call.params.params_val[1]
				= be_expr;
			cex->cast_expr_u_u.call.params.params_val[2]
				= max_string_length_cexpr;

			if (ptr_ctype->kind == CAST_TYPE_POINTER)
				ptr_target_ctype = ptr_ctype->cast_type_u_u.
						   pointer_type.target;
			else
				ptr_target_ctype = ptr_ctype->cast_type_u_u.
						   array_type.element_type;
			
			break_glob(); /* XXX --- Necessary? KBF - Yes. */
			if ((op & MUST_ALLOCATE) &&
			    (ptr_ctype->kind == CAST_TYPE_POINTER))
				mu_pointer_alloc(ptr,
						 ptr_target_ctype,
						 /* +1 for terminating NUL. */
						 cast_new_expr_lit_int(
							 max_string_length + 1,
							 0),
						 mem_alloc);
			add_stmt(cast_new_stmt_expr(cex));
			if ((op & MUST_DEALLOCATE) &&
			    (ptr_ctype->kind == CAST_TYPE_POINTER))
				mu_pointer_alloc(ptr,
						 ptr_target_ctype,
						 /* +1 for terminating NUL. */
						 cast_new_expr_lit_int(
							 max_string_length + 1,
							 0),
						 mem_alloc);
			
		} else {
			/*
			 * The `string' macro is the default method.  It
			 * allocates/deallocates space for the string data and
			 * moves bytes.
			 */
			macro_name = flick_asprintf("flick_%s_%s_string",
						    get_encode_name(),
						    get_buf_name());
			cex = cast_new_expr_call(
				cast_new_expr_name(macro_name),
				4);
			cex->cast_expr_u_u.call.params.params_val[0]
				= ptr;
			cex->cast_expr_u_u.call.params.params_val[1]
				= mem_alloc_func_cexpr;
			cex->cast_expr_u_u.call.params.params_val[2]
				= be_expr;
			cex->cast_expr_u_u.call.params.params_val[3]
				= max_string_length_cexpr;
			
			break_glob(); /* XXX --- Necessary? KBF - Yes, necessary... */
			add_stmt(cast_new_stmt_expr(cex));
		}
		this->align_bits = 0;
		this->align_ofs = 0;
		/* since these macros do their own globbing/chunking,
		   we need to update the max_msg_size ourselves */
		if ((max_msg_size + max_string_length + 1) <= max_msg_size)
			/* overflow */
			max_msg_size = MAXUINT_MAX;
		else
			max_msg_size += max_string_length + 1;
	}
}

/* End of file. */

