/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <mom/c/pbe.hh>

/*
 * This routine handles terminated-array presentations.
 *
 * For encoding, we first count the number of array elements before the
 * terminator, then marshal the count, and finally use the generic mu_array()
 * method to handle the array itself.  For decoding, we unmarshal the count,
 * then unmarshal the array itself, and finally add the array terminator just
 * past the last element.
 *
 * This array presentation always uses a pointer variable to hold the address
 * of the first element of the array, so also call mu_pointer_alloc() and
 * mu_pointer_free() to handle allocation/deallocation semantics for that
 * pointer.  Of course, we have to add 1 to account for the terminator.
 */

void mu_state::mu_mapping_terminated_array(
	cast_expr expr,
	cast_type ctype, 
	mint_ref itype, 
	pres_c_mapping_terminated_array *tmap)
{
	assert(itype >= 0);
	assert(itype < (signed int) pres->mint.defs.defs_len);
	
	mint_def *def = &pres->mint.defs.defs_val[itype];
	assert((def->kind == MINT_ARRAY)|| (def->kind == MINT_VOID));
	mint_array_def *adef;
	unsigned len_min, len_max;

	if (def->kind == MINT_ARRAY) {
		adef = &def->mint_def_u.array_def;
		mint_get_array_len(&pres->mint, itype,
				   &len_min, &len_max);
	} else {
		adef = 0;
		len_min = len_max = 1;
	}
	
	/*
	 * If the CAST type of our array is a named type, locate the actual
	 * array/pointer type.  (Note: The MIG FE/PG makes these kinds of
	 * direct associations between named C types and MAPPING_FIXED_ARRAY
	 * nodes.  Other PG's generally indirect through a MAPPING_STUB, and
	 * the named type is derefenced to an array/pointer type there.)
	 */
	ctype = cast_find_typedef_type(&(pres->cast), ctype);
	if (!ctype)
		panic("In `mu_state::mu_mapping_terminated_array', "
		      "can't locate `typedef' for a named type.");
	
	/* Find the first-element-pointer expression and C type. */
	cast_type ptr_ctype = ctype;
	
	while (ptr_ctype->kind == CAST_TYPE_QUALIFIED)
		ptr_ctype = ptr_ctype->cast_type_u_u.qualified.actual;
	
	assert(ptr_ctype->kind == CAST_TYPE_POINTER ||
	       ptr_ctype->kind == CAST_TYPE_ARRAY);
	
	/* Find the type of the array elements.  */
	cast_type elem_ctype;
	if (ptr_ctype->kind == CAST_TYPE_POINTER)
		elem_ctype = ptr_ctype->cast_type_u_u.pointer_type.target;
	else
		elem_ctype = ptr_ctype->cast_type_u_u.array_type.element_type;
	
	cast_type unqualified_elem_ctype = elem_ctype;
	
	while (unqualified_elem_ctype->kind == CAST_TYPE_QUALIFIED)
		unqualified_elem_ctype = unqualified_elem_ctype->cast_type_u_u.
					 qualified.actual;
	
	/* Create a temporary variable in which to hold the array length. */
	cast_type len_ctype;
	cast_expr len_var_expr;
	if (adef) {
		len_ctype = mint_to_ctype(&pres->mint, adef->length_type);
		/* If it's a string, treat it specially. */
		if (unqualified_elem_ctype->kind == CAST_TYPE_PRIMITIVE) {
			if (unqualified_elem_ctype->cast_type_u_u.primitive_type.kind
			    == CAST_PRIM_CHAR) {
				mu_mapping_string(expr, ptr_ctype, adef,
						  &(tmap->alloc));
				return;
			}
		}
		
		len_var_expr = add_temp_var("array_len", len_ctype);
		
		cast_stmt null_stmt = cast_new_stmt(CAST_STMT_NULL);
		
		/*
		 * If we're encoding, initialize the array length variable by counting
		 * the number of array elements before the terminator.
		 */
		if (op & MUST_ENCODE) {
			add_stmt(cast_new_for(
				cast_new_expr_assign(len_var_expr, 
						     cast_new_expr_lit_int(0, 0)),
				cast_new_binary_expr(
					CAST_BINARY_NE,
					cast_new_unary_expr(
						CAST_UNARY_DEREF,
						cast_new_binary_expr(CAST_BINARY_ADD,
								     expr,
								     len_var_expr)),
					cast_new_expr_lit_int(tmap->terminator, 0)),
				cast_new_unary_expr(CAST_UNARY_POST_INC, len_var_expr),
				null_stmt /* Do nothing. */ ));
		}
		
		/* Marshal/unmarshal the array length from/to the length variable. */
		mu_mapping_direct(len_var_expr, len_ctype, adef->length_type);
		itype = adef->element_type;
	} else {
		len_ctype = 0;
		len_var_expr = cast_new_expr_lit_int(len_max, 0);
	}

	/* Marshal/unmarshal the array itself. */
	mu_array(expr, ctype, &(tmap->alloc),
		 elem_ctype, itype, tmap->element_mapping,
		 len_var_expr, len_ctype, len_min, len_max);
	
	/*
	 * If we're decoding, produce code to add the terminator to the array.
	 */
	if (adef && (op & MUST_DECODE))
		add_stmt(cast_new_stmt_expr(
			cast_new_expr_assign(
				cast_new_unary_expr(
					CAST_UNARY_DEREF,
					cast_new_binary_expr(CAST_BINARY_ADD,
							     expr,
							     len_var_expr)),
				cast_new_expr_lit_int(tmap->terminator, 0))));
}

/* End of file. */

