/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>

/*
 * This is a utility function used to build logical-OR expressions
 * intelligently.
 *
 * To use it, initialize a `cast_expr' variable (e.g., `expr') to 0.  Each time
 * you want to add a new term to the expression, call this routine with the old
 * `expr' as `old_cond' and the new term as `new_cond', and assign the result
 * back to `expr'.  This will automatically handle the case of there being only
 * one term.  (Of course, if there are _no_ terms, then `expr' will remain null
 * --- so be careful.)
 */
cast_expr if_or(cast_expr old_cond, cast_expr new_cond)
{
	if (old_cond == 0)
		return new_cond;
	else
		return cast_new_binary_expr(CAST_BINARY_LOR,
					    old_cond, new_cond);
}

/*
 * Generate code to allocate memory to be pointed to by a pointer variable or
 * parameter, as dictated by the allocation semantics defined in the passed
 * `alloc' structure.
 *
 * `pexpr' is a C expression representing the pointer variable memory is being
 * allocated for; naturally, it must be an lvalue so that it can be assigned
 * to.
 *
 * `target_type' is the C type of whatever the pointer points to.  (The
 * pointer's type itself would be a CAST_TYPE_POINTER referring to that type.)
 *
 * If the pointer is actually going to point to the first element of an array,
 * then `length_expr' must be an expression that evaluates to the number of
 * array elements that memory should be allocated for.  If the pointer only
 * points to one element (a simple pointer indirection), then `length_expr'
 * should be a CAST constant `1'.
 */
void mu_state::mu_pointer_alloc(cast_expr pexpr,
				cast_type target_type,
				cast_expr length_expr,
				pres_c_allocation *alloc)
{
	cast_expr alloc_expr;
	
	if (!(op & MUST_ALLOCATE))
		return;
	
	if ((alloc->flags & PRES_C_ALLOC_EVER) == PRES_C_ALLOC_NEVER)
		return;
	
	/*
	 * Special case: If we would try to allocate a `void', we change the
	 * target type to something allocable --- `char' will do.
	 */
	if (target_type->kind == CAST_TYPE_VOID) {
		target_type = cast_new_type(CAST_TYPE_PRIMITIVE);
		target_type->
			cast_type_u_u.primitive_type.kind = CAST_PRIM_CHAR;
		target_type->
			cast_type_u_u.primitive_type.mod = 0;
	}
	
	assert(length_expr != 0);
	
	/*
	 * Generate a statement to allocate storage and assign the pointer.
	 *
	 * XXX --- We ought to replace this `if' with some kind of allocator
	 * method dispatch function.
	 */
	if (!strcmp(alloc->allocator, "static")) {
		
		cast_type static_target_type;
		
		/*
		 * The static allocator can allocate only positive, constant
		 * amounts of storage!
		 */
		assert(length_expr->kind
		       == CAST_EXPR_LIT_PRIM);
		assert(length_expr->cast_expr_u_u.lit_prim.u.kind
		       == CAST_PRIM_INT);
		assert(length_expr->cast_expr_u_u.lit_prim.u.cast_lit_prim_u_u.
		       i
		       > 0);
		
		/*
		 * Declare a static variable to provide the storage and assign
		 * the pointer to point to that.
		 *
		 * XXX --- It's a little goofy to have two variables, one for
		 * static storage and one to point to it.  But this is how
		 * things must be done until the back end is smarter.  To be
		 * smarter, we need to be able to determine `pexpr' here and
		 * then pass it up.  But implementing that feature correctly
		 * would seem to mean that *all* mapping functions should be
		 * able to determine their own CAST reference expressions.
		 */
		if (length_expr->cast_expr_u_u.lit_prim.u.cast_lit_prim_u_u.i
		    != 1) {
			/* `static_target_type' will be an array. */
			static_target_type = cast_new_type(CAST_TYPE_ARRAY);
			static_target_type->
				cast_type_u_u.array_type.element_type =
					target_type;
			static_target_type->
				cast_type_u_u.array_type.length =
				length_expr;
		} else
			/* `static_target_type' will be the `target_type'. */
			static_target_type = target_type;
		
		cast_expr static_var_expr = add_temp_var("storage",
							 static_target_type,
							 CAST_SC_STATIC);
		
		if (static_target_type != target_type)
			/* `alloc_expr' is the name of the local array. */
			alloc_expr = static_var_expr;
		else
			/* `alloc_expr' is the address of the local var. */
			alloc_expr = cast_new_unary_expr(CAST_UNARY_ADDR,
							 static_var_expr);
		
	} else if ((!strcmp(alloc->allocator, "auto")
		    || !strcmp(alloc->allocator, "out_of_line")
		    /*
		     * XXX --- The `auto_alloc' hack, which forces ``deep''
		     * auto allocation for `in' parameters.  The following
		     * condition is the test that `get_allocator' uses to force
		     * auto allocation.
		     */
		    || ((alloc->flags & PRES_C_ALLOC_ALWAYS)
			&& !strcmp(get_which_stub(), "server")
			&& (current_param_dir == PRES_C_DIRECTION_IN)
			&& ((alloc->flags & PRES_C_DEALLOC_EVER)
			    == PRES_C_DEALLOC_ALWAYS)))
		   && (length_expr->kind == CAST_EXPR_LIT_PRIM)) {
		
		cast_type auto_target_type;
		
		/*
		 * The auto allocator can deal with requests for variable
		 * amounts of storage, but such requests are not handled here.
		 * Here, we must have a positive constant request.
		 */
		assert(length_expr->cast_expr_u_u.lit_prim.u.kind
		       == CAST_PRIM_INT);
		assert(length_expr->cast_expr_u_u.lit_prim.u.cast_lit_prim_u_u.
		       i
		       > 0);
		
		/*
		 * Declare an automatic variable to provide the storage and
		 * assign the pointer to point to that.
		 *
		 * XXX --- See the comment in the ``static'' case about why we
		 * must currently have two variables (one for the pointer and a
		 * second for storage) instead of just one.
		 */
		if (length_expr->cast_expr_u_u.lit_prim.u.cast_lit_prim_u_u.i
		    != 1) {
			/* `auto_target_type' will be an array. */
			auto_target_type = cast_new_type(CAST_TYPE_ARRAY);
			auto_target_type->
				cast_type_u_u.array_type.element_type =
					target_type;
			auto_target_type->
				cast_type_u_u.array_type.length =
				length_expr;
		} else
			/* `auto_target_type' will be `target_type'. */
			auto_target_type = target_type;
		
		cast_expr auto_var_expr = add_temp_var("storage",
						       auto_target_type,
						       CAST_SC_AUTO);
		
		if (auto_target_type != target_type)
			/* `alloc_expr' is the name of the local array. */
			alloc_expr = auto_var_expr;
		else
			/* `alloc_expr' is the address of the local var. */
			alloc_expr = cast_new_unary_expr(CAST_UNARY_ADDR,
							 auto_var_expr);
		
	} else {
		/* Otherwise, call a function to allocate the storage. */
		char *name = get_allocator(alloc);
		
		alloc_expr = cast_new_expr_call(cast_new_expr_name(name), 1);
		
		if ((length_expr->kind == CAST_EXPR_LIT_PRIM)
		    && (length_expr->cast_expr_u_u.lit_prim.u.kind ==
			CAST_PRIM_INT)
		    && (length_expr->cast_expr_u_u.lit_prim.u.
			cast_lit_prim_u_u.i
			== 1)
			)
			/* Special case: Don't multiply the `sizeof' by 1. */
			alloc_expr->cast_expr_u_u.call.params.params_val[0] =
				cast_new_expr_sizeof_type(target_type);
		else
			/* General case: Multiply by `length_expr'. */
			alloc_expr->cast_expr_u_u.call.params.params_val[0] =
				cast_new_binary_expr(CAST_BINARY_MUL,
						     length_expr,
						     cast_new_expr_sizeof_type(
							     target_type));
	}
	
	cast_stmt alloc_stmt =
		cast_new_stmt_expr(cast_new_expr_assign(pexpr, alloc_expr));
	
	/* Now decide when that statement should be invoked. */
	if (alloc->flags & PRES_C_ALLOC_ALWAYS) {
		add_stmt(alloc_stmt);
	} else {
		cast_expr if_test = 0;
		
		if (alloc->flags & PRES_C_ALLOC_IF_NULL)
			if_test = if_or(if_test,
					cast_new_unary_expr(CAST_UNARY_LNOT,
							    pexpr));
		if (alloc->flags & PRES_C_ALLOC_IF_TOO_SMALL)
			assert(0); /* XXX */
		if (alloc->flags & PRES_C_ALLOC_IF_TOO_LARGE)
			assert(0); /* XXX */
		
		assert(if_test != 0);
		add_stmt(cast_new_if(if_test, alloc_stmt, 0));
	}
}

/*
 * This helper function simply returns the name of the allocator function for
 * a given `pres_c_allocation'.
 */
char *mu_state::get_allocator(pres_c_allocation *alloc) 
{
	char *name;
	
	if (alloc->flags & PRES_C_ALLOC_ALWAYS) {
		if (!strcmp(get_which_stub(), "server")
		    /*
		     * ...and this object is `in'.  (XXX --- Bogus check.  What
		     * we really want to know is that the receiver won't try to
		     * modify or reallocate the object.)
		     */
		    && (current_param_dir == PRES_C_DIRECTION_IN)
		    /*
		     * ...and we are responsible for later deallocating the
		     * object, too.  (Consider a special presentation in which
		     * an `in' parameter must persist after the server work
		     * function has exited.)
		     */
		    && ((alloc->flags & PRES_C_DEALLOC_EVER)
			== PRES_C_DEALLOC_ALWAYS)
			)
			/*
			 * Force `auto' allocation for `in' parameters within
			 * the server dispatch function; i.e., the `auto_alloc'
			 * hack.
			 *
			 * XXX --- This is a KLUDGE!  For allocating `in'
			 * parameters, we *should* be able to do the right
			 * thing automatically because the PRES_C for the
			 * request *should* say when to use `auto' allocation.
			 * But the PG doesn't get the allocation flags right!
			 *
			 * Fundamentally, the PG needs to start determining
			 * allocation flags based on data *roles*, not data
			 * *types*!
			 */
			name = "auto_flick_alloc";
		else
			name = flick_asprintf("%s_flick_alloc",
					      alloc->allocator);
		return name;
	} else
		return "null_flick_alloc";
}

/* End of file. */

