/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>

/*
 * Generate code to free the memory pointed to by a pointer variable or
 * parameter, as dictated by the deallocation semantics defined in the passed
 * `alloc' structure.
 *
 * `pexpr' is a C expression evaluating to the pointer whose memory is being
 * freed.
 *
 * `target_type' is the C type of whatever the pointer points to.  (The
 * pointer's type itself would be a CAST_TYPE_POINTER referring to that type).
 */
void mu_state::mu_pointer_free(cast_expr pexpr,
			       cast_type /*target_type*/,
			       cast_expr /*lexpr*/,
			       pres_c_allocation *alloc)
{
	cast_expr free_expr;

	if (!(op & MUST_DEALLOCATE)) /* XXX */
		return;
	
	/*
	 * Generate a statement to deallocate the storage pointed to by the
	 * pointer.
	 *
	 * XXX --- We ought to replace this `if' with some kind of allocator
	 * method dispatch function.
	 */
	if (!strcmp(alloc->allocator, "static"))
		/* Nothing ever needs to be done. */
		return;
	else if ((!strcmp(alloc->allocator, "auto"))
		   /*
		    * XXX --- The `auto_alloc' hack, which forces ``deep''
		    * auto allocation for `in' parameters.  The following
		    * condition is the test that `get_deallocator' uses to
		    * force auto allocation.
		    */
		   || ((alloc->flags & PRES_C_DEALLOC_ALWAYS)
		       && !strcmp(get_which_stub(), "server")
		       && (current_param_dir == PRES_C_DIRECTION_IN)
		       && ((alloc->flags & PRES_C_ALLOC_EVER)
			   == PRES_C_ALLOC_ALWAYS))
		)
		/* Nothing ever needs to be done. */
		return;
	else {
		/* Otherwise, call a function to allocate the storage. */
		char *name = get_deallocator(alloc);
		
		free_expr = cast_new_expr_call(cast_new_expr_name(name), 1);
		
		free_expr->cast_expr_u_u.call.params.params_val[0] = pexpr;
	}
	
	cast_stmt free_stmt = cast_new_stmt_expr(free_expr);
	
	/* Now decide when that statement should be invoked. */
	if (alloc->flags & PRES_C_DEALLOC_ALWAYS) {
		add_stmt(free_stmt);
	}
#if 0
	else
		/* XXX assert(0); */
		warn("In `mu_pointer_free', taking branch that used to be "
		     "`assert(0)'.");
#endif
	
	/* Nullify the pointer if requested.  */
	if (alloc->flags & PRES_C_DEALLOC_NULLIFY)
		add_stmt(cast_new_stmt_expr(
			cast_new_expr_assign(pexpr, 
					     cast_new_expr_lit_int(0, 0))
			));
}

/*
 * This helper function simply returns the name of the deallocator function for
 * a given `pres_c_allocation'.
 */
char *mu_state::get_deallocator(pres_c_allocation *alloc) 
{
	if (alloc->flags & PRES_C_DEALLOC_ALWAYS) {
		if (!strcmp(get_which_stub(), "server")
		    /*
		     * ...and this object is `in'.  (XXX --- Bogus check.  What
		     * we really want to know is what `mu_state::get_allocator'
		     * decided to use.)
		     */
		    && (current_param_dir == PRES_C_DIRECTION_IN)
		    /*
		     * ...and we were responsible for previously allocating the
		     * object, too.
		     */
		    && ((alloc->flags & PRES_C_ALLOC_EVER)
			== PRES_C_ALLOC_ALWAYS)
			)
			/*
			 * Force `auto' deallocation for `in' parameters within
			 * the server dispatch function; i.e., the `auto_alloc'
			 * hack.
			 *
			 * XXX --- This is a KLUDGE!  For allocating `in'
			 * parameters, we *should* be able to do the right
			 * thing automatically because the PRES_C for the
			 * request *should* say when to use `auto' allocation.
			 * But the PG doesn't get the allocation flags right!
			 *
			 * Fundamentally, the PG needs to start determining
			 * allocation flags based on data *roles*, not data
			 * *types*!
			 */
			return "auto_flick_free";
		else
			return flick_asprintf("%s_flick_free",
					      alloc->allocator);
	} else
		return "null_flick_free";
}

/* End of file. */

