/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/compiler.h>
#include <mom/c/libcast.h>

#include "mach3.h"

/* MAX_STATIC_MESSAGE_SIZE defines the largest message size that is
 * allowed on the stack.  If a message could potentially be larger (in
 * particular, unbounded), then a dynamically-allocated message buffer
 * is used (supplied by the runtime library).
 *
 * XXX - Note that this does not specify the maximum size of a message
 * received by the server.  Whatever creates the receive buffer for
 * the server designates the maximum message allowed to be passed by
 * the system.
 */
#define MAX_STATIC_MESSAGE_SIZE 65536

static void do_stub(pres_c_1 *pres, cast_ref c_func,
		    mint_ref request_itype, mint_ref reply_itype,
		    pres_c_inline request_i, pres_c_inline reply_i,
		    mint_ref target_itype, mint_ref client_itype,
		    pres_c_inline target_i, pres_c_inline client_i,
		    pres_c_stub_op_flags op_flags)
{
	assert(pres);
	
	cast_def *cfunc = &pres->cast.cast_scope_val[c_func];
	cast_func_type *cfunct = &cfunc->u.cast_def_u_u.func_type;

	mint_ref            operation_itype;
	pres_c_inline       operation_inline;
	
	int assumptions = RPCM_TRUSTED; /* XXX */
	char *emergency_return_string;
	
	/*****/
	
	if (cfunct->return_type->kind != CAST_TYPE_VOID)
		emergency_return_string = "return _return";
	else
		emergency_return_string = "return";
	emergency_return_value = cast_new_expr_name(emergency_return_string);
	
        /* Build the parameter marshal/deallocation code. */
	mach3_mu_state must_in(pres,
			       (MUST_ENCODE | MUST_DEALLOCATE), assumptions,
			       "client");
	/* set the initial maximum message size (for unseen header) */
	must_in.max_msg_size = 24;
	
	remove_idl_and_interface_ids(pres,
				     request_itype, request_i,
				     &operation_itype, &operation_inline);
	must_in.set_id_expected(operation_itype);
	
	must_in.mu_func_params(c_func, operation_itype, operation_inline);
	must_in.mu_end();
	
	/* Build the unmarshal code, but only if this is a two-way stub. */
	mach3_mu_state must_out(pres,
				(MUST_DECODE | MUST_ALLOCATE), assumptions,
				"client");
	/* set the initial maximum message size (for unseen header) */
	must_out.max_msg_size = 32;
	if (reply_i) {
		remove_idl_and_interface_ids(pres,
					     reply_itype, reply_i,
					     &operation_itype,
					     &operation_inline);
		/*
		 * Do not call `remove_operation_id' to remove the operation
		 * identifier.  MIG transmits operation reply codes, so we do,
		 * too.
		 */
		must_out.set_id_expected(operation_itype);
		must_out.mu_func_params(c_func,
					operation_itype, operation_inline);
		must_out.mu_end();
	} else {
		assert(must_out.max_msg_size == 32);
	}
	
	/* Build the target object marshaling code. */
	mach3_target_mu_state must_target(pres,
					  MUST_ENCODE, 0,
					  "client");
	must_target.mu_func_params(c_func, target_itype, target_i);
	must_target.mu_end();
	
	/* Build the client object marshaling code. */
	mach3_client_mu_state must_client(pres,
					  MUST_ENCODE, 0,
					  "client");
	if (client_i) {
		must_client.mu_func_params(c_func, client_itype, client_i);
		must_client.mu_end();
	} else {
		must_client.c_block = cast_new_stmt_expr(cast_new_expr_call_4(
			cast_new_expr_name(
				flick_asprintf("flick_%s_%s_%s_client",
					       must_client.get_be_name(),
					       must_client.get_which_stub(),
					       must_client.get_buf_name())),
			cast_new_expr_name(
				(op_flags & PRES_C_STUB_OP_FLAG_ONEWAY) ?
				"MACH_PORT_NULL" :
				"mig_get_reply_port()"),
			cast_new_expr_lit_int(0, 0),
			cast_new_expr_lit_int(0, 0),
			cast_new_expr_name(must_client.get_encode_name())));
	}
	
	/* Start writing the client stub.  */
	cast_w_func_type(cfunc->name, cfunct, 0);
	w_printf("\n{\n");
	if (must_in.max_msg_size < MAX_STATIC_MESSAGE_SIZE &&
	    must_out.max_msg_size < MAX_STATIC_MESSAGE_SIZE) {
		w_i_printf(1, "char _global_buf_start[%d];\n",
			   ((must_in.max_msg_size > must_out.max_msg_size) ?
			    must_in.max_msg_size:
			    must_out.max_msg_size));
		w_i_printf(1, "void *_buf_end = (void *)&(_global_buf_start[%d]);\n",
			   ((must_in.max_msg_size > must_out.max_msg_size) ?
			    must_in.max_msg_size:
			    must_out.max_msg_size));
	} else {
		if (must_in.msg_option_expr) {
			must_in.msg_option_expr =
				cast_new_binary_expr(
					CAST_BINARY_BOR,
					must_in.msg_option_expr,
					cast_new_expr_name("MACH_RCV_LARGE"));
		} else {
			must_in.msg_option_expr =
				cast_new_expr_name("MACH_RCV_LARGE");
		}
	}
	w_i_printf(1, "register mig_reply_header_t *_buf_start;\n");
	w_i_printf(1, "register void *_buf_current;\n");
	w_i_printf(1, "kern_return_t _ipc_return = KERN_SUCCESS;\n");

	switch (cfunct->return_type->kind) {
	case CAST_TYPE_VOID:
		w_printf("\n");
		break;
		
	case CAST_TYPE_NAME:
		if (!strcmp(cfunct->return_type->cast_type_u_u.name,
			    "kern_return_t")) {
			w_indent(1);
			cast_w_type("_return", cfunct->return_type, 1);
			w_printf(" = 0;\n\n");
			break;
		}
		
	default:
		w_indent(1);
		cast_w_type("_return", cfunct->return_type, 1);
		w_printf(";\n\n");
		cast_w_stmt(
			cast_new_stmt_expr(
				cast_new_expr_assign_to_zero(
					cast_new_expr_name("_return"),
					cfunct->return_type,
					&(pres->cast))),
			1);
	}
	
	w_i_printf(1,
		   "flick_%s_client_start_encode();\n",
		   must_in.get_be_name());
	
	cast_w_stmt(must_client.c_block, 1);
	cast_w_stmt(must_target.c_block, 1);
	
	/* Output the general marshaling code. */
	cast_w_stmt(must_in.c_block, 1);
	w_i_printf(1,
		   "flick_%s_client_end_encode();\n",
		   must_in.get_be_name());
	
	if (must_in.is_complex == (void *) 1) {
		cast_w_stmt(cast_new_stmt_expr(cast_new_expr_op_assign(
			CAST_BINARY_BOR,
			cast_new_expr_name("_buf_start->Head.msgh_bits"),
			cast_new_expr_name("MACH_MSGH_BITS_COMPLEX"))),
			1);
	}
	
	/* Invoke the appropriate runtime function to perform the RPC. */
	cast_expr rpc;
	rpc = cast_new_expr_call(cast_new_expr_name(
		flick_asprintf(
			"flick_%s_%s",
			must_in.get_be_name(),
			((op_flags & PRES_C_STUB_OP_FLAG_ONEWAY) ?
			 "send" :
			 ((must_in.max_msg_size < MAX_STATIC_MESSAGE_SIZE &&
			   must_out.max_msg_size < MAX_STATIC_MESSAGE_SIZE)?
			  "rpc_static" : "rpc_dynamic")))),
				 2);
	rpc->cast_expr_u_u.call.params.params_val[0] =
		must_in.msg_option_expr ?
		must_in.msg_option_expr :
		cast_new_expr_name("MACH_MSG_OPTION_NONE");
	rpc->cast_expr_u_u.call.params.params_val[1] =
		must_in.timeout_expr ?
		must_in.timeout_expr :
		cast_new_expr_name("MACH_MSG_TIMEOUT_NONE");
	
	cast_w_stmt(
		cast_new_stmt_expr(
			cast_new_expr_assign(
				cast_new_expr_name("_ipc_return"),
				rpc)),
		1);
	
	/* Check for errors from the IPC operation. */
	w_i_printf(1,
		   "if (_ipc_return != MACH_MSG_SUCCESS)\n");
	w_i_printf(2,
		   "flick_%s_decode_client_error"
		   "(%s, FLICK_ERROR_COMMUNICATION, %s, %s);\n",
		   must_in.pres->pres_context,
		   ((strcmp(must_in.pres->pres_context, "mig")) ?
		    emergency_return_string :
		    ((cfunct->return_type->kind != CAST_TYPE_VOID) ?
		     "return _ipc_return" : "return")),
		   must_in.get_encode_name(),
		   must_in.get_be_name());
	
	/* Output the reply unmarshaling code. */
	if (reply_i) {
		if (op_flags & PRES_C_STUB_OP_FLAG_ONEWAY) {
			/* we may still need to dealloc space */
			cast_w_stmt(must_out.c_block, 1);
		} else {
			w_i_printf(1,
				   "flick_%s_client_start_decode();\n",
				   must_in.get_be_name());
			cast_w_stmt(must_out.c_block, 1);
			w_i_printf(1,
				   "flick_%s_client_end_decode();\n",
				   must_in.get_be_name());
		}
	}
	
	if (cfunct->return_type->kind != CAST_TYPE_VOID)
		w_i_printf(1, "return _return;\n");
	w_printf("}\n\n");
}

/* Start the client stub */
void w_client_stub(pres_c_1 *pres, int stub_idx)
{
	assert(pres);

	pres_c_stub *stub = &pres->stubs.stubs_val[stub_idx];
	assert(stub->kind == PRES_C_CLIENT_STUB);
	pres_c_client_stub *cstub = &stub->pres_c_stub_u.cstub;

	assert(cstub->request_i);
	assert(cstub->reply_i);

	do_stub(pres, cstub->c_func,
		cstub->request_itype, cstub->reply_itype,
		cstub->request_i, cstub->reply_i,
		cstub->target_itype, cstub->client_itype,
		cstub->target_i, cstub->client_i,
		cstub->op_flags);
}

void w_send_stub(pres_c_1 *pres, int stub_idx)
{
	assert(pres);

	pres_c_stub *stub = &pres->stubs.stubs_val[stub_idx];
	assert(stub->kind == PRES_C_SEND_STUB);
	pres_c_send_stub *sstub = &stub->pres_c_stub_u.send_stub;

	do_stub(pres, sstub->c_func,
		sstub->msg_itype, mint_ref_null,
		sstub->msg_i, 0,
		sstub->target_itype, mint_ref_null,
		sstub->target_i, 0,
		PRES_C_STUB_OP_FLAG_ONEWAY /* XXX --- Right? */);
}

/* End of file. */

