/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/idl_id.h>
#include <mom/mint_special.h>
#include <mom/compiler.h>
#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include "mach3.h"

struct ool_check_functor : public functor
{
	virtual void func(mu_state *must);
	ool_check_functor() {};

	ool_check_functor(cast_expr cond_expr,
			  int e_size,
			  cast_expr arr_expr,
			  cast_type arr_ctype,
			  pres_c_allocation *arr_alloc,
			  cast_type el_ctype,
			  mint_ref el_itype,
			  pres_c_mapping el_map,
			  cast_expr ln_expr,
			  cast_type ln_ctype,
			  unsigned long ln_min,
			  unsigned long ln_max) :
		condition_expr(cond_expr),
		elem_size(e_size),
		array_expr(arr_expr),
		array_ctype(arr_ctype),
		array_alloc(arr_alloc),
		elem_ctype(el_ctype),
		elem_itype(el_itype),
		elem_map(el_map),
		len_expr(ln_expr),
		len_ctype(ln_ctype),
		len_min(ln_min),
		len_max(ln_max)
		{};
	
	cast_expr condition_expr;
	int elem_size;
	cast_expr array_expr;
	cast_type array_ctype;
	pres_c_allocation *array_alloc;
	cast_type elem_ctype;
	mint_ref elem_itype;
	pres_c_mapping elem_map;
	cast_expr len_expr;
	cast_type len_ctype;
	unsigned long len_min;
	unsigned long len_max;
};

struct ool_case_functor : public functor
{
	virtual void func(mu_state *must);
	ool_case_functor(cast_expr arr_expr,
			 cast_type arr_ctype,
			 pres_c_allocation *arr_alloc,
			 cast_type el_ctype,
			 mint_ref el_itype,
			 pres_c_mapping el_map,
			 cast_expr ln_expr,
			 cast_type ln_ctype,
			 unsigned long ln_min,
			 unsigned long ln_max) :
		array_expr(arr_expr),
		array_ctype(arr_ctype),
		array_alloc(arr_alloc),
		elem_ctype(el_ctype),
		elem_itype(el_itype),
		elem_map(el_map),
		len_expr(ln_expr),
		len_ctype(ln_ctype),
		len_min(ln_min),
		len_max(ln_max)
		{};
	
	cast_expr array_expr;
	cast_type array_ctype;
	pres_c_allocation *array_alloc;
	cast_type elem_ctype;
	mint_ref elem_itype;
	pres_c_mapping elem_map;
	cast_expr len_expr;
	cast_type len_ctype;
	unsigned long len_min;
	unsigned long len_max;
};

void ool_check_functor::func(mu_state *must)
{
	mu_memory_allocator_state *start_state;
	cast_stmt saved_block, then_block, else_block;

	must->break_chunk();
	start_state = must->memory_allocator_state();
	saved_block = must->c_block;
	
	// Generate the out-of-line branch
	must->c_block = cast_new_block(0, 0);
	pres_c_allocation aalloc = *array_alloc;
	aalloc.allocator = "out_of_line";
	ool_case_functor ooln(array_expr, array_ctype, &aalloc,
				elem_ctype, elem_itype, elem_map,
				len_expr, len_ctype,
				len_min, len_max);
	must->mu_union_case(&ooln);
	if (must->op & MUST_ENCODE) {
		must->add_stmt(
			cast_new_stmt_expr(cast_new_expr_op_assign(
				CAST_BINARY_BOR,
				cast_new_expr_name(
					"_buf_start->Head.msgh_bits"),
				cast_new_expr_name(
					"MACH_MSGH_BITS_COMPLEX"))));
	}
	then_block = must->c_block;
	
	// Generate the in-line branch
	must->set_memory_allocator_state(start_state);
	must->c_block = cast_new_block(0, 0);
	ool_case_functor inln(array_expr, array_ctype, array_alloc,
			      elem_ctype, elem_itype, elem_map,
			      len_expr, len_ctype,
			      len_min, len_max);
	must->mu_union_case(&inln);
	else_block = must->c_block;
	
	// Now make the if statement
	must->c_block = saved_block;
	must->add_stmt(cast_new_if(condition_expr,
				   then_block,
				   else_block));
}

void ool_case_functor::func(mu_state *must)
{
	must->mu_array(array_expr, array_ctype, array_alloc,
		       elem_ctype, elem_itype, elem_map,
		       len_expr, len_ctype,
		       len_min, len_max);
}


void mach3_mu_state::mu_aggregated_array(
	cast_expr array_expr, cast_type array_ctype,
	pres_c_allocation *array_alloc,
	cast_type elem_ctype, mint_ref elem_itype,
	pres_c_mapping elem_map,
	cast_expr len_expr, cast_type len_ctype,
	unsigned long len_min, unsigned long len_max,
	mint_ref prim_itype, int fixed_elts,
	int out_of_line)
{
	mach3_array = 1;

	/* Decide whether to use a shortform or longform type descriptor.  */
	int longform = (len_max >= 4096);

	/* Find the vital parameters of the primitive array elements.  */
	int prim_size = -1, prim_align_bits = -1;
	char *prim_macro_name = 0;
	if (prim_itype != pres->mint.standard_refs.void_ref)
		get_prim_params(prim_itype, &prim_size, &prim_align_bits,
				&prim_macro_name);

	cast_expr oolcheck = 0;
	if (len_max == 0xffffffff /* unbounded array */
	    && !out_of_line) {
		/* send inline if <= 2K, OOL if > 2K */
		oolcheck = add_temp_var("ool_check",
					cast_new_prim_type(CAST_PRIM_INT, 0));
	}

	cast_expr limit2m;
	if (prim_size > 0) {
		limit2m = cast_new_expr_lit_int(
			2048 / (prim_size * fixed_elts),
			0);
	} else {
		limit2m =
			cast_new_binary_expr(CAST_BINARY_DIV,
					     cast_new_expr_lit_int(2048, 0),
					     cast_new_expr_sizeof_type(
						     elem_ctype));
	}
	
	if (oolcheck && (op & MUST_ENCODE)) {
		add_stmt(
			cast_new_stmt_expr(
				cast_new_expr_assign(
					oolcheck,
					cast_new_binary_expr(
						CAST_BINARY_GT,
						len_expr,
						limit2m))));
	}
	
	/* XXX - this is an incomplete hack to distinguish between an
	   array of characters versus a fixed-lenght C-style string.
	   See comment in mu_mapping_fixed_array for more info. */
	   
	/* Output the proper tag for fixed-length strings */
	if (is_string ||
	    out_of_line && elem_map->kind == PRES_C_MAPPING_TERMINATED_ARRAY) {
		prim_macro_name = flick_asprintf("flick_%s_%s_string_c",
						 get_be_name(),
						 get_buf_name());
		is_string = 0;
	}
	
	if (prim_itype != pres->mint.standard_refs.void_ref) {
		/* Produce a macro invocation to marshal the type descriptor.  */
		char *macro_name = flick_asprintf("%s_type", prim_macro_name);
		cast_expr macro_expr = cast_new_expr_name(macro_name);
		cast_expr ofs_expr = cast_new_expr_lit_int(
			chunk_prim(2, longform ? 12 : 4), 0);
		cast_expr num_expr;
		if (fixed_elts > 1) {
			num_expr = cast_new_binary_expr(
				CAST_BINARY_MUL,
				len_expr,
				cast_new_expr_lit_int(fixed_elts, 0));
		} else 
			num_expr = len_expr;
		if (op & MUST_DECODE) {
			if (len_expr->kind == CAST_EXPR_LIT_PRIM) {
				/* if we are decoding an array, the length MUST be
				   in a variable in order to decode the type! */
				cast_expr lexpr =
					add_temp_var("array_length",
						     cast_new_prim_type(
							     CAST_PRIM_INT,
							     CAST_MOD_UNSIGNED));
				add_stmt(cast_new_stmt_expr(
					cast_new_expr_assign(
						lexpr, num_expr)));
				num_expr = lexpr;
			} else if (num_expr != len_expr) {
				add_stmt(cast_new_stmt_expr(
					cast_new_expr_assign(
						len_expr, num_expr)));
				num_expr = len_expr;
			}
		}
		cast_expr inl_expr;
		if (oolcheck)
			inl_expr = cast_new_unary_expr(CAST_UNARY_LNOT, oolcheck);
		else
			inl_expr = cast_new_expr_lit_int(!out_of_line, 0);
		cast_expr dealloc_expr =
			cast_new_expr_lit_int(
				(out_of_line &&
				 ((array_alloc->flags & PRES_C_DEALLOC_EVER) ==
				  PRES_C_DEALLOC_ALWAYS)), 0);
		cast_expr long_expr = cast_new_expr_name(
			longform ? "_long_type" : "_type");
	
		cast_expr cex;
		/* For ports, we need to add an extra param to the macro call --
		   the port type */
		if (pres->mint.defs.defs_val[prim_itype].kind == MINT_INTERFACE) {
			pres_c_mapping temp_map;
			
			/* XXX -- Dig through any stub that may be in the way */
			if (elem_map->kind == PRES_C_MAPPING_STUB) {
				int stub_num = pres_c_find_mu_stub(
					pres, elem_itype, elem_ctype,
					elem_map, PRES_C_MARSHAL_STUB);
				temp_map = pres->stubs.stubs_val[stub_num].
					   pres_c_stub_u.mstub.seethru_map;
			} else
				temp_map = elem_map;
			
			assert(temp_map->kind == PRES_C_MAPPING_REFERENCE);
			cex = cast_new_expr_call(macro_expr, 6);
			cex->cast_expr_u_u.call.params.params_val[0] = ofs_expr;
			cex->cast_expr_u_u.call.params.params_val[1] =
				cast_new_expr_name(
					get_mach_port_type(
						&temp_map->pres_c_mapping_u_u.ref,
						prim_itype));;
			cex->cast_expr_u_u.call.params.params_val[2] = num_expr;
			cex->cast_expr_u_u.call.params.params_val[3] = inl_expr;
			cex->cast_expr_u_u.call.params.params_val[4] = dealloc_expr;
			cex->cast_expr_u_u.call.params.params_val[5] = long_expr;
			is_complex = (cast_expr) 1;
		} else {
			cex = cast_new_expr_call(macro_expr, 5);
			cex->cast_expr_u_u.call.params.params_val[0] = ofs_expr;
			cex->cast_expr_u_u.call.params.params_val[1] = num_expr;
			cex->cast_expr_u_u.call.params.params_val[2] = inl_expr;
			cex->cast_expr_u_u.call.params.params_val[3] = dealloc_expr;
			cex->cast_expr_u_u.call.params.params_val[4] = long_expr;
		}
	
		add_stmt(cast_new_stmt_expr(cex));
	} else {
		cast_expr num_expr;
		if (fixed_elts > 1) {
			num_expr = cast_new_binary_expr(
				CAST_BINARY_MUL,
				len_expr,
				cast_new_expr_lit_int(fixed_elts, 0));
		} else 
			num_expr = len_expr;
	}
	
	/* Now descend into the array and marshal the individual elements. */
	
	if (oolcheck && (op & MUST_DECODE)) {
		add_stmt(
			cast_new_stmt_expr(
				cast_new_expr_assign(
					oolcheck,
					cast_new_binary_expr(
						CAST_BINARY_GT,
						len_expr,
						limit2m))));
	}
	
	/* Add a check to see if this should go inline or out-of-line, if
	   necessary. */
	if (oolcheck
	    && prim_itype != pres->mint.standard_refs.void_ref) {
		ool_check_functor  ocf(oolcheck, prim_size * fixed_elts,
				       array_expr, array_ctype, array_alloc,
				       elem_ctype, elem_itype, elem_map,
				       len_expr, len_ctype, len_min, len_max);
		mu_union(&ocf);
		if (is_complex != (void *) 1) {
			is_complex = oolcheck;
		}
		last_ool_check = oolcheck;
	} else if (oolcheck
		   && prim_itype == pres->mint.standard_refs.void_ref) {
		mem_mu_state *sub = (mem_mu_state *)clone();
		
		sub->c_block = 0;
		/* set up a sub m/u state to marshal the array out-of-line */
		pres_c_allocation aalloc = *array_alloc;
		aalloc.allocator = "mach_vm";
		aalloc.flags = PRES_C_DEALLOC_ALWAYS;
		sub->mem_mu_state::mu_array(array_expr, array_ctype, &aalloc,
					    elem_ctype, elem_itype, elem_map,
					    len_expr, len_ctype,
					    0, 0);
		add_stmt(cast_new_if(oolcheck,
				     sub->c_block,
				     0));
		delete sub;
	} else {
		mem_mu_state::mu_array(array_expr, array_ctype, array_alloc,
				       elem_ctype, elem_itype, elem_map,
				       len_expr, len_ctype, len_min, len_max);
	}
	
	mach3_array = 0;
	
	/* we need to align ourselves to a 4-byte boundary */
	if ((align_bits < 2) || (align_ofs & 3))
		chunk_prim(2, 0);
}

void mach3_mu_state::mu_array(
	cast_expr array_expr, cast_type array_ctype,
	pres_c_allocation *array_alloc,
	cast_type elem_ctype, mint_ref elem_itype,
	pres_c_mapping elem_map,
	cast_expr len_expr, cast_type len_ctype,
	unsigned long len_min, unsigned long len_max)
{
	/* Decide whether to transmit it in-line or out-of-line. */
	int out_of_line = 0;
	if (strcmp(array_alloc->allocator, "out_of_line") == 0) {
		out_of_line = 1;
		is_complex = (cast_expr) 1;
	}
	
	/* See if the whole array can be lumped under one MIG type descriptor.
	   This is possible iff this is an array of [fixed arrays of...]
	   primitive types.  At the same time, compute the total number of
	   primitive elements contained in each element of this (top-level,
	   possibly variable-length) array.  */
	mint_ref prim_itype = elem_itype;
	int fixed_elts = 1;

	if (out_of_line
	    || len_ctype
	    || (!len_ctype && elem_itype == pres->mint.standard_refs.void_ref)) while (1) {
		assert(prim_itype >= 0);
		assert(prim_itype < (signed int)pres->mint.defs.defs_len);
		
		mint_def *def = &pres->mint.defs.defs_val[prim_itype];

		/* Don't try to aggregate again if we're already aggregating. */
		if (mach3_array)
			break;
		
		if ((def->kind == MINT_INTEGER) ||
		    (def->kind == MINT_FLOAT) ||
		    (def->kind == MINT_CHAR) ||
		    (def->kind == MINT_VOID) ||
		    (def->kind == MINT_INTERFACE)) {
			/* We reached the primitive type at the bottom,
			   so aggregation is possible - do it.  */
			mu_aggregated_array(array_expr, array_ctype,
					    array_alloc,
					    elem_ctype, elem_itype, elem_map,
					    len_expr, len_ctype,
					    len_min, len_max,
					    prim_itype, fixed_elts,
					    out_of_line);
			return;
			
		} else if (def->kind == MINT_ARRAY) {
			unsigned array_min, array_max;
			mint_get_array_len(&pres->mint, prim_itype,
					   &array_min, &array_max);
			if (array_min != array_max) {
				/* Can't aggregate arrays of variable-length
                                   arrays.*/
				break;
			}
			fixed_elts *= array_min;
			/* simple overflow check */
			assert(fixed_elts % array_min == 0);
			prim_itype = def->mint_def_u.array_def.element_type;
		} else {
			/* Saw something else - no aggregation possible.  */
#if 0
			/*
			 * XXX - presentations other than MIG may cause this
			 * case to occur, so we can't panic here... But for
			 * MIG arrays, we have to know what type to send for
			 * the type tag.  If we get here, we don't know, and
			 * thus can't make a correct tag.  It would be wrong
			 * to handle it "the normal way", because no tag would
			 * be associated with this array, and the message
			 * would be (at least conceptually) wrong.
			 */
			if (len_max == 1)
				panic("Can't determine type of pointer data");
#endif
			break;
		}
	}

	/* No aggregation possible - deal with the array the normal way.  */
	mem_mu_state::mu_array(array_expr, array_ctype, array_alloc,
			       elem_ctype, elem_itype, elem_map,
			       len_expr, len_ctype, len_min, len_max);	
}

/* End of file. */
