/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <mom/c/libcast.h>
#include <mom/c/pbe.hh>

#include "mach3.h"

/*
 * Generate code to free the memory pointed to by a pointer variable or
 * parameter, as dictated by the deallocation semantics defined in the passed
 * `alloc' structure.
 *
 * `pexpr' is a C expression evaluating to the pointer whose memory is being
 * freed.
 *
 * `target_type' is the C type of whatever the pointer points to.  (The
 * pointer's type itself would be a CAST_TYPE_POINTER referring to that type).
 */
void mach3_mu_state::mu_pointer_free(cast_expr pexpr,
				     cast_type target_type,
				     cast_expr lexpr,
				     pres_c_allocation *alloc)
{
	cast_expr free_expr;

	if (!(op & MUST_DEALLOCATE)) /* XXX */
		return;

	/*
	 * For mach_vm frees, we have to pass the size as well
	 * For others, just run the default mu_pointer_free()
	 */
	if (!strcmp(alloc->allocator, "mach_vm")) {
		/* Call a function to allocate the storage. */
		char *name = get_deallocator(alloc);
		
		free_expr = cast_new_expr_call(cast_new_expr_name(name), 2);
		
		free_expr->cast_expr_u_u.call.params.params_val[0] = pexpr;

		if ((lexpr->kind == CAST_EXPR_LIT_PRIM)
		    && (lexpr->cast_expr_u_u.lit_prim.u.kind ==
			CAST_PRIM_INT)
		    && (lexpr->cast_expr_u_u.lit_prim.u.
			cast_lit_prim_u_u.i
			== 1)
			)
			/* Special case: Don't multiply the `sizeof' by 1. */
			free_expr->cast_expr_u_u.call.params.params_val[1] =
				cast_new_expr_sizeof_type(target_type);
		else
			/* General case: Multiply by `lexpr'. */
			free_expr->cast_expr_u_u.call.params.params_val[1] =
				cast_new_binary_expr(CAST_BINARY_MUL,
						     lexpr,
						     cast_new_expr_sizeof_type(
							     target_type));
		
		cast_stmt free_stmt = cast_new_stmt_expr(free_expr);
		
		/* Now decide when that statement should be invoked. */
		if (alloc->flags & PRES_C_DEALLOC_ALWAYS) {
			add_stmt(free_stmt);
		}
		/* Nullify the pointer if requested.  */
		if (alloc->flags & PRES_C_DEALLOC_NULLIFY)
			add_stmt(cast_new_stmt_expr(
				cast_new_expr_assign(
					pexpr,
					cast_new_expr_lit_int(0, 0))
				));
	} else if (!strcmp(alloc->allocator, "out_of_line")) {
		/*
		 * The deallocation has been taken care of by the
		 * deallocate flag in the IPC message.
		 */
	} else
		mu_state::mu_pointer_free(pexpr, target_type, lexpr, alloc);
}

/* End of file. */

