/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>
#include <mom/libmint.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include "trapeze.h"

trapeze_protocol_t what_protocol(pres_c_1 *pres)
{
	if (strcmp(pres->pres_context, "sun") == 0)
		return TRAPEZE_ONC;
	return TRAPEZE;
}

trapeze_mu_state::trapeze_mu_state(pres_c_1 *pres, mu_state_op op,
				   int assumptions, char *which,
				   int swap)
:	mem_mu_state(pres, op, assumptions, 2, 0, 120, which)
{
	should_swap = swap;
	has_payload = 0;
	protocol = what_protocol(pres);
}

trapeze_mu_state::trapeze_mu_state(const trapeze_mu_state &must)
:	mem_mu_state(must)
{
	should_swap = must.should_swap;
	has_payload = must.has_payload;
	protocol = must.protocol;
}

mu_state *trapeze_mu_state::another(mu_state_op op)
{
	return new trapeze_mu_state(pres, op, assumptions,
				    get_which_stub(), should_swap);
}

mu_state *trapeze_mu_state::clone()
{
	return new trapeze_mu_state(*this);
}

void trapeze_mu_state::get_prim_params(mint_ref itype, int *size,
				       int *align_bits, char **macro_name)
{
	mem_mu_state::get_prim_params(itype, size, align_bits, macro_name);
	
        mint_def *def = &(pres->mint.defs.defs_val[itype]);
	
	if ((def->kind == MINT_INTEGER)
            && (array_def != 0)) {
                /*
                 * We're processing an array length, and we always pad these
                 * out to four bytes.
                 */
                int bits, is_signed;
                
                assert(*size <= 4);
                *size = 4;
                *align_bits = 2;
                
                /* We must recompute the macro name, too. */
                mint_get_int_size(&pres->mint, itype, &bits, &is_signed);
                *macro_name = flick_asprintf("flick_%s_%s_%s%d",
                                             get_encode_name(),
                                             get_buf_name(),
                                             (is_signed ?
                                              "signed" : "unsigned"),
                                             ((*size) * 8));
        }
}

static void check_for_warning(unsigned len_max, cast_expr has_payload, cast_expr ptr)
{
	static int spitwarn1 = 0;
	static int spitwarn2 = 0;

	if (!spitwarn1
	    && len_max > TRAPEZE_MAX_PAYLOAD_SIZE) {
		warn("Maximum size of arrays cannot exceed %d bytes.",
		     TRAPEZE_MAX_PAYLOAD_SIZE);
		spitwarn1++;
	}
	
	if (!spitwarn2
	    && has_payload
	    && has_payload != ptr) {
		warn("Only one array can be sent in a message!");
		spitwarn2++;
	}
}

void trapeze_mu_state::mu_array(
	cast_expr array_expr,
	cast_type array_ctype,
	pres_c_allocation *array_alloc,
	cast_type elem_ctype,
	mint_ref elem_itype,
	pres_c_mapping elem_map,
	cast_expr len_expr, cast_type len_ctype,
	unsigned long len_min, unsigned long len_max)
{
	int elem_size, elem_align_bits;
	
	/* see if a bcopy is possible for this data */
	cast_expr bcopy_poss = mu_get_sizeof(elem_itype,
					     elem_ctype,
					     elem_map,
					     &elem_size,
					     &elem_align_bits);
	
	/* allocate a DMA buffer, if needed
	   (but not for the special environment var _ev) */
	if ((op & MUST_ALLOCATE) &&
	    (array_ctype->kind == CAST_TYPE_POINTER) &&
	    (elem_itype == pres->mint.standard_refs.void_ref) &&
	    array_expr->kind == CAST_EXPR_NAME &&
	    strcmp(array_expr->cast_expr_u_u.name, "_ev")) {
		add_stmt(cast_new_stmt_expr(cast_new_expr_assign(
			array_expr, cast_new_expr_name(
			flick_asprintf("flick_%s_%s_array__alloc()",
			       get_be_name(),
			       which_stub,
			       get_buf_name())))));
		return;
	}
	
	if (!len_ctype
	    || (bcopy_poss
		&& (len_max * elem_size + max_msg_size <
		    TRAPEZE_MAX_CONTROL_SIZE))) {
		/* do the normal thing and bail */
		mem_mu_state::mu_array(array_expr, array_ctype,
				       array_alloc, elem_ctype,
				       elem_itype, elem_map,
				       len_expr, len_ctype,
				       len_min, len_max);
		return;
	}
	
	/* calculate the length in bytes of the array */
	if (!bcopy_poss)
		bcopy_poss = cast_new_expr_sizeof_type(elem_ctype);

	cast_expr length_expr =
		cast_new_binary_expr(CAST_BINARY_MUL,
				     bcopy_poss,
				     len_expr);
	
	/* Check for bit translation */
	cast_stmt bt_if = mu_bit_translation_necessary(0);
	assert(bt_if);
	cast_stmt bt_end = mu_bit_translation_necessary(2);
	assert(bt_end);
	if (strncmp("#if 0", bt_if->cast_stmt_u_u.text, 5) == 0) {
		bt_if = bt_end = 0;
	}

	/* swap on encode */
	if (bt_if && (op & MUST_ENCODE)
	    && (elem_size == 8 || elem_size == 4 || elem_size == 2)) {
		add_stmt(bt_if);
		add_stmt(cast_new_stmt_expr(cast_new_expr_call_2(
			cast_new_expr_name(flick_asprintf(
				"flick_%s_array_swap%d",
				get_be_name(),
				elem_size * 8)),
			array_expr,
			length_expr)));
		add_stmt(bt_end);
	}
	
	/* call the macro to do the [un]marshaling */
	cast_expr macro = cast_new_expr_name(
		flick_asprintf("flick_%s_%s_%s_array",
			       get_be_name(),
			       which_stub,
			       get_buf_name()));
	
	cast_expr call = cast_new_expr_call(macro, 2);
	call->cast_expr_u_u.call.params.params_val[0]
		= array_expr;
	call->cast_expr_u_u.call.params.params_val[1]
		= length_expr;
	add_stmt(cast_new_stmt_expr(call));
	
	/* swap on decode */
	if (bt_if && (op & MUST_DECODE)
	    && (elem_size == 8 || elem_size == 4 || elem_size == 2)) {
		add_stmt(bt_if);
		add_stmt(cast_new_stmt_expr(cast_new_expr_call_2(
			cast_new_expr_name(flick_asprintf(
				"flick_%s_array_swap%d",
				get_be_name(),
				elem_size * 8)),
			array_expr,
			length_expr)));
		add_stmt(bt_end);
	}
	
	/* check the size of the array */
	check_for_warning(len_max, has_payload, array_expr);
	if (!has_payload)
		has_payload = array_expr;
	
	/* Can't dealloc until message is sent */
}

void trapeze_mu_state::mu_mapping_string(cast_expr ptr,
					 cast_type ptr_ctype,
					 mint_array_def *arr,
					 pres_c_allocation *mem_alloc)
{
	unsigned len_min, len_max;
	
	mint_ref itype = arr->length_type;
	
	assert(pres->mint.defs.defs_val[itype].kind == MINT_INTEGER);
	
	len_min = pres->mint.defs.defs_val[itype].
		  mint_def_u.integer_def.min;
	
	len_max = len_min + (pres->mint.defs.defs_val[itype].
			     mint_def_u.integer_def.range);
	
	if (len_max + max_msg_size < TRAPEZE_MAX_CONTROL_SIZE) {
		/* do the normal thing and bail */
		mem_mu_state::mu_mapping_string(ptr, ptr_ctype,
						arr, mem_alloc);
		return;
	}

	cast_expr length_expr;
	if (!(op & MUST_DECODE))
		length_expr =
			cast_new_expr_call_1(
				cast_new_expr_name("strlen"),
				ptr);
	else
		length_expr = cast_new_expr_lit_int(0, 0);
	
	/* call the macro to do the [un]marshaling */
	cast_expr macro = cast_new_expr_name(
		flick_asprintf("flick_%s_%s_%s_array",
			       get_be_name(),
			       which_stub,
			       get_buf_name()));
	
	cast_expr call = cast_new_expr_call(macro, 2);
	call->cast_expr_u_u.call.params.params_val[0]
		= ptr;
	call->cast_expr_u_u.call.params.params_val[1]
		= length_expr;
	add_stmt(cast_new_stmt_expr(call));

	/* check the size of the array */
	check_for_warning(len_max, has_payload, ptr);
	if (!has_payload)
		has_payload = ptr;
}

trapeze_target_mu_state::trapeze_target_mu_state(pres_c_1 *pres,
						 mu_state_op op,
						 int assumptions,
						 char *which)
	: target_mu_state(pres, op, assumptions, which)
{
	protocol = what_protocol(pres);
}

trapeze_target_mu_state::trapeze_target_mu_state(
	const trapeze_target_mu_state &must)
	: target_mu_state(must)
{
	protocol = must.protocol;
}

mu_state *trapeze_target_mu_state::clone()
{
	return new trapeze_target_mu_state(*this);
}

char *trapeze_target_mu_state::get_encode_name() 
{
	if (protocol == TRAPEZE_ONC)
		return "xdr";
	return "cdr";
}

char *trapeze_target_mu_state::get_be_name()
{
	return "trapeze";
}

trapeze_client_mu_state::trapeze_client_mu_state(pres_c_1 *pres, mu_state_op op, int assumptions, char *which)
	: client_mu_state(pres, op, assumptions, which)
{
}

trapeze_client_mu_state::trapeze_client_mu_state(const trapeze_client_mu_state &must)
	: client_mu_state(must)
{
}

mu_state *trapeze_client_mu_state::clone()
{
	return new trapeze_client_mu_state(*this);
}

char *trapeze_client_mu_state::get_encode_name() 
{
	return "cdr";
}

char *trapeze_client_mu_state::get_be_name()
{
    return "trapeze";
}

char *trapeze_mu_state::get_encode_name() 
{
	if (protocol == TRAPEZE_ONC) {
		if (should_swap == TRAPEZE_NO_SWAP || !(op & MUST_DECODE))
			return "xdr";
		else
			return "xdr_swap";
	} else {
		if (should_swap == TRAPEZE_NO_SWAP || !(op & MUST_DECODE))
			return "cdr";
		else
			return "cdr_swap";
	}
}

char *trapeze_mu_state::get_be_name()
{
   return "trapeze";
}

void trapeze_mu_state::mu_server_func(pres_c_inline inl, mint_ref tn_r,
				      pres_c_server_func *sfunc,
				      pres_c_server_skel *sskel)
{
	has_payload = 0; /* reset the payload ptr */
	mem_mu_state::mu_server_func(inl, tn_r, sfunc, sskel);
}
	
/* This handles PRES_C_MAPPING_SYSTEM_EXCEPTION nodes,
   which are used to opaquely represent system exceptions.
   They're opaque, because their presentation varies greatly.
*/
void trapeze_mu_state::mu_mapping_system_exception(cast_expr expr,
						   cast_type /* ctype */,
						   mint_ref itype)
{
	mint_def *idef = &pres->mint.defs.defs_val[itype];
	
	assert(idef->kind == MINT_SYSTEM_EXCEPTION);
	
	// This is a system exception - deal with it separately
	// Not sure if we should break the chunk &/| glob...
	break_chunk();
	cast_expr macro = cast_new_expr_name(
		flick_asprintf("flick_%s_%s_system_exception",
			       pres->pres_context,
			       get_buf_name()));
	
	cast_expr cex = cast_new_expr_call(macro, 3);
	
	cex->cast_expr_u_u.call.params.params_val[0]
		= expr;
	cex->cast_expr_u_u.call.params.params_val[1]
		= cast_new_expr_name(get_encode_name());
	cex->cast_expr_u_u.call.params.params_val[2]
		= cast_new_expr_name(get_be_name());
	
	add_stmt(cast_new_stmt_expr(cex));
}

void
w_header_includes(pres_c_1 *pres)
{
	w_printf("#define FLICK_PROTOCOL_TRAPEZE%s\n",
		 (what_protocol(pres) == TRAPEZE_ONC)? "_ONC" : "");
	w_printf("#include <flick/link/trapeze.h>\n");
	w_printf("#include <flick/encode/%cdr.h>\n",
		 (what_protocol(pres) == TRAPEZE_ONC)? 'x' : 'c');
}

void replace_operation_ids(pres_c_1 *pres,
			   mint_ref itype,
			   pres_c_inline inl)
{
	static int counter = 0;
	
	assert(pres->mint.defs.defs_val[itype].kind == MINT_UNION);
	mint_union_def *udef = &(pres->mint.defs.defs_val[itype].mint_def_u.
				 union_def);
	
	assert(inl->kind == PRES_C_INLINE_COLLAPSED_UNION);
	pres_c_inline_collapsed_union *cuinl
		= &(inl->pres_c_inline_u_u.collapsed_union);
	
	/* Find the appropriate case. */
	mint_ref case_var_r;
	for (int case_num = 0; ; case_num++) {
		if (case_num >= (signed int) udef->cases.cases_len) {
			if (udef->dfault == mint_ref_null)
				panic("union collapsed on nonexistent case");
			case_var_r = udef->dfault;
			break;
		}

		/* replace the mint_const here with a manufactured
		   constant integer, so we don't have to [un]marshal
		   an unbounded array -- just a 4-byte int! */
		if (mint_const_cmp(udef->cases.cases_val[case_num].val,
				   cuinl->discrim_val) == 0) {
			if (cuinl->discrim_val->kind != MINT_CONST_INT) {
				mint_const myconst =
					mint_new_const_int(counter++);
				cuinl->discrim_val = myconst;
				udef->cases.cases_val[case_num].val = myconst;
				udef->discrim =
					pres->mint.standard_refs.signed32_ref;
			}
			break;
		}
	}
}

/* End of file. */
