/*
 * Copyright (c) 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>

#include <mom/compiler.h>
#include <mom/libaoi.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include "pg_fluke.hh"

/*
 * The Fluke PG version of `p_variable_array_type' looks for sequences with a
 * certain magic name (stored in `fdev_seq_type_name') and gives them a special
 * presentation.
 *
 * The special presentation differs from the normal sequence presentation in
 * that (1) the `_buffer' is alloced/freed with a special allocation function
 * and (2) the `_buffer' of an `in' sequence will not be freed by a server
 * dispatch function when the invoked server work function exits (i.e., the
 * server may hold a reference to the buffer data indefinitely and must free
 * the buffer itself).
 *
 * Coincidentally, the Fluke BE will use ``gathering'' to marshal instances of
 * the special sequence type; see `c/pbe/fluke/mu_array.cc'.
 *
 * XXX --- All of this magic exists for the benefit of KVM's user-mode device
 * driver work.  When Flick has a real presentation modification facility, KVM
 * can use that to get all the special sequence handling he needs.  But until
 * then, we hack!
 */
void pg_fluke::p_variable_array_type(aoi_array *array,
				     cast_type *out_ctype,
				     pres_c_mapping *out_map)
{
	/* Do the normal CORBA thing... */
	pg_corba::p_variable_array_type(array, out_ctype, out_map);
	
	/* ... and return if the sequence type name is not magic. */
	if (strcmp(name, fdev_seq_type_name) != 0)
		return;
	
	/*********************************************************************/
	
	/*
	 * Modify the presentation of this sequence type as needed by KVM.  Use
	 * a special allocator to get special memory, and don't cause the
	 * server dispatch function to free an `in' instance of this type.
	 */
	pres_c_inline_counted_array *inline_array;
	pres_c_mapping new_element_mapping;
	pres_c_allocation *new_alloc;
	
	assert((*out_ctype)->kind == CAST_TYPE_STRUCT_NAME);
	assert(((*out_map)->kind == PRES_C_MAPPING_STRUCT)
	       && ((*out_map)->pres_c_mapping_u_u.struct_i->kind
		   == PRES_C_INLINE_COUNTED_ARRAY)
		);
	
	/* Get the counted array inline node. */
	inline_array = &((*out_map)->pres_c_mapping_u_u.struct_i->
			 pres_c_inline_u_u.counted_array);
	
	/* Copy the element mapping, which must be a MAPPING_POINTER. */
	assert(inline_array->ptr.mapping->kind == PRES_C_MAPPING_POINTER);
	
	new_element_mapping = pres_c_new_mapping(PRES_C_MAPPING_POINTER);
	*new_element_mapping = *(inline_array->ptr.mapping);
	
	/*
	 * Tweak the new mapping's allocation.  One set of flags works for both
	 * client and server.  We copy the allocator name to avoid warnings
	 * about `const'ness.
	 */
	new_alloc = &(new_element_mapping->pres_c_mapping_u_u.pointer.alloc);
	new_alloc->flags = (PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_NEVER);
	new_alloc->allocator = flick_asprintf("%s", fdev_seq_alloc_name);
	
	/* Insert the new element mapping back into the array inline. */
	inline_array->ptr.mapping= new_element_mapping;
	
	/*
	 * Finally, emit an `#include' directive to get the prototypes for the
	 * special allocator functions.
	 */
	p_emit_include_stmt(flick_asprintf("%s", fdev_seq_alloc_header), 1);
}

/* End of file. */

