/*
 * Copyright (c) 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

/* volkmar: I copied the stub-generation code to support propagation and 
 * non-propagation. I seperately generate two stubs - one without and one 
 * with the propagation. The flag-field of the propagation-function has the
 * PRES_C_STUB_OP_FLAG_PROPAGATE flag set for further checks in the back-end
 * generator. 
 */
 

#include <assert.h>
#include <string.h>

#include <mom/compiler.h>
#include <mom/libmint.h>
#include <mom/c/libpres_c.h>
#include <mom/libaoi.h>
#include <mom/c/libcast.h>

#include "pg_l4.hh"

void pg_l4::p_client_stubs_internal(aoi_ref this_ref, aoi_ref derived_ref)
{
	aoi_interface *this_interface, *derived_interface;
	
	aoi_ref saved_parent_ref, saved_derived_ref;
	
	u_int ops_len;
#ifdef ATTRIBUTE_STUBS
	u_int attribs_len;
#endif
	u_int parents_len;
	
	u_int i;
	
	int flags;
	
	/*
	 * Set `this_interface' and `derived_interface' to point at the AOI
	 * interfaces under consideration.
	 */
	assert((this_ref >= 0)
	       && (this_ref < ((aoi_ref) in_aoi->aoi_len)));
	assert(a(this_ref).binding
	       && (a(this_ref).binding->kind == AOI_INTERFACE));
	
	this_interface = &(a(this_ref).binding->aoi_type_u_u.interface_def);
	
	assert((derived_ref >= 0)
	       && (derived_ref < ((aoi_ref) in_aoi->aoi_len)));
	assert(a(derived_ref).binding
	       && (a(derived_ref).binding->kind == AOI_INTERFACE));
	
	derived_interface = &(a(derived_ref).binding->aoi_type_u_u.
			      interface_def);
	
	/*
	 * Set the `pg_state' interface reference data members to point at the
	 * interfaces under consideration so that `pg_state::calc_name' can
	 * compute names.
	 */
	saved_parent_ref = parent_interface_ref;
	saved_derived_ref = derived_interface_ref;
	
	parent_interface_ref = this_ref;
	derived_interface_ref = derived_ref;
	
	/* Get the lengths of arrays that are part of `this_interface'. */
	ops_len = this_interface->ops.ops_len;
#ifdef ATTRIBUTE_STUBS
	attribs_len = this_interface->attribs.attribs_len;
#endif
	parents_len = this_interface->parents.parents_len;
	
	/*
	 * Generate the stubs for the operations defined by `this_interface',
	 * which are inherited by `derived_interface'.
	 */
	for (i = 0; i < ops_len; ++i) {
		aoi_operation *ao = &(this_interface->ops.ops_val[i]);
		
		/*
		 * We must check if we have already generated a stub for this
		 * (interface, operation) pair.  This may occur, for example,
		 * when `this_interface' is inherited through multiple paths
		 * to `derived_interface'.
		 */
		flags = p_interface_table_get_value(derived_interface, ao);
		
		if (!(flags & P_CLIENT_STUB_OP_MARK)) {
			p_interface_table_set_value(derived_interface, ao,
						    (flags
						     | P_CLIENT_STUB_OP_MARK));
			p_client_stub(derived_interface, ao);
			p_client_stub_propagate(derived_interface, ao);
		}
	}
	
#ifdef ATTRIBUTE_STUBS
	/*
	 * Generate the stubs for the attributes defined by `this_interface',
	 * which are inherited by `derived_interface'.
	 */
	for (i = 0; i < attribs_len; ++i) {
		aoi_attribute *aa = &(this_interface->attribs.attribs_val[i]);
		
		/*
		 * As above, we must screen out attributes that we have already
		 * processed.
		 */
		flags = p_interface_table_get_value(derived_interface, aa);
		
		if (!(flags & P_CLIENT_STUB_OP_MARK)) {
			p_interface_table_set_value(derived_interface, aa,
						    (flags
						     | P_CLIENT_STUB_OP_MARK));
			p_client_attrib_stub(derived_interface, aa);
		}
	}
#endif /* ATTRIBUTE_STUBS */
	
	/*
	 * Generate the stubs for the operations and attributes that are
	 * inherited from parents of `this_interface'.
	 */
	for (i = 0; i < parents_len; ++i) {
		aoi_type parent_val = this_interface->parents.parents_val[i];
		aoi_ref  parent_ref;
		aoi_interface *parent_interface;
		
		/*
		 * All parent references must be through indirects so that
		 * we can find the name to go with the parent interface!
		 */
		assert(parent_val->kind == AOI_INDIRECT);
		
		parent_ref = aoi_deref_fwd(in_aoi, parent_val->aoi_type_u_u.indirect_ref);
		
		parent_interface = &(a(parent_ref).binding->aoi_type_u_u.
				     interface_def);
		
		/*
		 * As above, we check to see if we've already processed this
		 * parent.  (If we didn't do this check we would still generate
		 * correct code because all of the parent's operations and
		 * attributes are marked separately.  But doing this check can
		 * save us time.)
		 */
		flags = p_interface_table_get_value(derived_interface,
						    parent_interface);
		if (!(flags & P_CLIENT_STUB_PARENT_MARK)) {
			p_interface_table_set_value(
				derived_interface,
				parent_interface,
				(flags | P_CLIENT_STUB_PARENT_MARK));
			p_client_stubs_internal(parent_ref,
						derived_ref);
		}
	}
	
	/*
	 * Finally, restore the `pg_state' AOI interface references that we
	 * changed previously.
	 */
	parent_interface_ref = saved_parent_ref;	  
	derived_interface_ref = saved_derived_ref;	  
}



/*****************************************************************************/

/*
 * Generate a client stub presentation for an AOI interface operation.
 *
 * Note that `ao' may not be defined in `ai', but rather in some parent
 * interface of `ai'.
 */
void pg_l4::p_client_stub(aoi_interface *ai, aoi_operation *ao)
{
	pres_c_client_stub *cstub;
	int newstub;
	
	char *old_name;
	char *opname;
	int cast_params_len;
	int cr;
	cast_func_type cfunc;
	
	stub_special_params specials;
	int i, j;
	
	mint_ref request_ref;
	mint_ref reply_ref;
	
	mint_const oper_request;
	mint_const oper_reply;
	
	/*****/
	
	/*
	 * Some presentation styles dictate that client stubs are created only
	 * for operations that are *defined* within an interface, not for
	 * operations that are *inherited* from parent interfaces.  In such a
	 * presentation, one would invoke the parent's client stub to invoke an
	 * inherited operation on an instance of a derived object type.
	 */
	if ((parent_interface_ref != derived_interface_ref)
	    && !client_stubs_for_inherited_operations)
		return;
	
	if(!lookup_interface_mint_discrims(ai, ao,
					   &oper_request, &oper_reply)) {
		panic("In `pg_state::p_client_stub', "
		      "cannot find MINT request and reply discriminators.");
	}
	
	/*
	 * Find the MINT references to the request and reply types.
	 */
	p_client_stub_find_refs(ai, ao, oper_request, oper_reply,
				&request_ref,
				&reply_ref);
	
	/*
	 * Determine the special parameters for this stub: the target object
	 * reference, the environment reference, etc.
	 */
	p_client_stub_special_params(ao, &specials);
	
	/*
	 * Determine the total number of stub parameters.
	 */
	cast_params_len = ao->params.params_len;
	for (i = 0;
	     i < stub_special_params::number_of_stub_special_param_kinds;
	     ++i)
		if (specials.params[i].index != -1)
			++cast_params_len;
	
	/*
	 * Verify our set of special parameters.
	 */
	for (i = 0;
	     i < stub_special_params::number_of_stub_special_param_kinds;
	     ++i) {
		if (specials.params[i].index != -1) {
			/*
			 * Assert that this special parameter has a valid
			 * CAST type and a valid, unique index.
			 */
			assert(specials.params[i].ctype != 0);
			assert((specials.params[i].index >= 0)
			       && (specials.params[i].index
				   < cast_params_len));
			
			for (j = i + 1;
			     j < stub_special_params::
				 number_of_stub_special_param_kinds;
			     ++j)
				assert(specials.params[i].index
				       != specials.params[j].index);
		}
	}
	
	/* Save the original name context for when we return. */
	old_name = name;
	name = calc_client_stub_name(ao->name);
	opname = name;
	
	/*
	 * Now we are ready to start building the presentation PRES_C and CAST
	 * goo.
	 */
	
	if (cast_params_len) {
		cfunc.params.params_len = cast_params_len;
		cfunc.params.params_val 
			= (cast_func_param *)
			  mustcalloc(cast_params_len *
				     sizeof(cast_func_param));
	}
	
	/*
	 * Build the group of pres_c_inline structures containing the
	 * parameters, starting with level 4 (params struct) then levels
	 * 3, 2, 1 (unions).
	 *
	 * KBF --- the L4 reply inline is no long a struct, it's a UNION of 3
	 * items:
	 * 0  - the old reply params struct
	 * 1  - union of all user defined exceptions that this operation
	 *      supports
	 * -1 - a system exception (out of memory, etc.)
	 */
	pres_c_inline request_l4_inl = pres_c_new_inline_func_params_struct(0);
	pres_c_inline reply_l4_inl = pres_c_new_inline_func_params_struct(0);
	pres_c_mapping return_type_mapping;
	cast_type out_ctype_temp;
	
	process_client_params(&cfunc,
			      &specials,
			      request_ref, reply_ref,
			      ao,
			      request_l4_inl, reply_l4_inl);
	
	/*
	 * Now process the return type.
	 *
	 * An inline atom index of `-1' indicates a return value.
	 * The MINT type of the return value is stored in the last slot of the
	 * reply structure (thus, the hairy expression for the second argument
	 * in the call below).  See `tam_operation_reply_struct' in the file
	 * `aoi_to_mint.c' for more information.
	 *
	 * KBF - This is now in a UNION before the structure...
	 */
	mint_ref reply_struct_ref = (m(reply_ref).mint_def_u.union_def.
				     cases.cases_val[0].var);
	p_client_stub_return_type(ao,
				  (m(reply_struct_ref).mint_def_u.struct_def.
				   slots.
				   slots_val[m(reply_struct_ref).mint_def_u.
					    struct_def.slots.slots_len - 1]),
				  &out_ctype_temp,
				  &return_type_mapping);
	cfunc.return_type = out_ctype_temp;
	
	/*
	 * Allocate a pres_c_stub for this operation --- UNLESS we have already
	 * created an appropriate client stub.
	 *
	 * Perhaps we are inheriting the current AOI operation from a parent
	 * interface and the user has changed the rule for creating stub names
	 * so that we now have a name conflict.  (In this case, however, the
	 * user should instead tweak the flag that controls the creation of
	 * client stubs for inherited operations.)
	 *
	 * XXX --- We should be more careful about this check.  Checking just
	 * the name is bad news.  We must check for MINT and PRES_C equality as
	 * well!
	 */
	if (cast_find_def(&(out_pres->cast), name) >= 0) {
		/*
		 * XXX --- Check that our MINT and PRES_C is equal to that of
		 * the existing client stub.  (Hell, check that `name' refers
		 * to a client stub and not some typedef or something else!)
		 *
		 * In lieu of being smart, warn the user.
		 */
		warn("Suppressing extra definition for client stub `%s'.",
		     name);
		
		/* Restore the name context. */
		name = old_name;
		return;
	}
	
	newstub = p_add_stub(out_pres);
	s(newstub).kind = PRES_C_CLIENT_STUB;
	cstub = &s(newstub).pres_c_stub_u.cstub;
	
	cstub->op_flags = PRES_C_STUB_OP_FLAG_NONE;
	
	/* Determine if the operation is oneway. */
        if (ao->flags & AOI_OP_FLAG_ONEWAY)
		cstub->op_flags |= PRES_C_STUB_OP_FLAG_ONEWAY;

	/*
	 * Set request_itype and reply_itype to the top-level "mom_msg" MINT
	 * union.
	 */
	cstub->request_itype = top_union;
	cstub->reply_itype = top_union;
	
	cr = cast_add_func_def(&out_pres->cast, opname, 0);
	out_pres->cast.cast_scope_val[cr].u.cast_def_u_u.func_type = cfunc;
	out_pres->cast.cast_scope_val[cr].included = a(cur_aoi_idx).included;
	cstub->c_func = cr;
	
	/*
	 * Allocate and set the return value slot in the PRES_C `reply_l4_inl'.
	 * The MINT type for the return value is always the last slot in the
	 * reply's MINT struct type.
	 *
	 * Additionally, initialize the request `return_slot' to null.
	 */
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot
		= ((pres_c_inline_struct_slot *)
		   mustmalloc(sizeof(*(reply_l4_inl->
				       pres_c_inline_u_u.func_params_i.
				       return_slot))));
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot->
		mint_struct_slot_index
		= (m(reply_struct_ref).mint_def_u.struct_def.slots.slots_len
		   - 1);
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot->
		inl
		= pres_c_new_inline_atom(pres_c_func_return_index,
					 return_type_mapping);
	
	request_l4_inl->pres_c_inline_u_u.func_params_i.return_slot = 0;
	
	/*
	 * We need to turn the reply into the union of good, bad, & ugly values
	 */
	p_do_return_union(ao, &reply_l4_inl, reply_ref, cr);
	
	/*
	 * level 3
	 *
	 * XXX - might need to modify the operation request code here,
	 * e.g. add a prefix in the CORBA case (perhaps in overriding
	 * function)
	 */
	pres_c_inline request_l3_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l3_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		oper_request;
	request_l3_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l4_inl;
	
	pres_c_inline reply_l3_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l3_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		oper_reply;
	reply_l3_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		reply_l4_inl;
	
	/* level 2 */
	pres_c_inline request_l2_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l2_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_from_aoi_const(ai->code);
	request_l2_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l3_inl;
	
	pres_c_inline reply_l2_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l2_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_from_aoi_const(ai->code);
	reply_l2_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		reply_l3_inl;
	
	/* level 1 */
	pres_c_inline request_l1_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l1_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_int((int)ai->idl);
	request_l1_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l2_inl;
	
	pres_c_inline reply_l1_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l1_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_int((int)ai->idl);
	reply_l1_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		reply_l2_inl;
	
	/* Assign the unions and param struct to cstub fields. */
	cstub->request_i = request_l1_inl;
	cstub->reply_i = reply_l1_inl;
	
	/* Set the target_i field. */
	pres_c_mapping target_mapping
		= pres_c_new_mapping(PRES_C_MAPPING_REFERENCE);
	
	target_mapping->pres_c_mapping_u_u.ref.kind = PRES_C_REFERENCE_COPY;
	target_mapping->pres_c_mapping_u_u.ref.ref_count = 1;
	cstub->target_i =
		pres_c_new_inline_atom(
			specials.params[stub_special_params::object_ref].index,
			target_mapping);
	cstub->target_itype = out_pres->mint.standard_refs.interface_name_ref;
	
	/* Set the client_i field. */
	cstub->client_i = 0;
	cstub->client_itype = mint_ref_null;
	
	/* Set the error_i field. */

	/* XXX until exception handling is implemented */
	cstub->error_i = reply_l1_inl;
	cstub->error_itype = top_union;
	
	/* Restore the name context. */
	name = old_name;
}


void pg_l4::p_client_stub_propagate(aoi_interface *ai, aoi_operation *ao)
{
	pres_c_client_stub *cstub;
	int newstub;
	
	char *old_name;
	char *opname;
	int cast_params_len;
	int cr;
	cast_func_type cfunc;
	
	stub_special_params specials;
	int i, j;
	
	mint_ref request_ref;
	mint_ref reply_ref;
	
	mint_const oper_request;
	mint_const oper_reply;

	aoi_kind old_return_kind;
	
	/*****/
	
	/*
	 * Some presentation styles dictate that client stubs are created only
	 * for operations that are *defined* within an interface, not for
	 * operations that are *inherited* from parent interfaces.  In such a
	 * presentation, one would invoke the parent's client stub to invoke an
	 * inherited operation on an instance of a derived object type.
	 */
	if ((parent_interface_ref != derived_interface_ref)
	    && !client_stubs_for_inherited_operations)
		return;
	
	if(!lookup_interface_mint_discrims(ai, ao,
					   &oper_request, &oper_reply)) {
		panic("In `pg_state::p_client_stub', "
		      "cannot find MINT request and reply discriminators.");
	}
	
	/*
	 * Find the MINT references to the request and reply types.
	 */
	p_client_stub_find_refs(ai, ao, oper_request, oper_reply,
				&request_ref,
				&reply_ref);
	
	/*
	 * Determine the special parameters for this stub: the target object
	 * reference, the environment reference, etc.
	 */
	p_client_stub_prop_special_params(ao, &specials);
	
	/*
	 * Determine the total number of stub parameters.
	 */
	cast_params_len = ao->params.params_len;
	for (i = 0;
	     i < stub_special_params::number_of_stub_special_param_kinds;
	     ++i)
		if (specials.params[i].index != -1)
			++cast_params_len;
	
	/*
	 * Verify our set of special parameters.
	 */
	for (i = 0;
	     i < stub_special_params::number_of_stub_special_param_kinds;
	     ++i) {
		if (specials.params[i].index != -1) {
			/*
			 * Assert that this special parameter has a valid
			 * CAST type and a valid, unique index.
			 */
			assert(specials.params[i].ctype != 0);
			assert((specials.params[i].index >= 0)
			       && (specials.params[i].index
				   < cast_params_len));
			
			for (j = i + 1;
			     j < stub_special_params::
				 number_of_stub_special_param_kinds;
			     ++j)
				assert(specials.params[i].index
				       != specials.params[j].index);
		}
	}
	
	/* Save the original name context for when we return. */
	old_name = name;
	name = calc_client_stub_propagate_name(ao->name);
	opname = name;
	
	/*
	 * Now we are ready to start building the presentation PRES_C and CAST
	 * goo.
	 */
	
	if (cast_params_len) {
		cfunc.params.params_len = cast_params_len;
		cfunc.params.params_val 
			= (cast_func_param *)
			  mustcalloc(cast_params_len *
				     sizeof(cast_func_param));
	}
	/*
	 * Build the group of pres_c_inline structures containing the
	 * parameters, starting with level 4 (params struct) then levels
	 * 3, 2, 1 (unions).
	 *
	 * KBF --- the L4 reply inline is no long a struct, it's a UNION of 3
	 * items:
	 * 0  - the old reply params struct
	 * 1  - union of all user defined exceptions that this operation
	 *      supports
	 * -1 - a system exception (out of memory, etc.)
	 */
	pres_c_inline request_l4_inl = pres_c_new_inline_func_params_struct(0);
	pres_c_inline reply_l4_inl = pres_c_new_inline_func_params_struct(0);
	pres_c_mapping return_type_mapping;
	cast_type out_ctype_temp;
	
	process_client_prop_params(&cfunc,
				   &specials,
				   request_ref, reply_ref,
				   ao,
				   request_l4_inl, reply_l4_inl);

	/*
	 * Now process the return type.
	 *
	 * An inline atom index of `-1' indicates a return value.
	 * The MINT type of the return value is stored in the last slot of the
	 * reply structure (thus, the hairy expression for the second argument
	 * in the call below).  See `tam_operation_reply_struct' in the file
	 * `aoi_to_mint.c' for more information.
	 *
	 * KBF - This is now in a UNION before the structure...
	 */
	mint_ref reply_struct_ref = (m(reply_ref).mint_def_u.union_def.
				     cases.cases_val[0].var);

	/* volkmar: propagation always has void as return type */

	old_return_kind = ao->return_type->kind;
	ao->return_type->kind = AOI_VOID;

	p_client_stub_return_type(ao,
				  (m(reply_struct_ref).mint_def_u.struct_def.
				   slots.
				   slots_val[m(reply_struct_ref).mint_def_u.
					    struct_def.slots.slots_len - 1]),
				  &out_ctype_temp,
				  &return_type_mapping);

	cfunc.return_type = out_ctype_temp;

	/* volkmar: you never know, where they use it elsewhere... */
	ao->return_type->kind = old_return_kind;
	
	/*
	 * Allocate a pres_c_stub for this operation --- UNLESS we have already
	 * created an appropriate client stub.
	 *
	 * Perhaps we are inheriting the current AOI operation from a parent
	 * interface and the user has changed the rule for creating stub names
	 * so that we now have a name conflict.  (In this case, however, the
	 * user should instead tweak the flag that controls the creation of
	 * client stubs for inherited operations.)
	 *
	 * XXX --- We should be more careful about this check.  Checking just
	 * the name is bad news.  We must check for MINT and PRES_C equality as
	 * well!
	 */
	if (cast_find_def(&(out_pres->cast), name) >= 0) {
		/*
		 * XXX --- Check that our MINT and PRES_C is equal to that of
		 * the existing client stub.  (Hell, check that `name' refers
		 * to a client stub and not some typedef or something else!)
		 *
		 * In lieu of being smart, warn the user.
		 */
		warn("Suppressing extra definition for client stub `%s'.",
		     name);
		
		/* Restore the name context. */
		name = old_name;
		return;
	}
	
	newstub = p_add_stub(out_pres);
	s(newstub).kind = PRES_C_CLIENT_STUB;
	cstub = &s(newstub).pres_c_stub_u.cstub;
	
	cstub->op_flags = PRES_C_STUB_OP_FLAG_NONE;

	/* this is a propagation function */
	cstub->op_flags |= PRES_C_STUB_OP_FLAG_PROPAGATE;
	
	/* Determine if the operation is oneway. */
        if (ao->flags & AOI_OP_FLAG_ONEWAY)
		cstub->op_flags |= PRES_C_STUB_OP_FLAG_ONEWAY;

	/*
	 * Set request_itype and reply_itype to the top-level "mom_msg" MINT
	 * union.
	 */
	cstub->request_itype = top_union;
	cstub->reply_itype = top_union;
	
	cr = cast_add_func_def(&out_pres->cast, opname, 0);
	out_pres->cast.cast_scope_val[cr].u.cast_def_u_u.func_type = cfunc;
	out_pres->cast.cast_scope_val[cr].included = a(cur_aoi_idx).included;
	cstub->c_func = cr;
	
	/*
	 * Allocate and set the return value slot in the PRES_C `reply_l4_inl'.
	 * The MINT type for the return value is always the last slot in the
	 * reply's MINT struct type.
	 *
	 * Additionally, initialize the request `return_slot' to null.
	 */
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot
		= ((pres_c_inline_struct_slot *)
		   mustmalloc(sizeof(*(reply_l4_inl->
				       pres_c_inline_u_u.func_params_i.
				       return_slot))));
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot->
		mint_struct_slot_index
		= (m(reply_struct_ref).mint_def_u.struct_def.slots.slots_len
		   - 1);
	reply_l4_inl->
		pres_c_inline_u_u.func_params_i.return_slot->
		inl
		= pres_c_new_inline_atom(pres_c_func_return_index,
					 return_type_mapping);
	
	request_l4_inl->pres_c_inline_u_u.func_params_i.return_slot = 0;
	
	/*
	 * We need to turn the reply into the union of good, bad, & ugly values
	 */
	p_do_return_union(ao, &reply_l4_inl, reply_ref, cr);
	
	/*
	 * level 3
	 *
	 * XXX - might need to modify the operation request code here,
	 * e.g. add a prefix in the CORBA case (perhaps in overriding
	 * function)
	 */
	pres_c_inline request_l3_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l3_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		oper_request;
	request_l3_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l4_inl;
	
	pres_c_inline reply_l3_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l3_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		oper_reply;
	reply_l3_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		reply_l4_inl;
	
	/* level 2 */
	pres_c_inline request_l2_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l2_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_from_aoi_const(ai->code);
	request_l2_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l3_inl;
	
	pres_c_inline reply_l2_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l2_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_from_aoi_const(ai->code);
	reply_l2_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		reply_l3_inl;
	
	/* level 1 */
	pres_c_inline request_l1_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	request_l1_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_int((int)ai->idl);
	request_l1_inl->pres_c_inline_u_u.collapsed_union.selected_case =
		request_l2_inl;

	pres_c_inline reply_l1_inl =
		pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	reply_l1_inl->pres_c_inline_u_u.collapsed_union.discrim_val =
		mint_new_const_int((int)ai->idl);
	reply_l1_inl->pres_c_inline_u_u.collapsed_union.selected_case =
	reply_l2_inl;


	/* Assign the unions and param struct to cstub fields. */
	cstub->request_i = request_l1_inl;
	cstub->reply_i = reply_l1_inl;
	
	/* Set the target_i field. */
	pres_c_mapping target_mapping
		= pres_c_new_mapping(PRES_C_MAPPING_REFERENCE);
	
	target_mapping->pres_c_mapping_u_u.ref.kind = PRES_C_REFERENCE_COPY;
	target_mapping->pres_c_mapping_u_u.ref.ref_count = 1;
	cstub->target_i =
		pres_c_new_inline_atom(
			specials.params[stub_special_params::object_ref].index,
			target_mapping);
	cstub->target_itype = out_pres->mint.standard_refs.interface_name_ref;
	
	/* Set the client_i field. */
	cstub->client_i = 0;
	cstub->client_itype = mint_ref_null;
	
	/* Set the error_i field. */

	/* XXX until exception handling is implemented */
	cstub->error_i = reply_l1_inl;
	cstub->error_itype = top_union;
	
	/* Restore the name context. */
	name = old_name;
}


void pg_l4::p_client_stub_special_params(aoi_operation *ao,
					    stub_special_params *specials)
{
	/* Do the library thing... */
	pg_corba::p_client_stub_special_params(ao, specials);
	
	/*
	 * ...and then change the types of the client and required server SIDs
	 * to be constant...
	 */
	specials->params[stub_special_params::client_sid].ctype =
		cast_new_qualified_type(
			cast_new_type_name(
				calc_client_stub_client_sid_type_name(
					ao->name
					)),
			CAST_TQ_CONST);
	
	/* XXX --- Note that this is ultimately ineffectual; see code below. */
	specials->params[stub_special_params::required_server_sid].ctype =
		cast_new_qualified_type(
			cast_new_type_name(
				calc_client_stub_server_sid_type_name(
					ao->name
					)),
			CAST_TQ_CONST);
	
	/*
	 * ...and then remove the required and actual server SIDs from the
	 * argument list.  As of December 1997, Flask no longer requires the
	 * server SIDs in the client stub parameter list.
	 *
	 * Reset all the SID and environment reference indices, to be safe.
	 */
	specials->params[stub_special_params::client_sid].index
		= (gen_sids ?
		   (ao->params.params_len + 1) :
		   -1);
	
	specials->params[stub_special_params::required_server_sid].index = -1;
	specials->params[stub_special_params::actual_server_sid].index   = -1;

	/* check if there are OUT or INOUT fpages as parameters */

	int has_out_fpages = 0;
	for (int i=0; i < ao->params.params_len; i++) {
		if ((ao->params.params_val[i].direction != AOI_DIR_IN) &&
		    (ao->params.params_val[i].type->kind == AOI_FPAGE))
		{
			has_out_fpages=1;
			break;
		}
	}
	

	/* volkmar: and finally some weared flexpage stuff...*/
	if (has_out_fpages) {
		specials->params[stub_special_params::rcv_fpage_desc].index = 1;
		specials->params[stub_special_params::rcv_fpage_desc].name =
			calc_rcv_fpage_descriptor_param_name(ao->name);
		specials->params[stub_special_params::rcv_fpage_desc].ctype =
			cast_new_type_name(
				calc_rcv_fpage_descriptor_type_name(ao->name));
	}

	specials->params[stub_special_params::environment_ref].index
		= (ao->params.params_len
		   + 1 /* the target object reference */
		   + (has_out_fpages ? 1 : 0)
		   + (gen_sids ?
		      1 /* after the client sid */ :
		      0 /* no SID arguments */ ));


}

void pg_l4::p_client_stub_prop_special_params(aoi_operation *ao,
					      stub_special_params *specials)
{
	stub_special_params::stub_param_info *this_param;

	/* Do the library thing... */
	pg_corba::p_client_stub_special_params(ao, specials);

	/* just do some renaming... */
	specials->params[stub_special_params::object_ref].name =
		calc_client_stub_prop_object_param_name(ao->name);
	specials->params[stub_special_params::object_ref].ctype = 
		cast_new_type_name(
			calc_client_stub_prop_object_type_name(ao->name));

	
	/*
	 * ...and then change the types of the client and required server SIDs
	 * to be constant...
	 */
	specials->params[stub_special_params::client_sid].ctype =
		cast_new_qualified_type(
			cast_new_type_name(
				calc_client_stub_client_sid_type_name(
					ao->name
					)),
			CAST_TQ_CONST);
	
	/* XXX --- Note that this is ultimately ineffectual; see code below. */
	specials->params[stub_special_params::required_server_sid].ctype =
		cast_new_qualified_type(
			cast_new_type_name(
				calc_client_stub_server_sid_type_name(
					ao->name
					)),
			CAST_TQ_CONST);
	
	/*
	 * ...and then remove the required and actual server SIDs from the
	 * argument list.  As of December 1997, Flask no longer requires the
	 * server SIDs in the client stub parameter list.
	 *
	 * Reset all the SID and environment reference indices, to be safe.
	 */
	specials->params[stub_special_params::client_sid].index
		= (gen_sids ?
		   (ao->params.params_len + 1) :
		   -1);
	
	specials->params[stub_special_params::required_server_sid].index = -1;
	specials->params[stub_special_params::actual_server_sid].index   = -1;


	/* huhhhh - this is not clean - we use the special fpage-param
	 * for the target-service!!!
	 */
	specials->params[stub_special_params::rcv_fpage_desc].index = 1;
	specials->params[stub_special_params::rcv_fpage_desc].name =
			calc_client_stub_object_param_name(ao->name);
	specials->params[stub_special_params::rcv_fpage_desc].ctype =
	    cast_new_type_name(
		calc_client_stub_object_type_name(ao->name));

	specials->params[stub_special_params::environment_ref].index
		= (ao->params.params_len
		   + 1 /* the target object reference */
		   + 1 /* the target service */
		   + (gen_sids ?
		      1 /* after the client sid */ :
		      0 /* no SID arguments */ ));

}


/* End of file. */

