/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <mom/compiler.h>
#include <mom/c/pfe.hh>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <assert.h>

#include "private.hh"

/*
 * Generate pres_c_inline (pres_c_inline_struct) for an operation parameter.
 */
cast_func_param pg_state::p_param(
	aoi_parameter *ap,
	int cast_param_index,			// NOT a CAST ref!
	mint_ref mr,			
	pres_c_inline request_i,
	pres_c_inline reply_i,
	int request_mint_struct_slot_index,	// NOT a MINT ref!
	int reply_mint_struct_slot_index	// NOT a MINT ref!
	)
{
	cast_type param_ctype;
	pres_c_mapping param_map;
	char *old_name;
	cast_func_param res;
	
	/* Some initial assertions and local variable assignments. */
	assert(request_i->kind == PRES_C_INLINE_FUNC_PARAMS_STRUCT);
	assert(reply_i->kind == PRES_C_INLINE_FUNC_PARAMS_STRUCT);
	
	/* Save the original name context.  */
	old_name = name;
	name = calc_stub_param_name(ap->name);
	
	/* Set the parameter name. */
	res.name = name;
	
	/* Set the parameter type. */
	p_param_type(ap->type, mr, ap->direction, &param_ctype, &param_map);
	res.type = param_ctype;
	
	/* Add the parameter to the request inline struct if appropriate. */
	if ((ap->direction == AOI_DIR_IN) ||
	    (ap->direction == AOI_DIR_INOUT)) {
		int inline_slot_index
			= pres_c_add_inline_struct_slot(request_i);
		
		request_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[inline_slot_index].
			mint_struct_slot_index
			= request_mint_struct_slot_index;
		
		request_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[inline_slot_index].
			inl
			= pres_c_new_inline_atom(cast_param_index, param_map);
	}
	if ((ap->direction == AOI_DIR_OUT)
	    && gen_server) {
		/*
		 *
		 */
		pres_c_mapping alloc_map;
		
		p_param_server_alloc_out(param_ctype, mr, param_map,
					 &alloc_map);
		if (alloc_map != 0) {
			int inline_slot_index
				= pres_c_add_inline_struct_slot(request_i);
			
			request_i->pres_c_inline_u_u.func_params_i.slots.
				slots_val[inline_slot_index].
				mint_struct_slot_index
				= mint_slot_index_null;
			
			request_i->pres_c_inline_u_u.func_params_i.slots.
				slots_val[inline_slot_index].
				inl
				= pres_c_new_inline_atom(cast_param_index,
							 alloc_map);
		}
	}
	
	/* Add the parameter to the reply inline struct if appropriate. */
	if ((ap->direction == AOI_DIR_INOUT) ||
	    (ap->direction == AOI_DIR_OUT)) {
		int inline_slot_index
			= pres_c_add_inline_struct_slot(reply_i);
		
		reply_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[inline_slot_index].
			mint_struct_slot_index
			= reply_mint_struct_slot_index;
		
		reply_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[inline_slot_index].
			inl
			= pres_c_new_inline_atom(cast_param_index, param_map);
	}
	
	/* Restore the name context.  */
	name = old_name;
	return res;
}

/*
 * Generate pres_c_inline for a client SID parameter.
 */
void pg_state::p_param_client_sid(int param_index,
				  pres_c_inline request_inline)
{
	pres_c_mapping	param_map;
	int		slot_index;
	
	/* Make the PRES_C mapping for the client SID. */
	param_map = pres_c_new_mapping(PRES_C_MAPPING_SID);
	
	param_map->pres_c_mapping_u_u.sid.kind = PRES_C_SID_CLIENT;
	
	/*
	 * Now make an inline atom containing our SID's mapping, and add that
	 * atom to the request inline structure.
	 *
	 * By setting our `mint_struct_slot_index' to `mint_slot_index_null',
	 * we indicate that this SID parameter has no representation in the
	 * MINT structure that describes the request message.  That is, the SID
	 * is outside of the IDL-specified interface.
	 */
	slot_index = pres_c_add_inline_struct_slot(request_inline);
	request_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		mint_struct_slot_index
		= mint_slot_index_null;
	request_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		inl
		= pres_c_new_inline_atom(param_index, param_map);
}

/*
 * Generate pres_c_inline for a required server SID parameter.
 */
void pg_state::p_param_required_server_sid(int param_index,
					   pres_c_inline request_inline)
{
	pres_c_mapping	param_map;
	int		slot_index;
	
	/* Make the PRES_C mapping for the server SID. */
	param_map = pres_c_new_mapping(PRES_C_MAPPING_SID);
	
	param_map->pres_c_mapping_u_u.sid.kind = PRES_C_SID_SERVER;
	
	/*
	 * Now make an inline atom containing our SID's mapping, and add that
	 * atom to the request inline structure.
	 *
	 * By setting our `mint_struct_slot_index' to `mint_slot_index_null',
	 * we indicate that this SID parameter has no representation in the
	 * MINT structure that describes the request message.  That is, the SID
	 * is outside of the IDL-specified interface.
	 */
	slot_index = pres_c_add_inline_struct_slot(request_inline);
	request_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		mint_struct_slot_index
		= mint_slot_index_null;
	request_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		inl
		= pres_c_new_inline_atom(param_index, param_map);
}

/*
 * Generate pres_c_inline for an actual server SID parameter.
 */
void pg_state::p_param_actual_server_sid(int param_index,
					 pres_c_inline reply_inline)
{
	pres_c_mapping	param_map;
	int		slot_index;
	
	/* Make the PRES_C mapping for the server SID. */
	param_map = pres_c_new_mapping(PRES_C_MAPPING_SID);
	
	param_map->pres_c_mapping_u_u.sid.kind = PRES_C_SID_SERVER;
	
	/*
	 * Now make an inline atom containing our SID's mapping, and add that
	 * atom to the reply inline structure.
	 *
	 * By setting our `mint_struct_slot_index' to `mint_slot_index_null',
	 * we indicate that this SID parameter has no representation in the
	 * MINT structure that describes the request message.  That is, the SID
	 * is outside of the IDL-specified interface.
	 */
	slot_index = pres_c_add_inline_struct_slot(reply_inline);
	reply_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		mint_struct_slot_index
		= mint_slot_index_null;
	reply_inline->pres_c_inline_u_u.struct_i.slots.slots_val[slot_index].
		inl
		= pres_c_new_inline_atom(param_index, param_map);
}


/*****************************************************************************/

/***** Auxiliary functions. *****/

/* This method determines how `p_param' processes a parameter type.
   Some presentation generators override this method. */
void pg_state::p_param_type(aoi_type at, mint_ref /*mr*/, aoi_direction dir,
			    cast_type *out_ctype, pres_c_mapping *out_mapping)
{
	/*
	 * The default behavior is to:
	 * (1) call `p_type' to compute the basic C type and PRES_C mapping;
	 * (2) interpose a pointer for `inout' and `out' parameters; and
	 * (3) encapsulate the mapping with information about the parameter
	 *     direction.
	 */
	
	p_type(at, out_ctype, out_mapping);
	
	if ((dir == AOI_DIR_INOUT) || (dir == AOI_DIR_OUT))
		pres_c_interpose_pointer(out_ctype,
					 out_mapping,
					 (PRES_C_ALLOC_NEVER |
					  PRES_C_DEALLOC_NEVER),
					 p_get_allocator());
	
	pres_c_interpose_direction(out_mapping, dir);
}

/*
 * THIS CODE HAS BEEN COPIED INTO `fe/mig/xlate_pres.c' AS
 * make_arg_server_alloc_out(). ANY CHANGES SHOULD BE REFLECTED THERE ALSO!
 * 
 * Decide if this is an `out' parameter for which the server muct allocate
 * storage beyond the root.  If so, create an `alloc_mapping' for doing this.
 */
void pg_state::p_param_server_alloc_out(cast_type param_ctype,
					mint_ref param_itype,
					pres_c_mapping param_mapping,
					pres_c_mapping *alloc_mapping)
{
	cast_type new_ctype;
	pres_c_mapping new_mapping;
	
	/*****/
	
	*alloc_mapping = 0;
	
	switch (param_mapping->kind) {
	default:
		/*
		 * By default, we assume that the server dispatch function is
		 * not required to do anything special beyond allocating the
		 * ``root'' of the out parameter.
		 */
		warn("In `p_param_server_alloc_out', "
		     "unexpected mapping type %d.",
		     param_mapping->kind);
		break;
		
	case PRES_C_MAPPING_POINTER:
		/*
		 *
		 */
		*alloc_mapping = pres_c_new_mapping(PRES_C_MAPPING_POINTER);
		
		(*alloc_mapping)->pres_c_mapping_u_u.pointer.alloc.flags
			= (PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS);
		(*alloc_mapping)->pres_c_mapping_u_u.pointer.alloc.allocator
			= "auto";
		(*alloc_mapping)->pres_c_mapping_u_u.pointer.target
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
		
	case PRES_C_MAPPING_FIXED_ARRAY:
		/*
		 *
		 */
		*alloc_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_FIXED_ARRAY);
		
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.alloc.flags
			= (PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS);
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.alloc.
			allocator
			= "auto";
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.length
			= param_mapping->pres_c_mapping_u_u.fixed_array.length;
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
			element_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		
		break;
		
	case PRES_C_MAPPING_STUB:
		/*
		 *
		 */
		new_ctype = param_ctype;
		new_mapping = param_mapping;
		
		pres_descend_mapping_stub(out_pres, param_itype,
					  &new_ctype, &new_mapping);
		p_param_server_alloc_out(new_ctype, param_itype, new_mapping,
					 alloc_mapping);
		break;
		
	case PRES_C_MAPPING_DIRECTION:
		p_param_server_alloc_out(param_ctype, param_itype,
					 (param_mapping->pres_c_mapping_u_u.
					  direction.mapping),
					 alloc_mapping);
		break;
	}
}

/* End of file. */

