/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>
#include <stdlib.h>

#include <mom/compiler.h>
#include <mom/libaoi.h>
#include <mom/libmint.h>
#include <mom/c/pfe.hh>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include "private.hh"

#define FLICK_SERVER_TYPE_NAME "flick_dispatch_t"

/* Generate server skeleton stub presentation for an AOI interface. */
void pg_state::p_server_skel(aoi_interface * /*ai*/)
{
	pres_c_server_skel *sskel;
	char *sskel_name;
	int sskel_index;
	
	aoi_ref saved_parent_ref, saved_derived_ref;
	
	/* Save the original name context for when we return. */
	char *old_name = name;
	name = FLICK_SERVER_TYPE_NAME;
	
	/*
	 * Set the `pg_state' interface reference data members to point at the
	 * interfaces under consideration so that `pg_state::calc_name' can
	 * compute names.
	 */
	saved_parent_ref = parent_interface_ref;
	saved_derived_ref = derived_interface_ref;
	
	parent_interface_ref = cur_aoi_idx;
	derived_interface_ref = cur_aoi_idx;
	
	/* Allocate a PRES_C_STUB for this interface. */
	sskel_index = p_add_stub(out_pres);
	s(sskel_index).kind = PRES_C_SERVER_SKEL;
	sskel = &s(sskel_index).pres_c_stub_u.sskel;
	
	/*
	 * Prepare the skeleton name.
	 */
	sskel_name = calc_server_skel_name("");
	

	/*
	 * Set `sskel->c_def' to the C declaration for this server skeleton.
	 */
	sskel->c_def = p_server_skel_cdef(sskel_name);

	out_pres->cast.cast_scope_val[sskel->c_def].included = a(cur_aoi_idx).
							       included;
	
	/*
	 * Set `sskel->request_itype' and `sskel->reply_itype' to the top-level
	 * MINT message union.
	 */
	sskel->request_itype = top_union;
	sskel->reply_itype = top_union;
	
	/*
	 * Restore the name context BEFORE we try to build the PRES_C for any
	 * server functions.
	 */
	name = sskel_name;
	
	/* Generate presentations for the server functions. */
	sskel->funcs.funcs_len = 0;
	sskel->funcs.funcs_val = 0;
	p_server_skel_internal(cur_aoi_idx, cur_aoi_idx, sskel);
	
	name = old_name;
	
	/*
	 * Finally, restore the `pg_state' AOI interface references that we
	 * changed previously.
	 */
	parent_interface_ref = saved_parent_ref;	  
	derived_interface_ref = saved_derived_ref;	  
}

void pg_state::p_server_skel_internal(aoi_ref this_ref,
				      aoi_ref derived_ref,
				      pres_c_server_skel *sskel)
{
	aoi_interface *this_interface, *derived_interface;
	
	aoi_ref saved_parent_ref, saved_derived_ref;
	
	u_int ops_len;
#ifdef ATTRIBUTE_STUBS
	u_int attribs_len;
#endif /* ATTRIBUTE_STUBS */
	u_int parents_len;
	
	u_int i;
	
	int flags;
	
	pres_c_server_func *sfuncs;
	u_int sfuncs_size;
	u_int sfuncs_count;
	
	/*
	 * Set `this_interface' and `derived_interface' to point at the AOI
	 * interfaces under consideration.
	 */
	assert((this_ref >= 0)
	       && (this_ref < ((aoi_ref) in_aoi->aoi_len)));
	assert(a(this_ref).binding
	       && (a(this_ref).binding->kind == AOI_INTERFACE));
	
	this_interface = &(a(this_ref).binding->aoi_type_u_u.interface_def);
	
	assert((derived_ref >= 0)
	       && (derived_ref < ((aoi_ref) in_aoi->aoi_len)));
	assert(a(derived_ref).binding
	       && (a(derived_ref).binding->kind == AOI_INTERFACE));
	
	derived_interface = &(a(derived_ref).binding->aoi_type_u_u.
			      interface_def);
	
	/*
	 * Set the `pg_state' interface reference data members to point at the
	 * interfaces under consideration so that `pg_state::calc_name' can
	 * compute names.
	 */
	saved_parent_ref = parent_interface_ref;
	saved_derived_ref = derived_interface_ref;
	
	parent_interface_ref = this_ref;
	derived_interface_ref = derived_ref;
	
	/* Get the lengths of arrays that are part of `this_interface'. */
	ops_len = this_interface->ops.ops_len;
#ifdef ATTRIBUTE_STUBS
	attribs_len = this_interface->attribs.attribs_len;
#endif
	parents_len = this_interface->parents.parents_len;
	
	/*
	 * Count the number of server functions that we might generate as part
	 * of `this_interface' and allocate a vector to hold the PRES_C stubs.
	 */
	sfuncs_size = ops_len;
#ifdef ATTRIBUTE_STUBS
	for (i = 0; i < attribs_len; i++)
		sfuncs_size +=
			((this_interface->attribs.attribs_val[i].readonly) ?
			 1 : /* Read-only attributes result in one stub. */
			 2); /* Read/write attributes result in two. */
#endif /* ATTRIBUTE_STUBS */
	
	sfuncs = (pres_c_server_func *)
		 mustmalloc(sizeof(pres_c_server_func) * sfuncs_size);
	sfuncs_count = 0;
	
	/*
	 * Generate the stubs for the operations defined by `this_interface',
	 * which are inherited by `derived_interface'.
	 */
	for (i = 0; i < ops_len; ++i) {
		aoi_operation *ao = &(this_interface->ops.ops_val[i]);
		
		/*
		 * We must check if we have already generated a stub for this
		 * (interface, operation) pair.  This may occur, for example,
		 * when `this_interface' is inherited through multiple paths
		 * to `derived_interface'.
		 */
		flags = p_interface_table_get_value(derived_interface, ao);
		
		if (!(flags & P_SERVER_SKEL_OP_MARK)) {
			p_interface_table_set_value(derived_interface, ao,
						    (flags
						     | P_SERVER_SKEL_OP_MARK));
			sfuncs[sfuncs_count++] =
				p_server_func(derived_interface, ao);
		}
	}
	
#ifdef ATTRIBUTE_STUBS
	/*
	 * Generate the stubs for the attributes defined by `this_interface',
	 * which are inherited by `derived_interface'.
	 */
	for (i = 0; i < attribs_len; ++i) {
		aoi_attribute *aa = &(this_interface->attribs.attribs_val[i]);
		
		/*
		 * As above, we must screen out attributes that we have already
		 * processed.
		 */
		flags = p_interface_table_get_value(derived_interface, aa);
		
		if (!(flags & P_SERVER_SKEL_OP_MARK)) {
			p_interface_table_set_value(derived_interface, aa,
						    (flags
						     | P_SERVER_SKEL_OP_MARK));
			sfuncs[sfuncs_count++] =
				p_server_attrib_func(derived_interface,
						     aa,
						     /* XXX: READ */);
			if (!(aa->readonly))
				sfuncs[sfuncs_count++] =
					p_server_attrib_func(derived_interface,
							     aa,
							     /* XXX: WRITE */);
		}
	}
#endif /* ATTRIBUTE_STUBS */
	
	/* Incorporate the server functions that we just made into `sskel'. */
	if (sfuncs_count > 0) {
		sskel->funcs.funcs_val =
			(pres_c_server_func *)
			mustrealloc(sskel->funcs.funcs_val,
				    (sizeof(pres_c_server_func)
				     * (sskel->funcs.funcs_len
					+ sfuncs_count)));
		
		for (i = 0; i < sfuncs_count; ++i)
			sskel->funcs.funcs_val[i + sskel->funcs.funcs_len] =
				sfuncs[i];
		
		sskel->funcs.funcs_len += sfuncs_count;
	}
	free(sfuncs);
	
	/*
	 * Generate the stubs for the operations and attributes that are
	 * inherited from parents of `this_interface'.
	 */
	for (i = 0; i < parents_len; ++i) {
		aoi_type parent_val = this_interface->parents.parents_val[i];
		aoi_ref  parent_ref;
		aoi_interface *parent_interface;
		
		/*
		 * All parent references must be through indirects so that
		 * we can find the name to go with the parent interface!
		 */
		assert(parent_val->kind == AOI_INDIRECT);
		
		parent_ref = aoi_deref_fwd(in_aoi, parent_val->aoi_type_u_u.indirect_ref);
		
		parent_interface = &(a(parent_ref).binding->aoi_type_u_u.
				     interface_def);
		
		/*
		 * As above, we check to see if we've already processed this
		 * parent.  (If we didn't do this check we would still generate
		 * correct code because all of the parent's operations and
		 * attributes are marked separately.  But doing this check can
		 * save us time.)
		 */
		flags = p_interface_table_get_value(derived_interface,
						    parent_interface);
		if (!(flags & P_SERVER_SKEL_PARENT_MARK)) {
			p_interface_table_set_value(
				derived_interface,
				parent_interface,
				(flags | P_SERVER_SKEL_PARENT_MARK));
			p_server_skel_internal(parent_ref,
					       derived_ref,
					       sskel);
		}
	}
	
	/*
	 * Finally, restore the `pg_state' AOI interface references that we
	 * changed previously.
	 */
	parent_interface_ref = saved_parent_ref;	  
	derived_interface_ref = saved_derived_ref;	  
}

/***** Auxiliary functions. *****/

/*
 * `p_server_skel_cdef' allocates and returns an index to the CAST declaration
 * of a server skeleton (a.k.a. a server dispatch function).
 */
cast_ref pg_state::p_server_skel_cdef(char *sskel_name)
{
	cast_ref  sskel_cref = cast_add_def(&(out_pres->cast));
	cast_def *sskel_cdef = &(out_pres->cast.cast_scope_val[sskel_cref]);
	
	sskel_cdef->name = sskel_name;
	sskel_cdef->sc = CAST_SC_EXTERN;
	sskel_cdef->u.kind = CAST_VAR_DECL;
	sskel_cdef->u.cast_def_u_u.var_type =
		cast_new_type_name(FLICK_SERVER_TYPE_NAME);
	
	return sskel_cref;
}

/* End of file. */

