/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/libaoi.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include <mom/c/pg_corba.hh>

void pg_corba::p_param_type(aoi_type at, int /*mr*/,
			    aoi_direction dir,
			    cast_type *out_ctype, pres_c_mapping *out_mapping)
{
	/*
	 * These variables determine how allocation/deallocation occurs for
	 * parameters passed by reference.  The flags are different between
	 * client and server.
	 *
	 * `in' parameters are passed by reference when they are non-atomic.
	 *
	 * `inout' and `out' parameters are passed by reference so that the
	 * callee can modify the parameter's value.  When an `out' parameter
	 * has a variable size, the callee cannot know in advance how large the
	 * returned object will be.  Therefore, in these cases, the data passed
	 * to the callee is actually a pointer to a pointer to the variable-
	 * sized type.  See `varobj_pointer_flags' below.
	 *
	 * `return' parameters are passed by reference only when they are
	 * variable-sized.  Because `varobj_pointer_flags' already describes
	 * how these references should be managed (for both `out' and `return'
	 * parameters), we don't need a separate `return_pointer_flags'.
	 */
	pres_c_alloc_flags in_pointer_flags;
	pres_c_alloc_flags inout_pointer_flags;
	pres_c_alloc_flags out_pointer_flags;
	/* pres_c_alloc_flags return_pointer_flags; */
	
	/*
	 * `varobj_pointer_flags' is the set of allocation flags used when a
	 * variable size object is to be passed by reference so that the callee
	 * can allocate/reallocate the memory for the object.
	 */
	pres_c_alloc_flags varobj_pointer_flags;
	
	/*
	 * `t' is the ``concrete'' AOI type referenced by `at'.  If `at' is an
	 * `AOI_INDIRECT', `t' is the type referenced through the indirection.
	 */
	aoi_type t;
	
	/*
	 * `min' and `max' are the subscript limits of an array type.
	 *
	 * `is_string', `is_sequence', `is_array', and `is_object' are flags
	 * indicating that the AOI type at hand is being presented as a CORBA
	 * string, sequence, array, or object reference, respectively.
	 *
	 * `is_variable' is a flag indicating that the AOI type at hand has a
	 * variable-size C presentation.  See Section 14.8 of the CORBA 2.0
	 * spec.
	 */
	unsigned int min, max;
	int is_string, is_sequence, is_array, is_object;
	int is_variable;
	
	/*********************************************************************/
	
	if (gen_client) {
		in_pointer_flags     = (PRES_C_ALLOC_NEVER |
					PRES_C_DEALLOC_NEVER);
		inout_pointer_flags  = (PRES_C_ALLOC_NEVER |
					PRES_C_DEALLOC_NEVER);
		out_pointer_flags    = (PRES_C_ALLOC_NEVER |
					PRES_C_DEALLOC_NEVER);
		varobj_pointer_flags = (PRES_C_ALLOC_ALWAYS |
					PRES_C_DEALLOC_NEVER);
		
	} else if (gen_server) {
		in_pointer_flags     = (PRES_C_ALLOC_ALWAYS |
					PRES_C_DEALLOC_ALWAYS);
		inout_pointer_flags  = (PRES_C_ALLOC_ALWAYS |
					PRES_C_DEALLOC_ALWAYS);
		out_pointer_flags    = (PRES_C_ALLOC_ALWAYS |
					PRES_C_DEALLOC_ALWAYS);
		varobj_pointer_flags = (PRES_C_ALLOC_NEVER |
					PRES_C_DEALLOC_ALWAYS);
		
	} else
		panic("In `pg_corba::p_param_type', "
		      "generating neither client nor server!");
	
	/*
	 * Until recently there was a great deal of code here for handling
	 * sequence type declarations within operation parameter lists.  That
	 * syntax was allowed by Sun's CORBA front end but isn't actually
	 * allowed by the CORBA 2.0 IDL grammar.
	 *
	 * Because the sequence-declaration-within-parameter-list code is long
	 * and now very out-of-date (and therefore potentially confusing), I
	 * decided to remove the code rather than leave it in as ``dead code.''
	 * If you need to review the code that was here, check out an old
	 * (pre-1997) copy of this file.
	 */
	if (at->kind == AOI_ARRAY)
		/*
		 * Just to be safe, if we have a direct AOI reference to an
		 * array type, assert that it must be a string.  Now that we
		 * don't allow sequence-declaration-within-parameter-list
		 * syntax, all references to non-string arrays should be
		 * through `AOI_INDIRECT's (to typedef'ed types).
		 *
		 * Direct references to string types are possible because
		 * `string<N> foo' is legal syntax for a CORBA operation
		 * parameter.
		 */
		assert(at->aoi_type_u_u.array_def.flgs
		       & AOI_ARRAY_FLAG_NULL_TERMINATED_STRING);
	
	/*
	 * Get the basic presentation for this type: the `out_ctype' and the
	 * `out_mapping'.
	 */
	p_type(at, out_ctype, out_mapping);
	
	/*
	 * Get the actual AOI type (not an AOI_INDIRECT type).  If the actual
	 * AOI type is an AOI_ARRAY, determine if it corresponds to a CORBA
	 * string, sequence, or array.
	 */
	t = at;
	while (t->kind == AOI_INDIRECT)
		t = in_aoi->aoi_val[t->aoi_type_u_u.indirect_ref].binding;
	
	if (t->kind == AOI_ARRAY) {
		aoi_get_array_len(in_aoi, &(t->aoi_type_u_u.array_def),
				  &min, &max);
		
		is_string   = (t->aoi_type_u_u.array_def.flgs
			       == AOI_ARRAY_FLAG_NULL_TERMINATED_STRING);
		is_sequence = ((min != max) && !is_string);
		is_array    = ((min == max) && !is_string);
	} else {
		min = 0;
		max = 0;
		is_string = 0;
		is_sequence = 0;
		is_array = 0;
	}
	if ((t->kind == AOI_INTERFACE) || (t->kind == AOI_FWD_INTRFC))
		is_object = 1;
	else
		is_object = 0;
	
	is_variable = isVariable(at);
	
	/*
	 * Modify the presentation of a few specific, special types: strings,
	 * arrays, and variable-size data.
	 */
	if (is_string) {
		/*
		 * Strings are presented as `char *'s by the CORBA version of
		 * `p_variable_array_type'.  Essentially, this means that
		 * strings are presented as atomic data: the pointer fits in a
		 * word.  The CORBA rules for string parameters (in section
		 * 14.19 of the CORBA 2.0 spec) make sense if you think of
		 * `char *' as an atomic object like an `int'.
		 *
		 * Nevertheless, the non-atomicity of strings often shows
		 * through.  Here we ensure that the data buffer for the string
		 * will be allocated/deallocated at the right times and with
		 * the right allocator.
		 */
		if ((*out_mapping)->kind != PRES_C_MAPPING_TERMINATED_ARRAY) {
			/*
			 * XXX --- We can't modify the presentation of a
			 * typedef'ed type (e.g., `typedef string<256> str')
			 * because the allocator information is hidden away
			 * within the type's marshal and unmarshal stubs.
			 * Gack!
			 */
			warn("In `pg_corba::p_param_type', "
			     "can't modify presentation of string type.");
		} else {
			/*
			 * XXX --- It is a bad idea to munge the PRES_C
			 * returned by `p_type'.  We should peel off the
			 * `TERMINATED_ARRAY' mapping and make a new one.
			 */
			pres_c_allocation *alloc = &((*out_mapping)->
						     pres_c_mapping_u_u.
						     terminated_array.alloc);
			
			switch (dir) {
			case AOI_DIR_IN:
				alloc->flags = in_pointer_flags;
				alloc->allocator = "auto";
				break;
			case AOI_DIR_INOUT:
				/*
				 * We don't use `inout_pointer_flags' here
				 * because we must handle the case in which the
				 * returned string has a different length.
				 * Here the non-atomicity of strings is
				 * revealed.
				 *
				 * XXX --- We should use fancier flags like
				 * (PRES_C_REALLOC_IF_TOO_BIG | PRES_C_REALLOC_
				 * IF_TOO_SMALL | PRES_C_DEALLOC_NEVER), but
				 * the ``realloc'' options aren't implemented
				 * by the back end.  So we do the obvious...
				 */
				alloc->flags = (PRES_C_ALLOC_ALWAYS |
						PRES_C_DEALLOC_ALWAYS);
				alloc->allocator = p_get_allocator();
				break;
			case AOI_DIR_OUT:
				/*
				 * We don't use `out_pointer_flags' here
				 * because the string data was allocated by the
				 * callee.  The flags for handling variable-
				 * size, callee-allocated data are in
				 * `varobj_pointer_flags'.
				 */
				alloc->flags = varobj_pointer_flags;
				alloc->allocator = p_get_allocator();
				break;
			case AOI_DIR_RET:
				/*
				 * Again, the string buffer will have been
				 * allocated by the callee, so we use
				 * `varobj_pointer_flags'.
				 */
				alloc->flags = varobj_pointer_flags;
				alloc->allocator = p_get_allocator();
				break;
			}
		}
		
	} else if (is_array) {
		/*
		 * CORBA arrays are generally presented a C arrays, and C
		 * arrays are automatically passed by reference when they are
		 * passed to functions.  They are therefore like strings in
		 * that they are ``semi-atomic'': the address of the array is
		 * always passed.
		 *
		 * The principal difference is that, since the length of an
		 * array cannot change, the presentation of `inout' and `out'
		 * can be somewhat simpler.  (See section 14.19 of the CORBA
		 * 2.0 spec.  An `inout' string is a `char **' so that the
		 * string can be resized.  An `inout' array is simply `array':
		 * automatically passed by reference, and no extra level of
		 * indirection is necessary.)
		 */
		switch (dir) {
		case AOI_DIR_IN:
			break;
		case AOI_DIR_INOUT:
			break;
		case AOI_DIR_OUT:
			if (!is_variable)
				break;
			/* FALLTHROUGH */
		case AOI_DIR_RET:
			/*
			 * The presentation of a `return' array is a pointer
			 * to an array slice.  The presentation of an `out'
			 * array of variable-size elements is a pointer to a
			 * pointer to an array slice.  See section 14.19 of the
			 * CORBA 2.0 specification.
			 *
			 * XXX --- We should have a better method for computing
			 * the name of the array slice type.
			 */
			assert((*out_ctype)->kind == CAST_TYPE_NAME);
			
			*out_ctype
				= cast_new_type_name(
					flick_asprintf("%s_slice",
						       ((*out_ctype)->
							cast_type_u_u.name)
						));
			*out_ctype
				= cast_new_pointer_type(*out_ctype);
			/*
			 * The additional pointer for `out' arrays is added by
			 * subsequent code in this function.
			 */
			break;
		}
		
	} else if (is_variable && !is_object) {
		/*
		 * When a variable-size type is used as an `out' or `return'
		 * value, the callee is responsible for allocating the storage
		 * for the returned data (because the caller cannot do so).
		 * The address of that storage will be communicated to the
		 * caller --- either as a `return' value, or as the value of
		 * an `out' parameter.
		 *
		 * Object references are an exception to this rule.  Although
		 * they are listed as variable-length types in Section 14.8 of
		 * the CORBA 2.0 spec, they are presented as if they were
		 * fixed-length types when used as parameters (according to the
		 * rules shown in Section 14.19).  This is because object
		 * references are essentially pointers, and can therefore be
		 * returned ``by value.''  Or, in other words, we don't need to
		 * *add* a pointer because an object reference already *is* a
		 * pointer.
		 *
		 * The code here handles variable-size structs, variable-size
		 * unions, sequences, and anys.  The special presentations for
		 * variable-size strings and arrays are handled previously.
		 */
		switch (dir) {
		case AOI_DIR_IN:
			break;
		case AOI_DIR_INOUT:
			/*
			 * When a variable-sized datum is `inout', the caller
			 * allocates the ``root'' of the variable-size object
			 * (e.g., the C structure that represents a sequence).
			 * The callee can then change the variable-size objects
			 * contained *within* the `inout' parameter (e.g.,
			 * change the buffer pointer wihin the sequence
			 * structure) but cannot reallocate the root `inout'
			 * parameter.  This is why `inout' variable-sized
			 * structs, variable-size unions, sequences, and CORBA
			 * anys are presented with one fewer levels of
			 * indirection that their `out' counterparts.
			 *
			 * Since the caller, not the callee, allocates storage
			 * for the `inout' parameter, we don't need to insert
			 * any levels of indirection here.  (Later on, we will
			 * add the one level of indirection that is required
			 * for all `inout' parameters.)
			 */
			break;
		case AOI_DIR_OUT:
			pres_c_interpose_pointer(out_ctype, out_mapping,
						 varobj_pointer_flags,
						 p_get_allocator());
			break;
		case AOI_DIR_RET:
			pres_c_interpose_pointer(out_ctype, out_mapping,
						 varobj_pointer_flags,
						 p_get_allocator());
			break;
		}
	}
	
	/*
	 * Now, modify the presentation of the parameter based (principally) on
	 * its direction: `in', `inout', `out', or `return'.
	 */
	switch (dir) {
	case AOI_DIR_IN:
		/*
		 * Nonatomic data (including most variable-length data) are
		 * passed by reference; the data itself will be held in
		 * auto-allocated (stack) storage.  Atomic data are simply
		 * passed by value.
		 *
		 * This distiction has three caveats:
		 *
		 * First, strings are considered ``atomic'' because they are
		 * basically presented as `char *'s, which may be passed by
		 * value.  (The CORBA rules for strings make sense if you think
		 * of `char *' as an atomic data type.)
		 *
		 * Second, arrays are considered atomic for the same reason:
		 * they are fundamentally presented as C pointers, which may be
		 * passed by value.
		 *
		 * CORBA sequences are *not* atomic because they are presented
		 * as C structures.
		 *
		 * Finally, object references are considered to be atomic,
		 * because they are essentially (opaque) pointers.
		 */
		if ((t->kind == AOI_STRUCT)
		    || (t->kind == AOI_UNION)
		    || is_sequence
		    || (t->kind == AOI_ANY)
			)
			pres_c_interpose_pointer(out_ctype, out_mapping,
						 in_pointer_flags,
						 "auto");
		break;
		
	case AOI_DIR_INOUT:
		/*
		 * All `inout' data (except CORBA arrays) are passed by
		 * reference; the data itself is contained in auto-allocated
		 * storage.  The contents of that storage may be modified by
		 * the callee.
		 *
		 * This level of indirection is on top of any levels provided
		 * for the purpose of allowing the callee to reallocate storage
		 * for the `inout' parameter.
		 *
		 * CORBA arrays are handled specially.  Because CORBA arrays
		 * are presented as C arrays, they are automatically passed by
		 * reference and we therefore don't need any extra level of
		 * indirection.
		 */
		if (is_array)
			/* Do nothing. */
			;
		else
			pres_c_interpose_pointer(out_ctype, out_mapping,
						 inout_pointer_flags,
						 "auto");
		break;
		
	case AOI_DIR_OUT:
		/*
		 * All `out' data (except CORBA arrays of fixed-size elements)
		 * are passed by reference to auto-allocated storage.  The
		 * contents of that storage is expected to be set by the
		 * callee.
		 *
		 * Note that if the object is variable-length, then the auto
		 * allocated storage will consist of an uninit'ed pointer which
		 * the callee will set to point to the returned value.  This
		 * presentation was taken care of previously.
		 */
		if (is_array && !is_variable)
			/* Do nothing. */
			;
		else
			pres_c_interpose_pointer(out_ctype, out_mapping,
						 out_pointer_flags,
						 "auto");
		break;
		
	case AOI_DIR_RET:
		/*
		 * Nothing additional is required.  Most fixed-size data are
		 * returned by value.  Variable-size data are returned by
		 * reference, and the required level of indirection was added
		 * previously.
		 */
		break;
	}
	
	/*
	 * Finally, wrap the mapping in a `hint' that tells the back end what
	 * kind of parameter this is: `in', `out', etc.
	 */
	pres_c_interpose_direction(out_mapping, dir);
}

/* End of file. */

