/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>
#include <string.h>

#include <mom/compiler.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <mom/libaoi.h>
#include <mom/Strings.h>

#include <mom/c/pg_corba.hh>

const Strings SEQUENCE_NAME_INITIAL_PREFIX("CORBA_sequence_");
const Strings SEQUENCE_NAME("sequence_");
const Strings LENGTH("_length");
const Strings MAXIMUM("_maximum");
const Strings BUFFER("_buffer");

/*
 * First, an auxiliary function to compute the name of a CORBA sequence type.
 */
Strings pg_corba::sequence_type_string(aoi_type at, aoi *in_aoi, char *name)
{
	Strings s;
	int bits, is_signed;
	aoi_type target_at;
	
	/* Return a string corresponding to the argument aoi_type. */
	switch (at->kind) {
	case AOI_INDIRECT:
		/*
		 * Section 17.11 of the CORBA 2.1 specification: ``The type
		 * name used in the C mapping is the type name of the effective
		 * type...'' meaning that one digs down to the base type.
		 */
		target_at = aoi_indir_1(in_aoi, at);
		while (target_at->kind == AOI_INDIRECT) {
			at = target_at;
			target_at = aoi_indir_1(in_aoi, at);
		}
		switch (target_at->kind) {
		case AOI_STRUCT:
		case AOI_UNION:
			s = calc_name_from_ref(at->aoi_type_u_u.indirect_ref);
			break;
			
		default:
			s = sequence_type_string(target_at, in_aoi, name);
			break;
		}
		break;
		
	case AOI_INTEGER:
	case AOI_SCALAR:
		if (at->kind == AOI_INTEGER)
			aoi_get_int_size(&(at->aoi_type_u_u.integer_def),
					 &bits, &is_signed);
		else {
			bits = at->aoi_type_u_u.scalar_def.bits;
			is_signed = !(at->aoi_type_u_u.scalar_def.flags
				      & AOI_SCALAR_FLAG_UNSIGNED);
		}
		
		switch (bits) {
		case 1:
			assert(!is_signed);
			s = "boolean";
			break;
		case 8:
			assert(!is_signed);
			s = "octet";
			break;
		case 16:
			s = (is_signed ? "short" : "unsigned_short");
			break;
		case 32:
			s = (is_signed ? "long" : "unsigned_long");
			break;
		case 64:
			s = (is_signed ? "long_long" : "unsigned_long_long");
			break;
		default:
			panic("In `pg_corba::sequence_type_string', "
			      "unrecognized number of integer bits: %d.",
			      bits);
			break;
		}
		break;
		
	case AOI_ENUM:
		s = "enum";
		break;
		
	case AOI_FLOAT:
		bits = at->aoi_type_u_u.float_def.bits;
		switch (bits) {
		case 32:
			s = "float";
			break;
		case 64:
			s = "double";
			break;
		default:
			panic("In `pg_corba::sequence_type_string', "
			      "unrecognized number of float bits: %d.",
			      bits);
			break;
		}
		break;
		
	case AOI_CHAR:
		s = "char";
		break;
		
	case AOI_ARRAY:
		/* Recognize string types. */
		if (at->aoi_type_u_u.array_def.element_type->kind == AOI_CHAR)
			s = "string";
		else
			s = SEQUENCE_NAME
			    + sequence_type_string((at->aoi_type_u_u.array_def.
						    element_type),
						   in_aoi,
						   name);
		break;
		
	case AOI_ANY:
		s = "any";
		break;
		
	case AOI_INTERFACE:
	case AOI_FWD_INTRFC:
		s = "Object";
		break;
		
	case AOI_STRUCT:
	case AOI_UNION:
	case AOI_EXCEPTION:
	case AOI_VOID:
	case AOI_CONST:
	case AOI_NAMESPACE:
	case AOI_OPTIONAL:
		panic("In `pg_corba::sequence_type_string', "
		      "invalid AOI type kind %d.",
		      at->kind);
		break;
		
	default:
		panic("In `pg_corba::sequence_type_string', "
		      "unknown AOI type kind %d.",
		      at->kind);
		break;
	}
	
	return s;
}

/*
 * Here is the CORBA-specific version of `pg_state::p_variable_array_type'.
 */
void pg_corba::p_variable_array_type(aoi_array *array,
				     cast_type *out_ctype,
				     pres_c_mapping *out_map)
{
	cast_type ctype, element_ctype;
	pres_c_mapping map, element_map;
	
	if (array->flgs & AOI_ARRAY_FLAG_NULL_TERMINATED_STRING) {
		/*
		 * We are mapping a null-terminated array --- most likely, a
		 * string.  The CORBA mapping for a null-terminated array is
		 * straightforward.
		 */
		
		/* Create the ctype and mapping for the array element type. */
		p_type(array->element_type, &element_ctype, &element_map);
		
		ctype = cast_new_pointer_type(element_ctype);
		
		/* Create the map. */
		map = pres_c_new_mapping(PRES_C_MAPPING_TERMINATED_ARRAY);
		/*
		 * The terminator is the literal value of the terminator,
		 * not a reference to a literal value.
		 */
		map->pres_c_mapping_u_u.terminated_array.terminator = 0;
		map->pres_c_mapping_u_u.terminated_array.max = 0;
		map->pres_c_mapping_u_u.terminated_array.element_mapping
			= element_map;
		if (gen_client)
			map->pres_c_mapping_u_u.terminated_array.alloc.flags 
				= (PRES_C_ALLOC_ALWAYS
				   | PRES_C_DEALLOC_NEVER);
		else if (gen_server)
			map->pres_c_mapping_u_u.terminated_array.alloc.flags 
				= (PRES_C_ALLOC_ALWAYS
				   | PRES_C_DEALLOC_ALWAYS);
		else
			panic("In `pg_corba::p_variable_array_type', "
			      "generating neither client nor server.");
		
		map->pres_c_mapping_u_u.terminated_array.alloc.allocator 
			= p_get_allocator();
		
	} else {
		/*
		 * We are mapping a counted array.  This maps onto a structure
		 * with `_maximum', `_length', and `_buffer' fields (i.e., a
		 * CORBA sequence type).
		 */
		aoi_type at, target_at;
		
		cast_struct_type *array_struct;
		cast_type max_ctype, len_ctype;
		pres_c_mapping max_map, len_map;
		char *max_name, *len_name, *val_name;
		
		pres_c_inline the_inline;
		pres_c_inline_counted_array *inline_array;
		
		/*
		 * Create the ctype and mapping for the array element type.
		 *
		 * Section 17.11 or the CORBA 2.1 spec says that the type of
		 * the sequence buffer elements is the ``type name of the
		 * effective type,'' so we have to dig down through the AOI.
		 */
		at = array->element_type;
		target_at = aoi_indir_1(in_aoi, at);
		while (target_at->kind == AOI_INDIRECT) {
			at = target_at;
			target_at = aoi_indir_1(in_aoi, at);
		}
		switch (target_at->kind) {
		case AOI_STRUCT:
		case AOI_UNION:
			/*
			 * Use the struct/union name (e.g., ``struct foo'') and
			 * not the raw, unamed struct/union definition (e.g.,
			 * ``struct { ... }''.
			 */
			p_type(at, &element_ctype, &element_map);
			break;
			
		default:
			/* Use the actual ``base'' type. */
			p_type(target_at, &element_ctype, &element_map);
			break;
		}
		
		/* Create the structure. */
		ctype = cast_new_struct_type(3);
		array_struct = &(ctype->cast_type_u_u.struct_type);
		array_struct->name
			= (char *) (SEQUENCE_NAME_INITIAL_PREFIX + 
				    sequence_type_string(array->element_type,
							 in_aoi,
							 name)
				);
		
		/*
		 * Determine the names of the structure fields.
		 */
		max_name = (char *) MAXIMUM;
		len_name = (char *) LENGTH;
		val_name = (char *) BUFFER;
		
		/* Now fill in the structure field definitions. */
		p_variable_array_maximum_type(array, &max_ctype, &max_map);
		p_variable_array_length_type(array, &len_ctype, &len_map);
		
		array_struct->slots.slots_val[0].name = max_name;
		array_struct->slots.slots_val[0].type = max_ctype;
		array_struct->slots.slots_val[1].name = len_name;
		array_struct->slots.slots_val[1].type = len_ctype;
		array_struct->slots.slots_val[2].name = val_name;
		array_struct->slots.slots_val[2].type
			= cast_new_pointer_type(element_ctype);
		
		/* Create the counted-array mapping. */
		the_inline = pres_c_new_inline(PRES_C_INLINE_COUNTED_ARRAY);
		inline_array = &(the_inline->pres_c_inline_u_u.counted_array);
		inline_array->max = pres_c_new_inline_atom(0, max_map);
		inline_array->len = pres_c_new_inline_atom(1, len_map);
		inline_array->ptr.index = 2;
		
		/*
		 * Interpose a pointer.  As stated in `pres_c.x', there must be
		 * at least one level of PRES_C_MAPPING_POINTER here.
		 */
		if (gen_client)
			/*
			 * XXX --- These pointer flags are *wrong* for `inout'
			 * sequences.  Currently, the client is expected to
			 * free the `in' version and allocate the `out' version
			 * --- it won't reuse space.
			 */
			pres_c_interpose_pointer(&element_ctype,
						 &element_map,
						 (PRES_C_ALLOC_ALWAYS |
						  PRES_C_DEALLOC_NEVER),
						 p_get_allocator());
		else if (gen_server)
			pres_c_interpose_pointer(&element_ctype,
						 &element_map,
						 (PRES_C_ALLOC_ALWAYS |
						  PRES_C_DEALLOC_ALWAYS),
						 p_get_allocator());
		else
			panic("In `pg_corba::p_variable_array_type', "
			      "generating neither client nor server.");
		
		inline_array->ptr.mapping = element_map;
		/* There are zero additional levels of pointer indirection. */
		inline_array->ptr_levels = 0;
		
		/* Finally, create the map. */
		map = pres_c_new_mapping(PRES_C_MAPPING_STRUCT);
		map->pres_c_mapping_u_u.struct_i = the_inline;
		
		/* Emit a `typedef' for this sequence type. */
		int cdef = cast_add_def(&(out_pres->cast));
		c(cdef).name = array_struct->name;
		c(cdef).sc = CAST_SC_NONE;
		c(cdef).u.kind = CAST_TYPEDEF;
		c(cdef).u.cast_def_u_u.typedef_type = ctype;
		
		ctype = cast_new_type(CAST_TYPE_STRUCT_NAME);
		ctype->cast_type_u_u.struct_name = array_struct->name;
	}
	
	*out_ctype = ctype;
	*out_map = map;
}

/* End of file. */

