/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <assert.h>

#include <mom/compiler.h>
#include <mom/libaoi.h>
#include <mom/c/pfe.hh>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include "pg_sun.hh"

/*
 * This method determines how `p_param' processes a parameter type.
 */
void pg_sun::p_param_type(aoi_type at, mint_ref /* mr */, aoi_direction dir,
			  cast_type *out_ctype, pres_c_mapping *out_mapping)
{
	pres_c_alloc_flags in_pointer_flags;
	
	/*
	 * `real_at' is the ``concrete'' AOI type referenced by `at'.  If `at'
	 * is an `AOI_INDIRECT', `real_at' is the type referenced through the
	 * indirection.
	 */
	aoi_type real_at;
	
	/*
	 * `is_string' and `is_array' are flags indicating that the AOI type at
	 * hand is being presented as an C string or array, respectively.
	 */
	int is_string, is_array;
	
	/*********************************************************************/
	
	/*
	 * The library verion of this method checks for `inout' and `out'
	 * parameters at this point.  However, Sun RPC doesn't allow for these
	 * parameter types.
	 */
	assert(dir == AOI_DIR_IN);
	
	if (gen_client)
		/*
		 * An ONC RPC client stub accepts a pointer to the type
		 * specified in the IDL file.  The client never allocates or
		 * deallocates this pointer.
		 */
		in_pointer_flags = (PRES_C_ALLOC_NEVER
				    | PRES_C_DEALLOC_NEVER);
	else if (gen_server)
		/*
		 * Similarly, an ONC RPC server work function receives a
		 * pointer to the type.  This pointer must be allocated and
		 * deallocated by the server skeleton.
		 */
		in_pointer_flags = (PRES_C_ALLOC_ALWAYS
				    | PRES_C_DEALLOC_ALWAYS);
	else
		panic("In `pg_sun::p_param_type', "
		      "generating neither client nor server.");
	
	/*
	 * Get the basic presentation for this type: the `out_ctype' and the
	 * `out_mapping'.
	 */
	p_type(at, out_ctype, out_mapping);
	
	/*
	 * Get the actual AOI type (not an AOI_INDIRECT type).  If the actual
	 * AOI type is an AOI_ARRAY, determine if it corresponds to an ONC RPC
	 * string or fixed-length array.
	 */
	real_at = at;
	while (real_at->kind == AOI_INDIRECT)
		real_at = in_aoi->
			  aoi_val[real_at->aoi_type_u_u.indirect_ref].binding;
	
	if (real_at->kind == AOI_ARRAY) {
		unsigned int min, max;
		
		aoi_get_array_len(in_aoi, &(real_at->aoi_type_u_u.array_def),
				  &min, &max);
		
		is_string   = (real_at->aoi_type_u_u.array_def.flgs
			       == AOI_ARRAY_FLAG_NULL_TERMINATED_STRING);
		is_array    = ((min == max) && !is_string);
	} else {
		is_string = 0;
		is_array = 0;
	}
	
	/*
	 * Now determine the C type of this parameter.
	 */
	if (is_array)
		/*
		 * Do nothing.  IDL arrays are presented as C arrays, which are
		 * automatically passed by reference.
		 */
		;
	else
		/*
		 * Otherwise, interpose a pointer so that the parameter will be
		 * passed by reference.
		 *
		 * I think it is surprising that `rpcgen' doesn't omit this
		 * ``extra'' pointer before string types (which are presented
		 * as `char *'s) and optional types (which are presented as
		 * pointers).
		 */
		pres_c_interpose_pointer(out_ctype,
					 out_mapping,
					 in_pointer_flags,
					 "auto");
	
	/*
	 * Finally, wrap the mapping in a `hint' that tells the back end what
	 * kind of parameter this is: `in', `out', etc.
	 */
	pres_c_interpose_direction(out_mapping, dir);
}

/* End of file. */

