/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

/* General Notes: The presentation structure creates linkages between
 * the MINT and CAST structures. Presentations specify both mapping and 
 * linkage types. 
 *
 * Note, some presentations are merely used to walk down a structure
 * For example, a PRES_C_INLINE_COLLAPSED_UNION presentation moves down
 * to a specific case in a union.
 */

#include <stdio.h>
#include <rpc/types.h>

/* #include "cpu.h" */
#include "message.h"
#include "type.h"
#include "routine.h"
#include "statement.h"
#include "mom_routines.h"
#include "global.h"
#include "assert.h"
#include "boolean.h"
#include "xlate_util.h"

#include <mom/types.h>
#include <mom/compiler.h>
#include <mom/mint_special.h>
#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>
#include <mom/idl_id.h>

#define NONE 0x0
#define COUNT_ARR 0x1
#define NOIN_OUT 0x2

/* Macro to define cast scope */
#define c_scope out_pres_c.cast

/* Macro to define stubs location */
#define c_stubs out_pres_c.stubs

/* Quickie lookup macro into the mint structure. */
#define m(n) (out_pres_c.mint.defs.defs_val[n])

/* Lookup to get at func def */
#define c(n) (c_scope.cast_scope_val[n].u.cast_def_u_u.func_type)

/* Lookup into stubs array */
#define s(n) (c_stubs.stubs_val[n])

/* Useful macro in generating routine presentations. */
#define sl(n) (pres_in_2->pres_c_inline_u_u.struct_i.slots.slots_val[n])


/* Useful macros for deciphering an Argument */

#define CurTypeNameUsed(CurType, which_stub, which_mess)		\
	(((which_mess == akIn && which_stub == PRES_C_CLIENT_STUB)	\
	  || (which_mess == akOut && which_stub == PRES_C_SERVER_SKEL))	\
	 ? CurType->itInName : CurType->itOutName)
	
	
#define IsArg(arg, which_stub)					\
	(akCheckAll(arg->argKind,				\
		    which_stub == PRES_C_CLIENT_STUB		\
		    ? akbUserArg				\
		    : akbServerArg))
	
#define IsArgIn(arg, which_stub)					      \
	((akCheckAll(arg->argKind,					      \
		     akAddFeature(akInTest,				      \
				  which_stub == PRES_C_CLIENT_STUB	      \
				  ? akAddFeature(akbUserArg, akbSendSnd)      \
				  : akAddFeature(akbServerArg, akbSendRcv)))) \
	 || (((akIdent(arg->argKind) == akeMsgSeqno)			      \
	      || (akIdent(arg->argKind) == akeWaitTime)			      \
	      || (akIdent(arg->argKind) == akeMsgOption)		      \
	      || (akIdent(arg->argKind) == akeRequestPort))		      \
	     && (IsArg(arg, which_stub))))
	
#define IsArgOut(arg, which_stub)					     \
	(akCheckAll(arg->argKind,					     \
		    akAddFeature(akOutTest,				     \
				 which_stub == PRES_C_CLIENT_STUB	     \
				 ? akAddFeature(akbUserArg, akbReturnRcv)    \
				 : akAddFeature(akbServerArg, akbReturnSnd))))
	
#define IsArgInOut(arg, which_stub)					   \
	(akCheckAll(arg->argKind,					   \
		    akAddFeature(akAddFeature(akInTest, akOutTest),	   \
				 which_stub == PRES_C_CLIENT_STUB	   \
				 ? akAddFeature(akbUserArg,		   \
						akAddFeature(akbReturnRcv, \
							   akbSendSnd))	   \
				 : akAddFeature(akbServerArg,		   \
						akAddFeature(akbReturnSnd, \
							   akbSendRcv)))))
	
	
/* Variable to store presentation stuff */
extern pres_c_1 out_pres_c;

static const char * get_ctype_name(ipc_type_t *Type, int which_stub)
{
	const char *ret;
	if (which_stub == PRES_C_CLIENT_STUB)
		ret = Type->itUserType;
	else
	        if (Type->itTransType) 
			ret = Type->itTransType;
		else
			ret = Type->itServerType;
	if (ret == NULL) return "";
	return ret;
}


static void interpose_direction_mapping(pres_c_mapping *map,
					argument_t *arg,
					int which_stub)
{
	/*
	 * Wrap the given mapping within a special ``direction'' mapping that
	 * communicates the argument direction to the back end.  This is a bit
	 * of hackery that should be eliminated someday.
	 * These tests should be done in a specific order for identification
	 */
	if      (IsArgInOut(arg, which_stub))
		pres_c_interpose_direction(map, AOI_DIR_INOUT);
	else if (akCheckAll(arg->argKind, akReturn))
		pres_c_interpose_direction(map, AOI_DIR_RET);
	else if (IsArgIn(arg, which_stub))
		pres_c_interpose_direction(map, AOI_DIR_IN);
	else if (IsArgOut(arg, which_stub))
		pres_c_interpose_direction(map, AOI_DIR_OUT);
}

/* add the levels of pointer indirection that may be necessary,
   and convert it to an inline. */
static pres_c_inline interpose_inline_indirect(cast_type *ctypep,
					       pres_c_mapping *mapp,
					       argument_t *CurArg,
					       int cparam, int extern_indir,
					       int which_stub)
{
        /* If the external ctype will be pointer-indirected,
           then pointer-indirect the internal ctype correspondingly.  */

        if (extern_indir) {
		if (which_stub == PRES_C_CLIENT_STUB)
			pres_c_interpose_pointer(ctypep, mapp,
						 (PRES_C_ALLOC_NEVER
						  | PRES_C_DEALLOC_NEVER),
						 "mig");
		else if (which_stub == PRES_C_SERVER_SKEL)
			pres_c_interpose_pointer(ctypep, mapp,
						 (PRES_C_ALLOC_ALWAYS
						  | PRES_C_DEALLOC_ALWAYS),
						 "auto");
		else
			panic("In `interpose_inline_indirect', "
			      "generating neither client stub "
			      "nor server skeleton!");
	}

	/* Wrap the `map' in a direction mapping. */
	interpose_direction_mapping(mapp, CurArg, which_stub);
	
	/* Map this argument to a function parameter with a
	   pres_c_inline_atom.  */
	return pres_c_new_inline_atom(cparam, *mapp);
}

/* Make a direct, simple atom argument. Makes the MINT, CAST, and mapping all
   at once. */
static void make_simple_arg(argument_t *CurArg, int which_stub, int which_mess,
			    mint_ref *out_itype, cast_type *out_ctype,
			    pres_c_mapping *out_map)
{
    ipc_type_t *CurType = CurArg->argType; 
    ipc_type_t *ct;

    mint_ref itype;
    cast_type ctype;
    pres_c_mapping map;

    int ctype_flags = CAST_PRIM_INT; /* What C type */
    int cmod_flags = 0; /* signed, unsigned, long, short... */
    
    if (akIdent(CurArg->argKind) == akeMsgSeqno)
	    itype = out_pres_c.mint.standard_refs.signed32_ref;
    else 
    {
	    switch (CurTypeNameUsed(CurType, which_stub, which_mess))
	    {
	    case MACH_MSG_TYPE_BOOLEAN:
		    itype = out_pres_c.mint.standard_refs.bool_ref;
		    break;
	    case MACH_MSG_TYPE_INTEGER_64:
		    panic("64 bit integers not supported.");
		    break;
	    case MACH_MSG_TYPE_INTEGER_32:
		    itype = out_pres_c.mint.standard_refs.signed32_ref;
		    break;
	    case MACH_MSG_TYPE_INTEGER_16:
		    itype = out_pres_c.mint.standard_refs.signed16_ref;
		    cmod_flags = CAST_MOD_SHORT;
		    break;
	    case MACH_MSG_TYPE_INTEGER_8:
		    itype = out_pres_c.mint.standard_refs.signed8_ref;
		    ctype_flags = CAST_PRIM_CHAR;
		    break;
	    case MACH_MSG_TYPE_REAL:
		    switch (CurType->itSize) {
		    case 32:
			    itype = out_pres_c.mint.standard_refs.float32_ref;
			    break;
		    case 64:
			    itype = out_pres_c.mint.standard_refs.float64_ref;
			    break;
		    default:
			    panic("make_simple_arg: unsupported floating-"
				  "point size (%d-bit)", CurType->itSize);
		    }
		    ctype_flags = CAST_PRIM_FLOAT;
		    break;
	    case MACH_MSG_TYPE_CHAR:
		    itype = out_pres_c.mint.standard_refs.char8_ref;
		    ctype_flags = CAST_PRIM_CHAR;
		    break;
	    default:
		    panic("make_simple_arg: unknown primitive type %d", 
			  CurTypeNameUsed(CurType, which_stub, which_mess));
	    }
    }
   
    /* Make the internal ctype. */

    /* find the base element type */
    for (ct = CurType; ct->itElement; ct = ct->itElement);
    
    /* make the ctype with the given name */
    ctype = cast_new_prim_alias(ctype_flags, cmod_flags,
				get_ctype_name(ct, which_stub));
    
    /* Make the internal mapping. We will impose the translation
       mapping later, as we may not always use a translation. */
    
    if ((akIdent(CurArg->argKind) == akeWaitTime) || 
	(akIdent(CurArg->argKind) == akeMsgOption) ||
	(akIdent(CurArg->argKind) == akeMsgSeqno) ||
	(akIdent(CurArg->argKind) == akeServerCopy))
    {
	    /*
	     * XXX - akWaitTime and akMsgOption arguments should be marshaled
	     * into the Mach message header, and akMsgSeqno should be
	     * unmarshaled from the message header.  For right now,
	     * ignore them.
	     */
	    map = pres_c_new_mapping(PRES_C_MAPPING_MESSAGE_ATTRIBUTE);
	    switch (akIdent(CurArg->argKind)) {
	    case akeWaitTime:
		    map->pres_c_mapping_u_u.message_attribute.kind
			    = PRES_C_MESSAGE_ATTRIBUTE_TIMEOUT;
		    break;
		    
	    case akeMsgOption:
		    map->pres_c_mapping_u_u.message_attribute.kind
			    = PRES_C_MESSAGE_ATTRIBUTE_FLAGS;
		    break;
		    
	    case akeMsgSeqno:
    		    map->pres_c_mapping_u_u.message_attribute.kind
			    = PRES_C_MESSAGE_ATTRIBUTE_SEQUENCE_RECEIVED;
		    break;
		    
	    case akeServerCopy:
    		    map->pres_c_mapping_u_u.message_attribute.kind
			    = PRES_C_MESSAGE_ATTRIBUTE_SERVERCOPY;
		    break;

	    default:
		    /* XXX */
		    break;
	    }
	    itype = out_pres_c.mint.standard_refs.void_ref; /* Not Used */
    }
    else
	    map = pres_c_new_mapping(PRES_C_MAPPING_DIRECT);

    *out_itype = itype;
    *out_ctype = ctype;
    *out_map = map;
}

/* adds arg_inline to the MINT and to the PRES_C trees.
   If the itype is mint_ref_null, no MINT is added.
 */
static void add_arg_to_top_inlines(mint_ref itype, pres_c_inline arg_inline,
				   mint_ref struct_m, pres_c_inline struct_i)
{
	int struct_m_index, struct_i_index;
	
	/* Add the itype to the struct_m */
	if (itype != mint_ref_null)
	{
		struct_m_index = mint_add_struct_slot(&out_pres_c.mint,
						      struct_m);
		m(struct_m).mint_def_u.struct_def.
			slots.slots_val[struct_m_index]
			= itype;
	}
	else
		struct_m_index = mint_slot_index_null;
	
        /* add the arg_inline to the struct_i.  */
	struct_i_index = pres_c_add_inline_struct_slot(struct_i);

	switch (struct_i->kind) {
	case PRES_C_INLINE_STRUCT:
		/*
		 * Place the new slot just before the last one in the list.
		 * This effectively puts the deallocation slots first
		 * (in order), and pushes the virtual union to the end.
		 */
		assert(struct_i_index > 0);
		struct_i->pres_c_inline_u_u.struct_i.slots.
			slots_val[struct_i_index] =
			(struct_i->pres_c_inline_u_u.struct_i.slots.
			 slots_val[struct_i_index - 1]);
		
		struct_i->pres_c_inline_u_u.struct_i.slots.
			slots_val[struct_i_index-1].mint_struct_slot_index
			= struct_m_index;
		struct_i->pres_c_inline_u_u.struct_i.slots.
			slots_val[struct_i_index-1].inl
			= arg_inline;
		break;
		
	case PRES_C_INLINE_FUNC_PARAMS_STRUCT:
		struct_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[struct_i_index].mint_struct_slot_index
			= struct_m_index;
		struct_i->pres_c_inline_u_u.func_params_i.slots.
			slots_val[struct_i_index].inl
			= arg_inline;
		break;

	default:
		assert(struct_i->kind == PRES_C_INLINE_FUNC_PARAMS_STRUCT
		       || struct_i->kind == PRES_C_INLINE_STRUCT);
	}
}

static void make_ref_arg(argument_t *CurArg, int which_stub, int which_mess,
		         mint_ref *out_itype, cast_type *out_ctype, 
			 pres_c_mapping *out_map)
{
    ipc_type_t *CurType = CurArg->argType;

    mint_ref itype;
    cast_type ctype;
    pres_c_mapping map;
    
    /* This is the internal type for ports. The user will see something 
       different. */

    ctype = cast_new_type_name("mach_port_t");

    map = pres_c_new_mapping(PRES_C_MAPPING_REFERENCE);
    
    /* Okay, find out the port type. Note. It can be different between 
       the client and the server. (This is transmission types at work.) */
    
    switch (CurTypeNameUsed(CurType, which_stub, which_mess))
    {
    case MACH_MSG_TYPE_POLYMORPHIC:
	    /* assign dummy values which won't be used */
    case MACH_MSG_TYPE_PORT_NAME:
	    map->pres_c_mapping_u_u.ref.kind = PRES_C_REFERENCE_COPY;
	    itype = out_pres_c.mint.standard_refs.interface_name_ref;
	    break;
    case MACH_MSG_TYPE_COPY_SEND:
	    map->pres_c_mapping_u_u.ref.kind = PRES_C_REFERENCE_COPY;
	    itype = out_pres_c.mint.standard_refs.interface_invoke_ref;
	    break;
    case MACH_MSG_TYPE_MAKE_SEND:	
	    map->pres_c_mapping_u_u.ref.kind
		    = PRES_C_REFERENCE_COPY_AND_CONVERT;
	    itype = out_pres_c.mint.standard_refs.interface_invoke_ref;
	    break;
    case MACH_MSG_TYPE_MAKE_SEND_ONCE:
	    map->pres_c_mapping_u_u.ref.kind
		    = PRES_C_REFERENCE_COPY_AND_CONVERT;
	    itype = out_pres_c.mint.standard_refs.interface_invoke_once_ref;
	    break;
    case MACH_MSG_TYPE_MOVE_RECEIVE:
	    map->pres_c_mapping_u_u.ref.kind
		    = PRES_C_REFERENCE_MOVE;
	    itype = out_pres_c.mint.standard_refs.interface_service_ref;
	    break;
    case MACH_MSG_TYPE_MOVE_SEND:
	    map->pres_c_mapping_u_u.ref.kind
		    = PRES_C_REFERENCE_MOVE;
	    itype = out_pres_c.mint.standard_refs.interface_invoke_ref;
	    break;
    case MACH_MSG_TYPE_MOVE_SEND_ONCE:
	    map->pres_c_mapping_u_u.ref.kind
		    = PRES_C_REFERENCE_MOVE;
	    itype = out_pres_c.mint.standard_refs.interface_invoke_once_ref;
	    break;
    default:
	    panic("make_ref: unknown reference type %d", 
		  CurTypeNameUsed(CurType, which_stub, which_mess));
    }
    
    /* ref_count means how many references we're sending/receiving,
       which must be 1 or more. */
    map->pres_c_mapping_u_u.ref.ref_count = 1;
    *out_itype = itype;
    *out_ctype = ctype;
    *out_map = map;
}
    
/* 
 * Makes a string argument. This mapping is a terminated array mapping;
 * an array whose end is marked by a fencepost.
 */
static void make_string_arg(ipc_type_t *CurType, mint_ref *out_itype, 
			    cast_type *out_ctype, pres_c_mapping *out_map,
			    int which_stub)
{
    mint_ref itype;
    cast_type ctype;
    
    pres_c_mapping map;
    
    itype = xl_array(out_pres_c.mint.standard_refs.char8_ref,
		     CurType->itVarArray? 0:CurType->itNumber,
		     CurType->itNumber);
    
    ctype = cast_new_array_type(
	    cast_new_expr_lit_int(CurType->itNumber, 0),
	    cast_new_prim_type(CAST_PRIM_CHAR, 0));
    
    map = pres_c_new_mapping(PRES_C_MAPPING_TERMINATED_ARRAY);
    
    map->pres_c_mapping_u_u.terminated_array.terminator = 0;
    map->pres_c_mapping_u_u.terminated_array.max = CurType->itNumber;
    map->pres_c_mapping_u_u.terminated_array.element_mapping =
	    pres_c_new_mapping(PRES_C_MAPPING_DIRECT);
    
    if (which_stub == PRES_C_SERVER_SKEL) 	    
	    map->pres_c_mapping_u_u.terminated_array.alloc.flags = 
		    (PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS);
    else
	    map->pres_c_mapping_u_u.terminated_array.alloc.flags = 
		    (PRES_C_ALLOC_NEVER | PRES_C_DEALLOC_NEVER);
    
    map->pres_c_mapping_u_u.terminated_array.alloc.allocator =
	    "auto";
    
    *out_itype = itype;
    *out_ctype = ctype;
    *out_map = map;
}

static void make_top_inlines(int is_reply,
			     int mig_code,
			     mint_ref procs_union,
			     mint_ref *out_mint_struct,
			     pres_c_inline *out_top_inline,
			     pres_c_inline *out_struct_inline,
			     pres_c_inline *out_dealloc_inline)
{
	pres_c_inline top_i, intf_i, proc_i, union_i = 0, struct_i;
	mint_ref mint_union_ref, mint_struct_ref;
	int procs_case;
	
	pres_c_inline proc_variant_i;
	mint_ref proc_variant_ref;
	
	/*
	 * Create a MINT_STRUCT to hold the parameters for this routine.
	 * The slots array will be built on demand.
	 */
	mint_struct_ref = get_def();
	m(mint_struct_ref).kind = MINT_STRUCT;
	m(mint_struct_ref).mint_def_u.struct_def.slots.slots_val = 0;
	m(mint_struct_ref).mint_def_u.struct_def.slots.slots_len = 0;
	
	if (!is_reply)
		/*
		 * For requests, the MINT_STRUCT is the variant of the union
		 * of operations (`procs_union').
		 */
		proc_variant_ref = mint_struct_ref;
	else {
		/*
		 * For replies, the MINT_STRUCT we just created is a variant of
		 * a MINT_UNION, representing the union of normal replies and
		 * exceptional replies (i.e., errors).
		 */
		mint_union_def *union_def;

		mint_union_ref
			= mint_add_union_def(
				&out_pres_c.mint,
				out_pres_c.mint.standard_refs.unsigned32_ref,
				1 /* one discrimintated case */);
		union_def = &(m(mint_union_ref).mint_def_u.union_def);
		
		union_def->cases.cases_val[0].val
			= mint_new_symbolic_const(MINT_CONST_INT,
						  "KERN_SUCCESS");
		union_def->cases.cases_val[0].var = mint_struct_ref;
		
		/* The exceptional case: a system exception. */
		union_def->dfault = out_pres_c.mint.standard_refs.
				    system_exception_ref;
		
		proc_variant_ref = mint_union_ref;
	}
	
	/* Add `mint_variant_ref' to the union of operations `procs_union'. */
	procs_case = mint_add_union_case(&out_pres_c.mint, procs_union);
	
	m(procs_union).mint_def_u.union_def.cases.cases_val[procs_case].val
		= mint_new_const_int(mig_code);
	m(procs_union).mint_def_u.union_def.cases.cases_val[procs_case].var
		= proc_variant_ref;
	
	/*
	 * Now create the necessary PRES_C inlines, starting with the inline
	 * for `mint_struct_ref' and going up to the one for the union of all
	 * IDLs.
	 */
	struct_i = pres_c_new_inline_func_params_struct(0);
	/* The `return_slot' is never used by the MIG FE/PG. */
	struct_i->pres_c_inline_u_u.func_params_i.return_slot = 0;
	
	if (!is_reply)
		/*
		 * For requests, the `struct_i' inline will be a variant of
		 * the `proc_i' below.
		 */
		proc_variant_i = struct_i;
	else {
		/*
		 * For replies, we need to create a PRES_C_INLINE_VIRTUAL_UNION
		 * corresponding to the MINT_UNION we created above.  The
		 * virtual union inline will be contained within `proc_i'.
		 */
		pres_c_inline_virtual_union *union_def;
		
		union_i = pres_c_new_inline_virtual_union(1);
		union_def = &(union_i->pres_c_inline_u_u.virtual_union);
		
		/*
		 * The discriminator is a `kern_return_t' and is the return
		 * value of our stub.
		 */
		union_def->discrim.ctype
			= cast_new_type_name("kern_return_t");
		union_def->discrim.map
			= pres_c_new_mapping(PRES_C_MAPPING_DIRECT);
		union_def->discrim.name
			= "_return";
		
		/*
		 * This is the ``magic'' (aka ``crap'') for the normal reply
		 * case; see `pfe/lib/p_do_return_union.cc'.
		 */
		union_def->cases.cases_val[0].map
			= pres_c_new_mapping(PRES_C_MAPPING_STRUCT);
		union_def->cases.cases_val[0].map->pres_c_mapping_u_u.struct_i
			= struct_i;
		union_def->cases.cases_val[0].name = ";;;reply";
		union_def->cases.cases_val[0].ctype = 0;
		
		/*
		 * This is the ``magic'' (aka ``crap'') for the system
		 * exception case; see `pfe/sun/p_exceptions.cc'.
		 */
		union_def->dfault
			= (pres_c_inline_virtual_union_case *)
			  mustmalloc(sizeof(pres_c_inline_virtual_union_case));
		union_def->dfault->map
			= pres_c_new_mapping(PRES_C_MAPPING_SYSTEM_EXCEPTION);
		union_def->dfault->ctype
			= cast_new_type(CAST_TYPE_VOID);
		union_def->dfault->name
			= "\"no data available\"";
		
		proc_variant_i = union_i;
	}
	
	/* Now create some ``magic'' to allow deallocation of in parameters
	   in the reply */
	if (is_reply) {
		/*
		 * Create a PRES_C_INLINE_FUNC_PARAMS_STRUCT to hold
		 * the virtual union created above, as well as the `in'
		 * parameters that (possibly) need to be deallocated.
		 */
		pres_c_inline params_i;
		pres_c_inline_struct *params_def;
		mint_ref params_ref;
		
		params_i = pres_c_new_inline_struct(1);
		params_def = &(params_i->pres_c_inline_u_u.struct_i);
		
		/* The `return_slot' is never used for this */
		params_def->slots.slots_val[0].mint_struct_slot_index = 0;
		params_def->slots.slots_val[0].inl = union_i;

		proc_variant_i = params_i;
		
		/*
		 * Create a MINT_STRUCT to hold the parameters & virtual union.
		 */
		params_ref = get_def();
		m(params_ref).kind = MINT_STRUCT;
		m(params_ref).mint_def_u.struct_def.slots.slots_len = 1;
		m(params_ref).mint_def_u.struct_def.slots.slots_val =
			mustcalloc(sizeof(mint_ref));
		m(params_ref).mint_def_u.struct_def.slots.slots_val[0] =
			proc_variant_ref;
		
		m(procs_union).mint_def_u.union_def.cases.cases_val[procs_case].var
			= params_ref;

		assert(out_dealloc_inline);
		*out_dealloc_inline = params_i;
	}
	
	/* Now create the inline that traverses the union of procedures. */
	proc_i = pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	
	proc_i->pres_c_inline_u_u.collapsed_union.discrim_val
		= mint_new_const_int(mig_code);
	proc_i->pres_c_inline_u_u.collapsed_union.selected_case
		= proc_variant_i;
	
	/* Then create the inline that traverses the union of interfaces. */
	intf_i = pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	
	intf_i->pres_c_inline_u_u.collapsed_union.discrim_val
		= mint_new_const_int(SubsystemBase);
	intf_i->pres_c_inline_u_u.collapsed_union.selected_case = proc_i;
	
	/* Finally, create the inline that traverses the union of IDLs. */
	top_i = pres_c_new_inline(PRES_C_INLINE_COLLAPSED_UNION);
	
	top_i->pres_c_inline_u_u.collapsed_union.discrim_val
		= mint_new_const_int(IDL_MIG); /* MIG magic number */
	top_i->pres_c_inline_u_u.collapsed_union.selected_case = intf_i;
	
	*out_mint_struct = mint_struct_ref;
	*out_struct_inline = struct_i;
	*out_top_inline = top_i;
}

/* Adds a fixed array mapping onto a element mapping, which of course may be
   another fixed array mapping. Currently, we limit the nesting of 
   fixed_array mappings to two levels, even though MIG doesn't require it. 
   The element mapping is replaced by the fixed array map. */

static void interpose_fixed_array_arg(mint_ref *itypep, cast_type *ctypep, 
				      pres_c_mapping *mapp, maxint array_len,
				      int which_stub)
{
	pres_c_mapping array_map;
	
	/* It must be fixed length. */
	*itypep = xl_array(*itypep, array_len, array_len);
	
	*ctypep = cast_new_array_type(cast_new_expr_lit_int(array_len, 0), 
				      *ctypep);
	
	array_map = pres_c_new_mapping(PRES_C_MAPPING_FIXED_ARRAY);
	
	array_map->pres_c_mapping_u_u.fixed_array.length = array_len;
	array_map->pres_c_mapping_u_u.fixed_array.element_mapping = *mapp;
	
	/* alloc/dealloc a fixed array on the server side only. */
	if (which_stub == PRES_C_SERVER_SKEL) {
		array_map->pres_c_mapping_u_u.fixed_array.alloc.flags
			= (PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS);
	} else {
		array_map->pres_c_mapping_u_u.fixed_array.alloc.flags
			= (PRES_C_ALLOC_NEVER | PRES_C_DEALLOC_NEVER);
	}
	array_map->pres_c_mapping_u_u.fixed_array.alloc.allocator
		= "auto";
	*mapp = array_map;
}

/* If type translation information was given, make the necessary 
   mapping. */
static void interpose_mapping_xlate(argument_t *CurArg, int which_stub,
				    cast_type *ctypep, pres_c_mapping *mapp)
{
	cast_type ctype;
	pres_c_mapping map;
	const char *presented_ctype_name;
	
	/*
	 * Create the external (presented) C type.  Be careful to recognize C
	 * keywords and create corresponding CAST_PRIM_* types; if the internal
	 * and presented types are the same, we can avoid the translation node
	 * entirely.
	 */
	presented_ctype_name = get_ctype_name(CurArg->argType, which_stub);
	if      (!strcmp(presented_ctype_name, "int"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, 0);
	else if (!strcmp(presented_ctype_name, "long"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_LONG);
	else if (!strcmp(presented_ctype_name, "short"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_SHORT);
	else if (!strcmp(presented_ctype_name, "signed"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_SIGNED);
	else if (!strcmp(presented_ctype_name, "unsigned"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_UNSIGNED);
	else if (!strcmp(presented_ctype_name, "char"))
		ctype = cast_new_prim_type(CAST_PRIM_CHAR, 0);
	else if (!strcmp(presented_ctype_name, "float"))
		ctype = cast_new_prim_type(CAST_PRIM_FLOAT, 0);
	else if (!strcmp(presented_ctype_name, "double"))
		ctype = cast_new_prim_type(CAST_PRIM_DOUBLE, 0);
	else
		ctype = cast_new_type_name((char *) presented_ctype_name);
	
	map = pres_c_new_mapping(PRES_C_MAPPING_XLATE);
	map->pres_c_mapping_u_u.xlate.internal_mapping = *mapp;
	map->pres_c_mapping_u_u.xlate.internal_ctype = *ctypep;

	if (!CurArg->argType->itVarArray &&
	    !CurArg->argType->itIndefinite &&
	    CurArg->argType->itNumber > 1) {
		if (CurArg->argType->itPassByValue)
			map->pres_c_mapping_u_u.xlate.translator = "&";
		else
			map->pres_c_mapping_u_u.xlate.translator = "~";
	} else
		map->pres_c_mapping_u_u.xlate.translator = "";

	map->pres_c_mapping_u_u.xlate.destructor = "";

	*ctypep = ctype;
	*mapp = map;
}
	       

/* If type translation information was given, make the necessary 
   mapping. */

static void interpose_type_translation(argument_t *CurArg, 
				       cast_type *ctypep, pres_c_mapping *mapp,
				       int which_stub, int which_mess,
				       int *used_trans)
{
	pres_c_mapping server_map; 
	pres_c_mapping trans_map;

	ipc_type_t *CurType = CurArg->argType;
	
	if (CurType->itInTrans && which_mess == akIn)
	{
	    *used_trans = 1;
	    
	    trans_map = pres_c_new_mapping(PRES_C_MAPPING_XLATE);
	    
	    trans_map->pres_c_mapping_u_u.xlate.internal_mapping = *mapp;
	    
	    trans_map->pres_c_mapping_u_u.xlate.internal_ctype = 	   
		    cast_new_type_name((char *)CurType->itServerType);

	    trans_map->pres_c_mapping_u_u.xlate.translator = 
		    (char *) CurType->itInTrans;
	    
	    if (CurType->itDestructor)
	    {
		    trans_map->pres_c_mapping_u_u.xlate.alloc_flags = 
			    PRES_C_DEALLOC_ALWAYS;
		    
		    trans_map->pres_c_mapping_u_u.xlate.destructor = 
			    (char *) CurType->itDestructor;
	    }
	    else
		    trans_map->pres_c_mapping_u_u.xlate.destructor = "";

	    *mapp = trans_map;
	}

	else if (CurType->itOutTrans && which_mess == akOut)
	{	    
	    *used_trans = 1;
	    trans_map = pres_c_new_mapping(PRES_C_MAPPING_XLATE);
	    server_map = pres_c_new_mapping(PRES_C_MAPPING_XLATE);

	    server_map->pres_c_mapping_u_u.xlate.internal_mapping = *mapp;

	    server_map->pres_c_mapping_u_u.xlate.internal_ctype = *ctypep;

	    server_map->pres_c_mapping_u_u.xlate.translator = "";
		    
	    server_map->pres_c_mapping_u_u.xlate.destructor = "";
	    
	    trans_map->pres_c_mapping_u_u.xlate.internal_mapping = server_map;
	    
	    trans_map->pres_c_mapping_u_u.xlate.internal_ctype = 
		    cast_new_type_name((char *)CurType->itServerType);

	    trans_map->pres_c_mapping_u_u.xlate.translator = 
		    (char *) CurType->itOutTrans;
	    
	    trans_map->pres_c_mapping_u_u.xlate.alloc_flags = 
		PRES_C_DEALLOC_NEVER;
	    
	    trans_map->pres_c_mapping_u_u.xlate.destructor = "";

	    *mapp = trans_map;
	}
}

static void add_typedef(const char * presented_ctype_name, cast_type ctype,
			cast_type savetype)
{
	int new_def;
	
	/* Save the old ctype as an implied typedef for later */
	if (cast_find_typedef_type(&(out_pres_c.cast), ctype) == NULL) {
		new_def = cast_add_def(&(out_pres_c.cast));
		out_pres_c.cast.cast_scope_val[new_def].name
			= strmake(presented_ctype_name);
		out_pres_c.cast.cast_scope_val[new_def].sc = CAST_SC_EXTERN;
		out_pres_c.cast.cast_scope_val[new_def].u.kind = CAST_TYPEDEF;
		out_pres_c.cast.cast_scope_val[new_def].included = IMPLIED;
		out_pres_c.cast.cast_scope_val[new_def].u.cast_def_u_u.
			typedef_type = savetype;
	}
}

/* Impose a type translation at the inline level.
   This is used for _every_ MIG argument,
   so that the user-visible function prototype will use the user's 
   specified type instead of the actual internal type used for marshaling 
   or unmarshaling. This kludge is necessary only because, unlike 
   sensible IDL compilers, MIG doesn't produce typedefs for data types 
   declared in the IDL file; it only produces the function prototypes,
   and expects the user to have declared the appropriate typedefs.
   Since Flick can't know exactly what the user will declare a 
   particular C type to be, we just have to spit out a null translation node
   and hope a simple typecast does the trick... */

static void interpose_inline_xlate(argument_t *CurArg, int which_stub,
				   cast_type *ctypep, pres_c_inline *inlp,
				   int index, int extern_indir, 
				   int make_null)
{
	ipc_type_t *CurType = CurArg->argType;
	
	identifier_t presented_ctype_name;
	cast_type ctype;
	pres_c_inline xlate_inline;
	
	/*
	 * Create the external (presented) C type.  Be careful to recognize C
	 * keywords and create corresponding CAST_PRIM_* types; if the internal
	 * and presented types are the same, we can avoid the translation node
	 * entirely.
	 */
	presented_ctype_name = get_ctype_name(CurArg->argType, which_stub);
	if      (!strcmp(presented_ctype_name, "int"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, 0);
	else if (!strcmp(presented_ctype_name, "long"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_LONG);
	else if (!strcmp(presented_ctype_name, "short"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_SHORT);
	else if (!strcmp(presented_ctype_name, "signed"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_SIGNED);
	else if (!strcmp(presented_ctype_name, "unsigned"))
		ctype = cast_new_prim_type(CAST_PRIM_INT, CAST_MOD_UNSIGNED);
	else if (!strcmp(presented_ctype_name, "char"))
		ctype = cast_new_prim_type(CAST_PRIM_CHAR, 0);
	else if (!strcmp(presented_ctype_name, "float"))
		ctype = cast_new_prim_type(CAST_PRIM_FLOAT, 0);
	else if (!strcmp(presented_ctype_name, "double"))
		ctype = cast_new_prim_type(CAST_PRIM_DOUBLE, 0);
	else
		ctype = cast_new_type_name((char *) presented_ctype_name);
	
	/*
	 * Interpose a pointer on the presented C type if required (e.g., for
	 * an `out' parameter).
	 */
	if (extern_indir) {
		assert((*ctypep)->kind == CAST_TYPE_POINTER);
		add_typedef(presented_ctype_name, ctype,
			    (*ctypep)->cast_type_u_u.pointer_type.target);
		ctype = cast_new_pointer_type(ctype);
	} else {
		add_typedef(presented_ctype_name, ctype, *ctypep);
	}
	
	/*
	 * Decide if we need a translation inline at all.
	 */
	if (!cast_cmp_type(ctype, (*ctypep)))
		/*
		 * The presented and internal C types are the same, so no xlate
		 * node is necessary.
		 */
		;
	else {
		xlate_inline = pres_c_new_inline(PRES_C_INLINE_XLATE);
		
		xlate_inline->pres_c_inline_u_u.xlate.sub = *inlp;
		xlate_inline->pres_c_inline_u_u.xlate.index = index;
		
		if (make_null) {
			xlate_inline->pres_c_inline_u_u.xlate.internal_ctype
#if 0
				= ctype;
#else
			        = *ctypep;  /*ECP: I don't know why it
					      wouldn't be this*/
#endif
			
			xlate_inline->pres_c_inline_u_u.xlate.alloc_flags
				= PRES_C_DEALLOC_NEVER;
			
			xlate_inline->pres_c_inline_u_u.xlate.translator = "";
			xlate_inline->pres_c_inline_u_u.xlate.destructor = "";
			
		} else if (CurType->itVarArray || CurType->itIndefinite
			   || (CurType->itNumber <= 1) || (!CurType->itInLine)) {
			xlate_inline->pres_c_inline_u_u.xlate.internal_ctype
				= *ctypep;
			
			/*
			 * The translation for the xlate will always be a cast.
			 */
			xlate_inline->pres_c_inline_u_u.xlate.alloc_flags
				= (PRES_C_ALLOC_ALWAYS |
				   PRES_C_DEALLOC_ALWAYS);
			
			xlate_inline->pres_c_inline_u_u.xlate.translator = "";
			xlate_inline->pres_c_inline_u_u.xlate.destructor = "";
		} else {
			/*
			 * If we have a fixed-size array, make the internal
			 * type a pointer to the element type.  This pointer
			 * will be cast into the appropriate type.  If we have
			 * a struct, the address of the variable must be cast.
			 */
			xlate_inline->pres_c_inline_u_u.xlate.internal_ctype
				= *ctypep;
			
			xlate_inline->pres_c_inline_u_u.xlate.alloc_flags
				= (PRES_C_ALLOC_NEVER
				   | PRES_C_DEALLOC_NEVER);
			
			if (CurType->itPassByValue) /*Struct*/
				xlate_inline->pres_c_inline_u_u.xlate.
					translator = "&";
			else {
				if (which_stub == PRES_C_SERVER_SKEL) {
					/*
					 * no translation necessary, since the
					 * server creates pointer-to-element
					 * types instead of array types.
					 */
					xlate_inline->pres_c_inline_u_u.xlate.
						translator = "";
				} else
					xlate_inline->pres_c_inline_u_u.xlate.
						translator = "~";
			}
			
			xlate_inline->pres_c_inline_u_u.xlate.destructor = "";
		}
		
		/* Interpose the new inline and ctype on the old ones. */
		*ctypep = ctype;
		*inlp = xlate_inline;
	}
}
#if 0
static pres_c_inline
make_cond_inline(pres_c_inline truep, pres_c_inline falsep, int index)
{
	pres_c_inline new_i = pres_c_new_inline(PRES_C_INLINE_COND);

	new_i->pres_c_inline_u_u.cond.index = index;

	new_i->pres_c_inline_u_u.cond.true_inl = truep;

	new_i->pres_c_inline_u_u.cond.false_inl = falsep;
	
	return new_i;
}

static void interpose_dealloc_condition(pres_c_inline *inlp,
					pres_c_mapping dealloc_map, 
					pres_c_mapping nodealloc_map,
					argument_t *CurArg,
					cast_ref cfunc,
					int which_stub)
{
	int dealloc_param;
	
	pres_c_inline true_i;
	pres_c_inline false_i;
	
	assert(CurArg->argDealloc);
		
	false_i = mustmalloc(sizeof(pres_c_inline_u));
	
	true_i = (*inlp);
	*false_i = *(*inlp);
	
	true_i->pres_c_inline_u_u.counted_array.ptr.mapping = 
		dealloc_map;
	
	false_i->pres_c_inline_u_u.counted_array.ptr.mapping = 
		nodealloc_map;
	
	dealloc_param = cast_func_add_param(&c(cfunc));
	
	c(cfunc).params.params_val[dealloc_param].name = 
		(char *)CurArg->argDealloc->argName;
	
	c(cfunc).params.params_val[dealloc_param].type = 
		cast_new_type_name(
			(char *)get_ctype_name(CurArg->argDealloc->argType,
					       which_stub))
	
	/* Should be second parameter after count. */
	
	*inlp = make_cond_inline(true_i, false_i, dealloc_param);
}

static void interpose_scopy_inline(pres_c_inline *inlp, int index)
{
	pres_c_inline new_i = pres_c_new_inline(PRES_C_INLINE_ASSIGN);
	
	new_i->pres_c_inline_u_u.assign.sub = *inlp;
	new_i->pres_c_inline_u_u.assign.index = index;

	new_i->pres_c_inline_u_u.assign.value = cast_new_expr_lit_int(0,0);

	*inlp = new_i;
}
#endif
static void interpose_counted_array_arg(mint_ref *itypep,
					cast_type *ctypep,
					pres_c_mapping *mapp,
					argument_t *CurArg, 
					int *needs_extern_indir,
					int which_stub) 
{
        maxuint array_max;
	
	ipc_type_t *CurType = CurArg->argType;
	
        /* MIG supports variable-length arrays of fixed-length arrays.  */

	assert(CurType->itElement);
	
    	if (CurType->itElement->itNumber > 1)
	{
		/* take care of the array of array case */
		interpose_fixed_array_arg(itypep, ctypep, mapp, 
					  CurType->itElement->itNumber,
					  which_stub);
	}

	/* Create the array *itypep, *ctypep, and pointer mapping
	   interposition.  */
	if (CurType->itIndefinite || !CurType->itInLine) {
		array_max = (CurType->itLongForm) ? 0xffffffff : 0xffff;
		if (!CurType->itInLine) {
			pres_c_interpose_pointer(
				ctypep, mapp,
				((CurArg->argDeallocate == d_YES)?
				 PRES_C_ALLOC_NEVER|PRES_C_DEALLOC_ALWAYS:
				 PRES_C_ALLOC_NEVER|PRES_C_DEALLOC_NEVER),
				"out_of_line");
		} else {
			if (which_stub == PRES_C_SERVER_SKEL)
				pres_c_interpose_pointer(
					ctypep, mapp,
					PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS,
					"auto");
			else
				pres_c_interpose_pointer(
					ctypep, mapp,
					PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_NEVER,
					"mach_vm");
		}
	} else {
		pres_c_mapping map;
		
		array_max = CurType->itNumber;
		*ctypep = cast_new_array_type(cast_new_expr_lit_int(array_max, 0), 
					      *ctypep);
		map = pres_c_new_mapping(PRES_C_MAPPING_POINTER);
		if (which_stub == PRES_C_SERVER_SKEL) {
			map->pres_c_mapping_u_u.pointer.alloc.flags =
				PRES_C_ALLOC_ALWAYS | PRES_C_DEALLOC_ALWAYS;
			map->pres_c_mapping_u_u.pointer.alloc.allocator = "auto";
		} else {
			map->pres_c_mapping_u_u.pointer.alloc.flags =
				PRES_C_ALLOC_NEVER | PRES_C_DEALLOC_NEVER;
			map->pres_c_mapping_u_u.pointer.alloc.allocator = "mach_vm";
		}
		map->pres_c_mapping_u_u.pointer.target = *mapp;
		*needs_extern_indir = 0;
		*mapp = map;
	}
	
	*itypep = xl_array(*itypep, 0, array_max);
}


#if 0
     UNUSED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!11

     if (CurArg->argFlags & flMaybeDealloc)
	{
		cast_type ignore_type;
		
		/* Clone the old *mapp, as we will need two mappings if we
		   have a dealloc parameter */
		
		dealloc_map = mustmalloc(sizeof(pres_c_mapping_u));
		*dealloc_map = **mapp;
		
		ignore_type = *ctypep;

		pres_c_interpose_pointer(ctypep, mapp,
					 CurType->itInLine ? 
					 PRES_C_FAIL_IF_TOO_SMALL : 
					 PRES_C_ALLOC_ALWAYS,
					 CurType->itInLine ?
					 "mach_vm" : "out_of_line");

		pres_c_interpose_pointer(&ignore_type, &dealloc_map,
					 CurType->itInLine ? 
					 PRES_C_FAIL_IF_TOO_SMALL : 
					 PRES_C_ALLOC_ALWAYS | 
					 PRES_C_DEALLOC_ALWAYS,
					 CurType->itInLine ?
					 "mach_vm" : "out_of_line");
	}
	else if (CurArg->argFlags & flDealloc)
	{
		pres_c_interpose_pointer(ctypep, mapp,
					 CurType->itInLine ? 
					 PRES_C_FAIL_IF_TOO_SMALL : 
					 PRES_C_ALLOC_ALWAYS | 
					 PRES_C_DEALLOC_ALWAYS,
					 CurType->itInLine ?
					 "mach_vm" : "out_of_line");
	}

	else
	{
		pres_c_interpose_pointer(ctypep, mapp,
					 CurType->itInLine ? 
					 PRES_C_FAIL_IF_TOO_SMALL : 
					 PRES_C_ALLOC_NEVER |
					 PRES_C_DEALLOC_NEVER,
					 CurType->itInLine ?
					 "mach_vm" : "out_of_line");
	}
	
	/* If necessary, interpose a _second_ level of pointer indirection
	   for out and inout parameters.  */

        if (*needs_extern_indir) {
		if (which_stub == PRES_C_CLIENT_STUB)
			pres_c_interpose_pointer(ctypep, mapp,
						 (PRES_C_ALLOC_NEVER
						  | PRES_C_DEALLOC_NEVER),
						 "mig");
		else if (which_stub == PRES_C_SERVER_SKEL)
			pres_c_interpose_pointer(ctypep, mapp,
						 (PRES_C_ALLOC_ALWAYS
						  | PRES_C_DEALLOC_ALWAYS),
						 "auto");
		else
			panic("In `make_arg', generating neither client stub "
			      "nor server skeleton!");
		
		if (CurArg->argFlags & flMaybeDealloc)
			/* This can't whack on the `*ctypep' again. */
			pres_c_interpose_pointer(0 /*ctypep*/, &dealloc_map, 
						 PRES_C_ALLOC_NEVER, "");

	if (CurArg->argDealloc)
		interpose_direction_mapping(&dealloc_map, CurArg, which_stub);

    /* If we have a Dealloc parameter, we have to add an inline hack
       that will trigger deallocation if the parameter is true. */
    
	if (CurArg->argDealloc)
	{
		interpose_dealloc_condition(inlp, dealloc_map, *mapp, CurArg, 
					    &c(cfunc), which_stub);
	}
	
	/* If we have a ServerCopy parameter, we have to add an inline hack
	   that force out of line semantics. */

	if (CurArg->argSCopy)
	{
		int scopy_param;

		scopy_param = cast_func_add_param(&c(cfunc));
		
		c(cfunc).params.params_val[scopy_param].name = 
			(char *)CurArg->argSCopy->argName;

		c(cfunc).params.params_val[scopy_param].type = 
			cast_new_type_name(
				(char *)get_ctype_name(
					CurArg->argSCopy->argType,
					which_stub));
		
		interpose_scopy_inline(inlp, scopy_param);
	}
#endif
	
	
	
void make_arg_server_dealloc_in(cast_type arg_ctype,
				mint_ref arg_itype,
				pres_c_mapping arg_mapping,
				pres_c_mapping *alloc_mapping)
{
	/*****/
	
	*alloc_mapping = 0;
	
	switch (arg_mapping->kind) {
	default:
		/*
		 * By default, we assume that the server dispatch function is
		 * not required to do anything special beyond deallocating the
		 * ``root'' of the in parameter.
		 */
		break;
		
	case PRES_C_MAPPING_POINTER: {
		/*
		 * XXX - If we have a CAST_ARRAY, this means we have an
		 * inline_counted_array that is bounded.  For deallocation
		 * purposes, this simply needs to be a fixed-size array with
		 * the maximum number of elements.
		 */
		cast_type ctype =
			cast_find_typedef_type(&(out_pres_c.cast), arg_ctype);
		assert(ctype);
		
		*alloc_mapping = pres_c_new_mapping(PRES_C_MAPPING_POINTER);
		
		(*alloc_mapping)->pres_c_mapping_u_u.pointer
			= arg_mapping->pres_c_mapping_u_u.pointer;
		(*alloc_mapping)->pres_c_mapping_u_u.pointer.target
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
	}
	case PRES_C_MAPPING_FIXED_ARRAY:
		/*
		 *
		 */
		*alloc_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_FIXED_ARRAY);
		
		if (strcmp(arg_mapping->pres_c_mapping_u_u.
			   fixed_array.alloc.allocator, "auto"))
			break;
		
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array
			= arg_mapping->pres_c_mapping_u_u.fixed_array;
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
			element_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
		
	case PRES_C_MAPPING_TERMINATED_ARRAY:
		/*
		 *
		 */
		*alloc_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_TERMINATED_ARRAY);
		
		if (strcmp(arg_mapping->pres_c_mapping_u_u.
			   terminated_array.alloc.allocator, "auto"))
			break;
		
		(*alloc_mapping)->pres_c_mapping_u_u.terminated_array
			= arg_mapping->pres_c_mapping_u_u.terminated_array;
		(*alloc_mapping)->pres_c_mapping_u_u.terminated_array.
			element_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
		
#if 0
	case PRES_C_MAPPING_STUB: {
		cast_type new_ctype;
		pres_c_mapping new_mapping;
		
		/*
		 *
		 */
		new_ctype = arg_ctype;
		new_mapping = arg_mapping;
		
		pres_descend_mapping_stub(out_pres, arg_itype,
					  &new_ctype, &new_mapping);
		make_arg_server_dealloc_in(new_ctype, arg_itype, new_mapping,
					   alloc_mapping);
		break;
	}
#endif
	case PRES_C_MAPPING_DIRECTION:
		/*
		 * Ignore the mapping direction; just use the real mapping
		 */
		make_arg_server_dealloc_in(arg_ctype, arg_itype,
					   (arg_mapping->pres_c_mapping_u_u.
					    direction.mapping),
					   alloc_mapping);
		break;
	}
}

/*
 * THIS IS AN ALMOST DIRECT COPY FROM `c/pfe/lib/p_param.cc's
 * p_param_server_alloc_out().
 *
 * Decide if this is an `out' parameter for which the server muct allocate
 * storage beyond the root.  If so, create an `alloc_mapping' for doing this.
 */
void make_arg_server_alloc_out(cast_type arg_ctype,
			       mint_ref arg_itype,
			       pres_c_mapping arg_mapping,
			       pres_c_mapping *alloc_mapping)
{
	/*****/
	
	*alloc_mapping = 0;
	
	switch (arg_mapping->kind) {
	default:
		/*
		 * By default, we assume that the server dispatch function is
		 * not required to do anything special beyond allocating the
		 * ``root'' of the out parameter.
		 */
		fprintf(stderr,
			"Warning: In `make_arg_server_alloc_out', "
			"unexpected mapping type %d.\n",
			arg_mapping->kind);
		break;
		
	case PRES_C_MAPPING_POINTER: {
		/*
		 * XXX - If we have a CAST_ARRAY, this means we have an
		 * inline_counted_array that is bounded.  For allocation
		 * purposes, this simply needs to be a fixed-size array with
		 * the maximum number of elements.
		 */
		cast_type ctype =
			cast_find_typedef_type(&(out_pres_c.cast), arg_ctype);
		assert(ctype);
		if (ctype->kind == CAST_TYPE_ARRAY) {
			*alloc_mapping =
				pres_c_new_mapping(PRES_C_MAPPING_FIXED_ARRAY);
			(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
				alloc =
				arg_mapping->pres_c_mapping_u_u.pointer.alloc;
			(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
				length =
				m(m(arg_itype).mint_def_u.array_def.length_type).
				mint_def_u.integer_def.range;
			(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
				element_mapping =
				pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
			break;
		}
		
		*alloc_mapping = pres_c_new_mapping(PRES_C_MAPPING_POINTER);
		
		(*alloc_mapping)->pres_c_mapping_u_u.pointer
			= arg_mapping->pres_c_mapping_u_u.pointer;
		(*alloc_mapping)->pres_c_mapping_u_u.pointer.target
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
	}
	case PRES_C_MAPPING_FIXED_ARRAY:
		/*
		 *
		 */
		*alloc_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_FIXED_ARRAY);
		
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array
			= arg_mapping->pres_c_mapping_u_u.fixed_array;
		(*alloc_mapping)->pres_c_mapping_u_u.fixed_array.
			element_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
		
	case PRES_C_MAPPING_TERMINATED_ARRAY:
		/*
		 *
		 */
		*alloc_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_TERMINATED_ARRAY);
		
		(*alloc_mapping)->pres_c_mapping_u_u.terminated_array
			= arg_mapping->pres_c_mapping_u_u.terminated_array;
		(*alloc_mapping)->pres_c_mapping_u_u.terminated_array.
			element_mapping
			= pres_c_new_mapping(PRES_C_MAPPING_IGNORE);
		break;
		
#if 0
	case PRES_C_MAPPING_STUB: {
		cast_type new_ctype;
		pres_c_mapping new_mapping;
		
		/*
		 *
		 */
		new_ctype = arg_ctype;
		new_mapping = arg_mapping;
		
		pres_descend_mapping_stub(out_pres, arg_itype,
					  &new_ctype, &new_mapping);
		make_arg_server_alloc_out(new_ctype, arg_itype, new_mapping,
					  alloc_mapping);
		break;
	}
#endif
	case PRES_C_MAPPING_DIRECTION:
		/*
		 * Ignore the mapping direction; just use the real mapping
		 */
		make_arg_server_alloc_out(arg_ctype, arg_itype,
					  (arg_mapping->pres_c_mapping_u_u.
					   direction.mapping),
					  alloc_mapping);
		break;
	}
}

/* Determine the type of a argument, make it. */
/* make_arg returns a non-null alloc_mapp if it is an Out parameter and
   it needs allocation information put into the Request */

void make_arg(argument_t *CurArg, int which_stub, int which_mess, int cparam, 
	      cast_ref cfunc, cast_type *ctypep, mint_ref *itypep,
	      pres_c_inline *arg_ip, pres_c_mapping *alloc_mapp,
	      pres_c_mapping *dealloc_mapp, int *extern_indir)
{
	pres_c_mapping map;
	
	ipc_type_t *CurType = CurArg->argType;
	
	int made_trans = 0;
	
	int is_struct = 0;
	int add_ool = 0;
	
	/* This is set to true if the external type needs a level of 
	   pointer indirection.  */
	
	*extern_indir = IsArgOut(CurArg, which_stub);
	
	/* First produce a ctype, itype, and mapping for the 
	   basic "singleton" type.  */
	
	if (akIdent(CurArg->argKind) == akePoly)
	{
		*itypep = get_def();
		m(*itypep).kind = MINT_TYPE_TAG;
		
		*ctypep = cast_new_prim_alias(CAST_PRIM_INT, CAST_MOD_UNSIGNED,
					      "mach_msg_type_name_t");
		map = pres_c_new_mapping(PRES_C_MAPPING_TYPE_TAG);
	}
	else if ((akIdent(CurArg->argKind) == akeCount)
		 || (akIdent(CurArg->argKind) == akeCountInOut))
	{
		/* no MINT needed here: all is in MINT_ARRAY */
		*itypep = mint_ref_null;
		
		*ctypep = cast_new_prim_alias(CAST_PRIM_INT, CAST_MOD_UNSIGNED,
					      "mach_msg_type_number_t");

		/* Make the length mapping an argument mapping */
		map = pres_c_new_mapping(PRES_C_MAPPING_DIRECT);
		pres_c_interpose_argument(&map, "length");
	}
	else
		switch (CurTypeNameUsed(CurType, which_stub, which_mess))
		{
		case MACH_MSG_TYPE_PORT_NAME:
		case MACH_MSG_TYPE_MOVE_RECEIVE:
		case MACH_MSG_TYPE_MOVE_SEND:	
		case MACH_MSG_TYPE_MOVE_SEND_ONCE:
		case MACH_MSG_TYPE_COPY_SEND:
		case MACH_MSG_TYPE_MAKE_SEND:
		case MACH_MSG_TYPE_MAKE_SEND_ONCE:
		case MACH_MSG_TYPE_POLYMORPHIC:
			
			make_ref_arg(CurArg, which_stub, which_mess,
				     itypep, ctypep, &map);
			break;
			
		case MACH_MSG_TYPE_UNSTRUCTURED:
		case MACH_MSG_TYPE_INTEGER_64: 
		case MACH_MSG_TYPE_INTEGER_32:    
		case MACH_MSG_TYPE_INTEGER_16:
		case MACH_MSG_TYPE_INTEGER_8:
		case MACH_MSG_TYPE_REAL:
		case MACH_MSG_TYPE_CHAR:
			
			make_simple_arg(CurArg, which_stub, which_mess,
					itypep, ctypep, &map);
			break;
			
		case MACH_MSG_TYPE_STRING:
			
			/* this is handled below */
			
			break;
			
			
		default:
			panic("Unsupported message type %d",
			      CurTypeNameUsed(CurType, which_stub,
					      which_mess));
		}
	
	/* Handle fixed and variable-length strings and arrays. */
	
	if (CurTypeNameUsed(CurType, which_stub, which_mess)
	    == MACH_MSG_TYPE_STRING) {
		/*
		 * Strings are special cases:
		 *   They never need external pointer indirection
		 *     (even for out or inout parameters)
		 *   Variable length strings never have a count
		 *     parameter associated with them; their length
		 *     is implicitly determined by the terminating nul
		 */

		make_string_arg(CurType, itypep, ctypep, &map, which_stub);
		/* inline strings never require extra indirection */
		if (CurType->itInLine)
			*extern_indir = 0;
		else
			add_ool = 1;
	} else if (CurType->itIndefinite || CurType->itVarArray) {
                /* Counted array */
		interpose_counted_array_arg(itypep, ctypep, &map, CurArg,
					    extern_indir, which_stub);
	} else if (CurType->itNumber != 1) {
		/*
		 * If this fixed-length array really represents a structure,
		 * then it is pass-by-value and needs indirection for 
		 * out and inout parameters.
		 * If it's really a fixed-length array, no external type
		 * indirection is needed.
		 * If it's OOL data, this doesn't apply.
		 */

		if (CurType->itInLine) {
			if (!CurType->itPassByValue)
				*extern_indir = 0;
			else
				/* Inline structs needs special handling */
				is_struct = 1;
		}
		
		/* Handle plain fixed-length arrays.  */
		interpose_fixed_array_arg(itypep, ctypep, &map,
					  CurType->itNumber,
					  which_stub);
		/* Out-of-line data */
		if (!CurType->itInLine)
			add_ool = 1;
	} else if (!CurType->itInLine) {
		add_ool = 1;
	}
	
	if (add_ool) {
		/* Out-of-line data */
		/* create a pointer to the data */
		pres_c_interpose_pointer(
			ctypep, &map,
			((CurArg->argDeallocate == d_YES)?
			 PRES_C_ALLOC_NEVER|PRES_C_DEALLOC_ALWAYS:
			 PRES_C_ALLOC_NEVER|PRES_C_DEALLOC_NEVER),
			"out_of_line");
	}
	
	/* Check and make translation mapping, if needed. */
	if (which_stub == PRES_C_SERVER_SKEL) {
		interpose_type_translation(CurArg, ctypep,
					   &map, which_stub,
					   which_mess, &made_trans);
	}

	/* Interpose a mapping translation for inline structs. */
	if (is_struct)
		interpose_mapping_xlate(CurArg, which_stub,
					ctypep, &map);

        /* add the levels of pointer indirection that may be necessary,
	   and convert it to an inline. */
	*arg_ip = interpose_inline_indirect(ctypep, &map, CurArg,
					    cparam, *extern_indir, which_stub);

	/* Now that we have the complete internal type worked out,
	   encapsulate it in a pres_c_inline_xlate
	   so the external type can be the name the user specified.
	   (we've already translated inline structs, so exclude them here)
	*/
	if (!is_struct)
		interpose_inline_xlate(CurArg, which_stub, ctypep, arg_ip,
				       cparam, *extern_indir, made_trans);
	
	/* Load up the function parameter with the
	   final external name and *ctypep.*/
	c(cfunc).params.params_val[cparam].name = (char*)CurArg->argName;
	c(cfunc).params.params_val[cparam].type = *ctypep;
	
	/* if arg is an Out parameter (not InOut), create an alloc_map for the
	   request */
	if (IsArgOut(CurArg, which_stub) &&
	    !IsArgInOut(CurArg, which_stub))
		make_arg_server_alloc_out(*ctypep, *itypep, map,
					  alloc_mapp);
	
	/* if arg is an In parameter (not InOut), create a dealloc_map for the
	   reply */
	if (IsArgIn(CurArg, which_stub) &&
	    !IsArgInOut(CurArg, which_stub))
		make_arg_server_dealloc_in(*ctypep, *itypep, map,
					   dealloc_mapp);
}

void interpose_inline_typed(argument_t *TypeArg, int which_stub,
			    cast_ref cfunc,
			    mint_ref *arg_mp, pres_c_inline *arg_ip,
			    int *type_cparam,
			    pres_c_mapping *type_alloc_mapp,
			    pres_c_mapping *type_dealloc_mapp)
{
	/* the definitions for the tag type arg */
	mint_ref type_arg_m, typed_m;
	pres_c_inline type_arg_i, typed_i;
	cast_type type_arg_ctype;
	int extern_indir;
	
	assert(TypeArg->argParent);
	
	/* process the type arg */
	*type_cparam = cast_func_add_param(&c(cfunc));
	make_arg(TypeArg, which_stub, akIn, *type_cparam,
		 cfunc, &type_arg_ctype, &type_arg_m,
		 &type_arg_i, type_alloc_mapp,
		 type_dealloc_mapp, &extern_indir);
	
	/* build the PRES_C_INLINE_TYPED */
	typed_i = pres_c_new_inline(PRES_C_INLINE_TYPED);
	typed_i->pres_c_inline_u_u.typed.inl
		= *arg_ip;
	typed_i->pres_c_inline_u_u.typed.tag
		= type_arg_i;
	
	/* build the MINT_TYPED */
	typed_m = get_def();
	m(typed_m).kind = MINT_TYPED;
	m(typed_m).mint_def_u.typed_def.ref = *arg_mp;
	m(typed_m).mint_def_u.typed_def.tag = type_arg_m;
	
	/* now return the TYPED structure as the focus */
	*arg_mp = typed_m;
	*arg_ip = typed_i;
}

			    
void interpose_inline_counted_array(argument_t *CountArg, int which_stub,
				    cast_ref cfunc,
				    mint_ref *array_arg_mp,
				    pres_c_inline *array_arg_ip,
				    int length_cparam,
				    pres_c_mapping *length_alloc_mapp,
				    pres_c_mapping ptr_dealloc_map,
				    int extern_indir,
				    pres_c_inline *dealloc_ip)
{
	/* the definitions for the length arg and array */
	mint_ref length_arg_m;
	pres_c_inline length_arg_i, counted_array_i;
	pres_c_mapping length_dealloc_map;
	pres_c_inline dealloc_len_i = 0, dealloc_arr_i = 0;
	cast_type length_arg_ctype;
	int length_extern_indir;
	pres_c_inline lai;
	
	/* used to move any PRES_C_INLINE_XLATE above the
	   PRES_C_INLINE_COUNTED_ARRAY */
	pres_c_inline xlate_i = 0;
	
	/* process the length arg */
	
	make_arg(CountArg, which_stub, akIn, length_cparam,
		 cfunc, &length_arg_ctype, &length_arg_m,
		 &length_arg_i, length_alloc_mapp,
		 &length_dealloc_map, &length_extern_indir);

	if (length_arg_i->kind == PRES_C_INLINE_XLATE) {
		lai = length_arg_i->pres_c_inline_u_u.xlate.sub;
	} else {
		lai = length_arg_i;
	}
	assert(lai->kind == PRES_C_INLINE_ATOM);
	
	/* create a dealloc length for `in' arrays */
	if (IsArgIn(CountArg, which_stub) && !IsArgOut(CountArg, which_stub)) {
		dealloc_len_i = pres_c_new_inline(PRES_C_INLINE_ATOM);
		dealloc_len_i->pres_c_inline_u_u.atom.index
			= lai->pres_c_inline_u_u.atom.index;
		dealloc_len_i->pres_c_inline_u_u.atom.mapping = 0;
		pres_c_interpose_argument(
			&dealloc_len_i->pres_c_inline_u_u.atom.mapping,
			"length");
	}
	
	/* build the PRES_C_INLINE_COUNTED_ARRAY */

	counted_array_i = pres_c_new_inline(PRES_C_INLINE_COUNTED_ARRAY);
	/* make the dealloc inline for `in' arrays */
	if (IsArgIn(CountArg, which_stub) && !IsArgOut(CountArg, which_stub)) {
		dealloc_arr_i = pres_c_new_inline(PRES_C_INLINE_COUNTED_ARRAY);
	}
	
	/* save any translation done for later interposition */
	if ((*array_arg_ip)->kind == PRES_C_INLINE_XLATE) {
		xlate_i = *array_arg_ip;
		*array_arg_ip = (*array_arg_ip)->pres_c_inline_u_u.xlate.sub;
	}
	
	assert((*array_arg_ip)->kind == PRES_C_INLINE_ATOM);
	/* copy the inline_atom */
	counted_array_i->pres_c_inline_u_u.counted_array.ptr
		= (*array_arg_ip)->pres_c_inline_u_u.atom;
	/* create the pointer and dealloc map for `in' arrays */
	if (IsArgIn(CountArg, which_stub) && !IsArgOut(CountArg, which_stub)) {
		/* copy the inline_atom */
		dealloc_arr_i->pres_c_inline_u_u.counted_array.ptr
			= (*array_arg_ip)->pres_c_inline_u_u.atom;
		assert(ptr_dealloc_map);
		/* change the mapping */
		dealloc_arr_i->pres_c_inline_u_u.counted_array.ptr
			.mapping = ptr_dealloc_map;
	}
	
	/* free(*array_arg_ip);  should be able to free it now */
	
	counted_array_i->pres_c_inline_u_u.counted_array.len
		= length_arg_i;
	
	counted_array_i->pres_c_inline_u_u.counted_array.max = 0;
	
	counted_array_i->pres_c_inline_u_u.counted_array.ptr_levels
		= extern_indir;

	/* make the dealloc inline */
	if (IsArgIn(CountArg, which_stub) && !IsArgOut(CountArg, which_stub)) {
		dealloc_arr_i->pres_c_inline_u_u.counted_array.len
			= dealloc_len_i;
		dealloc_arr_i->pres_c_inline_u_u.counted_array.max = 0;
		dealloc_arr_i->pres_c_inline_u_u.counted_array.ptr_levels
			= extern_indir;
	}
	
	/* now return the PRES_C_INLINE_COUNTED_ARRAY as the focus */
	*array_arg_ip = counted_array_i;
	/*
	 * `in' arrays get the dealloc inline we just created (for
	 *  deallocation in the reply message), while all others are
	 *  deallocated (if necessary) within the counted array inline
	 *  itself.
	 */
	if (IsArgIn(CountArg, which_stub) && !IsArgOut(CountArg, which_stub)) {
		*dealloc_ip = dealloc_arr_i;
	} else {
		*dealloc_ip = 0;
	}
	
	/* interpose any saved translation */
	if (xlate_i != 0)
	{
		xlate_i->pres_c_inline_u_u.xlate.sub = *array_arg_ip;
		*array_arg_ip = xlate_i;
	}
}

void make_routine(routine_t *rt, int which_stub, mint_ref procs_union,
		  cast_ref *out_c_func,
		  mint_ref *out_target_m, pres_c_inline *out_target_i,
		  mint_ref *out_client_m, pres_c_inline *out_client_i,
		  pres_c_inline *out_request_i, pres_c_inline *out_reply_i)
{
    cast_ref c_func;
    cast_type arg_ctype;
    pres_c_mapping alloc_map, extra_alloc_map;
    pres_c_mapping dealloc_map, extra_dealloc_map;
    mint_ref arg_m, request_m, reply_m, target_m, client_m;
    pres_c_inline arg_i, target_i, client_i, request_i, reply_i, dealloc_i;
    pres_c_inline request_struct_i, reply_struct_i, reply_dealloc_i;
    argument_t *arg;
    int cparam, extra_cparam;
    int extern_indir;
    
    /* Create a function declaration.  */
    {
	const char *prefix = which_stub == PRES_C_CLIENT_STUB ? UserPrefix 
		: ServerPrefix;
	char *name = flick_asprintf("%s%s", prefix, rt->rtName);

	c_func = cast_add_func_def(&c_scope, name, 0);
	c(c_func).return_type = cast_new_type_name("kern_return_t");
    }
    
    make_top_inlines(0, /* request */
		     SubsystemBase+rt->rtNumber, procs_union,
    		     &request_m, &request_i, &request_struct_i,
		     &reply_dealloc_i);
    make_top_inlines(1, /* reply */
		     SubsystemBase+rt->rtNumber+100, procs_union,
    		     &reply_m, &reply_i, &reply_struct_i,
		     &reply_dealloc_i);
    assert(reply_dealloc_i);
    
    /* These will be set as necessary as we process the arguments. */
    target_i = 0;
    client_i = 0;
    target_m = mint_ref_null;
    client_m = mint_ref_null;
    
    /* Make the error mapping to send error codes to the return value
     */
        
    /*
     * Walk through the arguments for this routine.
     * index is the index into the presentation struct.
     * count is the index of the argument in the CAST function decl.
     */

    for (arg = rt->rtArgs; arg != NULL; arg=arg->argNext)
    {
	    /* Skip these types of arguments (for now) */
	    /* akePoly, akeCount, and akeCountInOut are processed by their
	       Parent directly. */
	    if (arg->argType == NULL
		|| akIdent(arg->argKind) == akeReturn
		|| akIdent(arg->argKind) == akeCount 
		|| akIdent(arg->argKind) == akePoly 
		|| akIdent(arg->argKind) == akeDealloc
		|| akIdent(arg->argKind) == akeCountInOut)
	    {
		    continue;
	    }
	    
	    if (akIdent(arg->argKind) == akeRequestPort)	
		{
		    cast_type ctype; pres_c_mapping map;

		    make_ref_arg(arg, which_stub, akIn, &arg_m, &ctype, &map);

		    cparam = cast_func_add_param(&c(c_func));

		    target_i = pres_c_new_inline_atom(cparam, map);
#if 0
		    /*
		     * XXX --- No translation is required; either the
		     * target is a port or it's not.  We can't ``fix''
		     * it by casting it.  Moreover, the code produced
		     * by an inline translation sometimes gets *lost*
		     * because it is output when the BE is invoking
		     * methods on an auxiliary `mu_target_...' object.
		     * Some BE's throw away the `mu_target_...' code,
		     * and keep just the side effect.  (See the Fluke
		     * BE, for example.)
		     *
		     * So even if we did try to translate the target
		     * object reference, we would sometimes end up with
		     * bogus output code.
		     */
		    interpose_inline_xlate(arg, which_stub, &ctype, 
					   &target_i, cparam, 0, 0);
#endif
		    
		    c(c_func).params.params_val[cparam].name
			    = (char*)arg->argName;
		    c(c_func).params.params_val[cparam].type = ctype;

		    if ((arg->argPoly != NULL)
			&& (CurTypeNameUsed(arg->argType, which_stub, akIn)
			    == MACH_MSG_TYPE_POLYMORPHIC))
			    interpose_inline_typed(
				    (arg->argPoly), which_stub,
				    c_func, &arg_m, &target_i,
				    &extra_cparam,
				    &extra_alloc_map,
				    &extra_dealloc_map);
		    
		    target_m = arg_m;
		    
		    continue;
		}
	    
	    if (akCheckAll(arg->argKind, akReplyPort) ||
		(akCheckAll(arg->argKind,akUReplyPort) && 
			which_stub == PRES_C_CLIENT_STUB) ||
		(akCheckAll(arg->argKind,akSReplyPort) && 
  		        which_stub == PRES_C_SERVER_SKEL))
	    {
		    cast_type ctype; pres_c_mapping map;

		    make_ref_arg(arg, which_stub, akIn, &arg_m, &ctype, &map);

		    cparam = cast_func_add_param(&c(c_func));

		    client_i = pres_c_new_inline_atom(cparam, map);
#if 0
		    /*
		     * XXX --- No translation is required; either the
		     * target is a port or it's not.  We can't ``fix''
		     * it by casting it.  Moreover, the code produced
		     * by an inline translation sometimes gets *lost*
		     * because it is output when the BE is invoking
		     * methods on an auxiliary `mu_target_...' object.
		     * Some BE's throw away the `mu_target_...' code,
		     * and keep just the side effect.  (See the Fluke
		     * BE, for example.)
		     *
		     * So even if we did try to translate the target
		     * object reference, we would sometimes end up with
		     * bogus output code.
		     */
		    interpose_inline_xlate(arg, which_stub, &ctype, 
					   &client_i, cparam, 0, 0);
#endif
		    
		    c(c_func).params.params_val[cparam].name
			    = (char*)arg->argName;
		    c(c_func).params.params_val[cparam].type = ctype;

		    if ((arg->argPoly != NULL)
			&& (CurTypeNameUsed(arg->argType, which_stub, akIn)
			    == MACH_MSG_TYPE_POLYMORPHIC))
			    interpose_inline_typed(
				    (arg->argPoly), which_stub,
				    c_func, &arg_m, &client_i,
				    &extra_cparam,
				    &extra_alloc_map,
				    &extra_dealloc_map);
		    
		    client_m = arg_m;
		    
		    continue;
	    }

	    if (akCheckAll(arg->argKind,akUReplyPort) ||
		akCheckAll(arg->argKind,akSReplyPort))
		continue;
	    
	    if (akIdent(arg->argKind) == akeServerCopy) {
		    if (which_stub == PRES_C_SERVER_SKEL) {
			    cparam = cast_func_add_param(&c(c_func));
			    make_arg(arg, which_stub, akIn,
				     cparam, c_func, &arg_ctype,
				     &arg_m, &arg_i,
				     &alloc_map, &dealloc_map,
				     &extern_indir);
			    add_arg_to_top_inlines(
				    arg_m, arg_i,
				    request_m,
				    request_struct_i);
		    }
		    continue;
	    }
	    
	    /* All arguments should be In and/or Out
	       or not be a User/Server Arg at all (like dummy args) */
	    if (IsArg(arg, which_stub)
		&& !IsArgIn(arg, which_stub)
		&& !IsArgOut(arg, which_stub))
		    fprintf(stderr,"In make_routine(): warning: "
			  "Skipping `%s' which is a valid argument!\n",
			  arg->argName);
	    
	    extra_cparam = -1;
	    
	    /* For normal arguments,
	       handle them for the request and/or reply
	       depending on whether they're in, out, or inout.  */
	    
	    if (IsArgInOut(arg, which_stub))
	    {
		    int which_mess = akIn;
		    mint_ref struct_m = request_m;
		    pres_c_inline struct_i = request_struct_i;
		    
		    /* Allocate a C function parameter for this MIG
		       parameter.  Additional C parameters may also
		       be allocated later.  */
		    cparam = cast_func_add_param(&c(c_func));
		    
		    /* loop to process first the request then the reply */
		    do {
			    
			    make_arg(arg, which_stub, which_mess, cparam,
				     c_func,
				     &arg_ctype, &arg_m, &arg_i, &alloc_map,
				     &dealloc_map, &extern_indir);
			    
			    if ((arg->argCount != NULL)
				&& (arg->argType->itIndefinite
				    || arg->argType->itVarArray)
				&& (CurTypeNameUsed(arg->argType, which_stub,
						    which_mess)
				    != MACH_MSG_TYPE_STRING)) {
				    /* create the function parameter,
				       but only do it once */
				    if (which_mess == akIn)
					    extra_cparam =
						    cast_func_add_param(&c(c_func));
				    interpose_inline_counted_array(
					    (arg->argCount), which_stub,
					    c_func, &arg_m,
					    &arg_i, extra_cparam,
					    &extra_alloc_map,
					    dealloc_map,
					    extern_indir,
					    &dealloc_i);
			    } else if ((arg->argPoly != NULL)
				&& (CurTypeNameUsed(
					arg->argType, which_stub, which_mess)
				    == MACH_MSG_TYPE_POLYMORPHIC))
				    interpose_inline_typed(
					    (arg->argPoly), which_stub,
					    c_func, &arg_m,
					    &arg_i, &extra_cparam,
					    &extra_alloc_map,
					    &extra_dealloc_map);
			    
			    /* Add the arg_m to the struct_m, and the
			       arg_i to the struct_i.  */
			    add_arg_to_top_inlines(arg_m, arg_i, struct_m,
						   struct_i);
			    if (which_mess == akOut)
				    break; /* quit loop */
			    which_mess = akOut;
			    struct_m = reply_m;
			    struct_i = reply_struct_i;
			    
		    } while (1) /* until after processing the Reply */;
	    }
	    else if (IsArgIn(arg, which_stub))
	    {
		    /* Allocate a C function parameter for this MIG
		       parameter.  Additional C parameters may also
		       be allocated later.  */
		    dealloc_i = 0;
		    cparam = cast_func_add_param(&c(c_func));
		    
		    make_arg(arg, which_stub, akIn, cparam, c_func,
			     &arg_ctype, &arg_m, &arg_i, &alloc_map,
			     &dealloc_map, &extern_indir);
		    
		    if ((arg->argCount != NULL)
			&& (arg->argType->itIndefinite
			    || arg->argType->itVarArray)
			&& (CurTypeNameUsed(arg->argType, which_stub, akIn)
			    != MACH_MSG_TYPE_STRING)) {
			    extra_cparam = cast_func_add_param(&c(c_func));
			    interpose_inline_counted_array(
				    (arg->argCount), which_stub,
				    c_func, &arg_m, &arg_i, extra_cparam,
				    &extra_alloc_map,
				    dealloc_map,
				    extern_indir,
				    &dealloc_i);
			    /* there may be a need to dealloc this, so add
			       it to the reply's deallocation inline */
			    if (which_stub == PRES_C_SERVER_SKEL)
				    add_arg_to_top_inlines(
					    mint_ref_null, dealloc_i,
					    reply_m, reply_dealloc_i);
		    } else if ((arg->argPoly != NULL)
			&& (CurTypeNameUsed(arg->argType, which_stub, akIn)
			    == MACH_MSG_TYPE_POLYMORPHIC))
			    interpose_inline_typed(
				    (arg->argPoly), which_stub,
				    c_func, &arg_m, &arg_i, &extra_cparam,
				    &extra_alloc_map,
				    &extra_dealloc_map);
		    
		    if (!dealloc_i) {
		    /* If necessary, add the deallocation to the reply */
		    if (dealloc_map != 0)
			    add_arg_to_top_inlines(
				    mint_ref_null,
				    pres_c_new_inline_atom(cparam, dealloc_map),
				    reply_m, reply_dealloc_i);
		    
		    if ((extra_cparam != -1) && (extra_dealloc_map != 0))
			    add_arg_to_top_inlines(
				    mint_ref_null,
				    pres_c_new_inline_atom(
					    extra_cparam,
					    extra_dealloc_map),
				    reply_m, reply_dealloc_i);
		    }
		    /* Add the arg_m to the request_m, and the
		       arg_i to the request_struct_i.  */
		    add_arg_to_top_inlines(arg_m, arg_i, request_m,
					   request_struct_i);
	    }
	    else if (IsArgOut(arg, which_stub))
	    {
		    pres_c_mapping alloc_map = 0;
		    
		    /* Allocate a C function parameter for this MIG
		       parameter.  Additional C parameters may also
		       be allocated later.  */
		    cparam = cast_func_add_param(&c(c_func));
		    
		    make_arg(arg, which_stub, akOut, cparam, c_func,
			     &arg_ctype, &arg_m, &arg_i, &alloc_map,
			     &dealloc_map, &extern_indir);
    
		    if ((arg->argCount != NULL)
			&& (arg->argType->itIndefinite
			    || arg->argType->itVarArray)
			&& (CurTypeNameUsed(arg->argType, which_stub, akOut)
			    != MACH_MSG_TYPE_STRING)) {
			    extra_cparam = cast_func_add_param(&c(c_func));
			    interpose_inline_counted_array(
				    (arg->argCount), which_stub,
				    c_func, &arg_m, &arg_i,
				    extra_cparam, &extra_alloc_map,
				    dealloc_map, extern_indir,
				    &dealloc_i);
		    } else if ((arg->argPoly != NULL)
			&& (CurTypeNameUsed(arg->argType, which_stub, akOut)
			    == MACH_MSG_TYPE_POLYMORPHIC))
			    interpose_inline_typed(
				    (arg->argPoly), which_stub,
				    c_func, &arg_m, &arg_i,
				    &extra_cparam, &extra_alloc_map,
				    &extra_dealloc_map);
			    
		    /* If necessary, add the allocation to the request */
		    if (alloc_map != 0)
			    add_arg_to_top_inlines(
				    mint_ref_null,
				    pres_c_new_inline_atom(cparam, alloc_map),
				    request_m, request_struct_i);
			    
		    if ((extra_cparam != -1) && (extra_alloc_map != 0))
			    add_arg_to_top_inlines(
				    mint_ref_null,
				    pres_c_new_inline_atom(
					    extra_cparam,
					    extra_alloc_map),
				    request_m, request_struct_i);
		    
		    /* Add the arg_m to the reply_m, and the
		       arg_i to the reply_struct_i.  */
		    add_arg_to_top_inlines(arg_m, arg_i, reply_m,
					   reply_struct_i);
	    }
    }

    /* Set the out parameters.  */
    *out_c_func = c_func;
    *out_target_i = target_i;
    *out_client_i = client_i;
    *out_target_m = target_m;
    *out_client_m = client_m;
    *out_request_i = request_i;
    *out_reply_i = reply_i;
}

/* End of file. */

