/*
 * Copyright (c) 1995, 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <mom/compiler.h>

/*
 * `CPP' and `CPPFLAGS' should be defined by the configuration process and
 * communicated to us to through the compilation command line.  Warning: If
 * `gcc -E' is the preprocessor, `CPPFLAGS' must contain `-x c' or else `gcc'
 * will assume that the input `.x' file is a linker file.
 */

#ifndef CPP
#  define CPP "/lib/cpp"
#endif
#ifndef CPPFLAGS
#  define CPPFLAGS "-C"
#endif

static char cpp_command[] = CPP;
static char cpp_flags[] = CPPFLAGS;

#define CPP_ARGC_MAX (128)
extern char *infilename;

/*
 * Invoke the C preprocessor on `infile', with command argument `flags', and
 * return a `FILE *' so that we receive the preprocessor's output.
 */

FILE *call_c_preprocessor(char *infile, char *flags)
{
	char *cpp_cmdline;
	
	char *cpp_argv[CPP_ARGC_MAX];
	int cpp_argc;
	
	char *cpp_program_name;
	
	int pd[2];
	int cmdlen = strlen(cpp_command) + 1 + strlen(cpp_flags) + 1;
	if (flags)
		cmdlen += (1 + strlen(flags));
	
	/* Parse the `cpp' command line into an argument vector. */
	cpp_cmdline = (char *) mustmalloc(cmdlen);
	
	strcpy(cpp_cmdline, cpp_command);
	strcat(cpp_cmdline, " ");
	strcat(cpp_cmdline, cpp_flags);
	if (flags) {
		strcat(cpp_cmdline, " ");
		strcat(cpp_cmdline, flags);
	}
	
	for (cpp_argc = 0; cpp_argc < CPP_ARGC_MAX; ++cpp_argc) {
		cpp_argv[cpp_argc] =
			strtok(((cpp_argc == 0) ? cpp_cmdline : 0), " \t");
		if (!cpp_argv[cpp_argc])
			/* When we get a null pointer, we're done. */
			break;
	}
	if (cpp_argc == 0)
		panic("The command to run the C preprocessor is empty.");
	if (cpp_argc >= CPP_ARGC_MAX)
		panic("Too many arguments to the C preprocessor.");
	
	if (!infile) {
		/* We may need to specify `-' on the command line to convince
		   the C preprocessor to read from stdin.  Of course, all this
		   is guesswork. */
		cpp_program_name = strrchr(cpp_argv[0], '/');
		if (cpp_program_name)
			++cpp_program_name;
		else
			cpp_program_name = cpp_argv[0];
		
		if (strstr(cpp_program_name, "gcc"))
			infile = "-";
	}
	
	if (infile) {
		/* At this point, `cpp_argc' is the index of the null string
		   that was returned by `strtok'. */
		cpp_argv[cpp_argc] = infile;
		++cpp_argc;
		if (cpp_argc >= CPP_ARGC_MAX)
			panic("Too many arguments to the C preprocessor.");
		cpp_argv[cpp_argc] = 0;
	}
	
	/* Set up our input pipeline. */
	infilename = (((infile == NULL) || (!strcmp(infile, "-"))) ?
		      "<stdin>" : infile);
	(void) pipe(pd);
	switch (fork()) {
	case 0:
		(void) close(1);
		(void) dup2(pd[1], 1);
		(void) close(pd[0]);
		execvp(cpp_argv[0], cpp_argv);
		perror("execvp");
		exit(1);
	case -1:
		perror("fork");
		exit(1);
	}
	(void) close(pd[1]);
	return fdopen(pd[0], "r");
}

/* End of file. */

