/*
 * Copyright (c) 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <string.h>
#include <stdlib.h>

#include <mom/compiler.h>

static int parse_arg(/* const */ char *rest, /* rest of the current argument */
		     /* const */ char *next, /* the next argument (may be 0) */
		     flags_in flag,
		     flags *out)
{
	char *arg_data;
	int arg_data_consumed = 0;
	int temp;
	
	if (rest[0]) {
		arg_data = rest;
		arg_data_consumed = 0;
	} else {
		arg_data = next;
		arg_data_consumed = (flag.which != fk_FLAG);
	}
	
	switch (flag.which) {
	case fk_FLAG:
		if (arg_data) {
			if (!strcmp(arg_data, "1")
			    || !strcmp(arg_data, "y")
			    || !strcmp(arg_data, "yes")) {
				out->flag = 1;
				arg_data_consumed = (arg_data == next);
				
			} else if (!strcmp(arg_data, "0")
				   || !strcmp(arg_data, "n")
				   || !strcmp(arg_data, "no")) {
				out->flag = 0;
				arg_data_consumed = (arg_data == next);
				
			} else if (rest[0] == 0) {
				/* Parse `-x' as setting the flag. */
				out->flag = 1;
				arg_data_consumed = 0;
				
			} else
				return -1;
		} else {
			/* Parse `-x' as setting the flag. */
			out->flag = 1;
			arg_data_consumed = 0;
		}
		break;
		
	case fk_STRING:
		if (arg_data)
			out->string = arg_data;
		break;
		
	case fk_NUMBER:
		if (arg_data) {
			out->number = 0;
			for (temp = 0; arg_data[temp]; temp++) {
				if ((arg_data[temp] < '0')
				    || (arg_data[temp] > '9'))
					break;
				out->number = out->number * 10
					      + (arg_data[temp] - '0');
			}
			if (arg_data[temp])
				return -1;
		}
		break;
	}
	
	return arg_data_consumed;
}

flags_out parse_args(int argc, char *argv[], int flag_count, flags_in *flgs)
{
	flags_out res;
	int i;
	res.error = 0;
	res.other_count = 0;
	res.other = 0;
	
	res.flgs = (flags *) mustmalloc(sizeof(flags) * flag_count);
	
	/* Set all the defaults */
	for (i = 0; i < flag_count; i++) {
		switch (flgs[i].which) {
		case fk_FLAG:
			res.flgs[i].flag = flgs[i].dfault.flag;
			break;
		case fk_STRING:
			res.flgs[i].string = flgs[i].dfault.string;
			break;
		case fk_NUMBER:
			res.flgs[i].number = flgs[i].dfault.number;
			break;
		default:
			panic("Incorrect value for a flag discriminator");
		}
	}
	
	res.progname = argv[0];
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') {
			/*
			 * It's not a valid option --- add it to the list of
			 * others arguments.
			 */
			res.other = (char **)
				    mustrealloc(res.other,
						(++res.other_count
						 * sizeof (char *)));
			res.other[res.other_count - 1] = argv[i];
			
		} else if (argv[i][1] != '-') {
			/*
			 * It's a short form (`-') option.  Short form options
			 * may be concatenated with their values (e.g., `-s1')
			 * or they may precede their values (e.g., `-s 1').
			 */
			int j, OK = 0;
			
			for (j = 0; j < flag_count && !OK; j++) {
				if (flgs[j].sng
				    && (argv[i][1] == flgs[j].sng)) {
					/*
					 * We found the flag, parse its args.
					 */
					int add = parse_arg(&(argv[i][2]),
							    argv[i+1],
							    flgs[j],
							    &(res.flgs[j]));
					if (add >= 0) {
						i += add;
						OK = 1;
					}
					break;
				}
			}
			if (!OK) {
				res.other = (char **)
					    mustrealloc(res.other,
							(++res.other_count
							 * sizeof(char *)));
				res.other[res.other_count - 1] = argv[i];
				res.error = 1;
			}
			
		} else {
			/*
			 * It's a long form (`--') option.  A long form option
			 * may be concatenated with its value if the value is
			 * preceded by `=' (e.g., `--this=that').  Long form
			 * options may also may precede their values (e.g.,
			 * `--this that').
			 *
			 * Concatenation requires `=' so that we can identify
			 * the end of the option name.  Otherwise, we couldn't
			 * allow string-valued long-form options to share a
			 * common prefix.  For example, we couldn't have both
			 * `--client' and `--client_type'.
			 */
			int j, OK = 0;
			
			for (j = 0; j < flag_count && !OK; j++) {
				int len = strlen(flgs[j].dbl);
				
				if (!strncmp(flgs[j].dbl, &(argv[i][2]),
					     len)
				    && ((argv[i][2+len] == 0)
					|| (argv[i][2+len] == '='))) {
					/*
					 * We found the flag, parse its args.
					 */
					int add =
						parse_arg(
							(argv[i][2+len] ?
							 /* Strip `='. */
							 &(argv[i][2+len+1]) :
							 /* Otherwise... */
							 &(argv[i][2+len])),
							argv[i+1],
							flgs[j],
							&(res.flgs[j]));
					
					if (add >= 0) {
						i += add;
						OK = 1;
					}
					break;
				}
			}
			if (!OK) {
				res.other = (char **)
					    mustrealloc(res.other,
							(++res.other_count
							 * sizeof(char *)));
				res.other[res.other_count - 1] = argv[i];
				res.error = 1;
			}
		}
	}
	
	return res;
}

void print_args_usage(char *progname,
		      int flag_count, flags_in *flgs,
		      char *cl_extra, char *extra) 
{
	int i;
	
	fprintf(stderr, "Usage:  %s <options> %s\n\n", progname, cl_extra);
	
	for (i = 0; i < flag_count; i++) {
		/* Print the short and long forms of the flag. */
		if (flgs[i].sng)
			fprintf(stderr, "-%c", flgs[i].sng);
		if (flgs[i].dbl)
			fprintf(stderr, "%s--%s",
				(flgs[i].sng ? " / " : ""),
				flgs[i].dbl);
		
		/* Print the type and explanation. */
		fprintf(stderr, "%s:\n\t%s\n",
			((flgs[i].which == fk_FLAG) ? "" :
			 (flgs[i].which == fk_NUMBER) ? " <number>" :
			 (flgs[i].which == fk_STRING) ? " <string>" :
			 " <unknown type>"),
			(flgs[i].explain ?
			 flgs[i].explain :
			 "(No documentation available.)"));
	}
	if (extra)
		fprintf(stderr, "\n%s\n", extra);
}

void print_args_flags(flags_out res, int flag_count, flags_in *flgs)
{
	int i;
	
	char **name, **value, **dfault;
	
	name = (char **) mustcalloc(sizeof(char *) * flag_count);
	value = (char **) mustcalloc(sizeof(char *) * flag_count);
	dfault = (char **) mustcalloc(sizeof(char *) * flag_count);
	
	fprintf(stderr, "Flags for %s:\n", res.progname);
	
	for (i = 0; i < flag_count; i++) {
		/* Print the short and long forms of the flag. */
		if (flgs[i].sng)
			fprintf(stderr, "-%c", flgs[i].sng);
		if (flgs[i].dbl)
			fprintf(stderr, "%s--%s",
				(flgs[i].sng ? " / " : ""),
				flgs[i].dbl);
		
		/* Print the type and explanation. */
		switch (flgs[i].which) {
		case fk_FLAG:
			fprintf(stderr,
				":\n\tdefault: %s\n\tvalue:   %s\n",
				(flgs[i].dfault.flag ? "true" : "false"),
				(res.flgs[i].flag ? "true" : "false"));
			break;
			
		case fk_NUMBER:
			fprintf(stderr,
				" <number>:\n\tdefault: %d\n\tvalue:   %d\n",
				flgs[i].dfault.number,
				res.flgs[i].number);
			break;
			
		case fk_STRING:
			fprintf(stderr,
				" <string>:\n");
			if (flgs[i].dfault.string)
				fprintf(stderr, "\tdefault: \"%s\"\n",
					flgs[i].dfault.string);
			else
				fprintf(stderr, "\tdefault: (null)\n");
			if (res.flgs[i].string)
				fprintf(stderr, "\tvalue:   \"%s\"\n",
					res.flgs[i].string);
			else
				fprintf(stderr, "\tvalue:   (null)\n");
			break;
			
		default:
			fprintf(stderr, " <unknown type>\n");
			break;
		}
	}
	
	if (res.other_count) 
		fprintf(stderr, "\nAdditional command line arguments:\n");
	for (i = 0 ; i < res.other_count; i++)
		fprintf(stderr, "%s\n", res.other[i]);
}

void set_def_flags(flags_in *in) 
{
	in[0].sng = 'v';
	in[0].dbl = "version";
	in[0].which = fk_FLAG;
	in[0].dfault.flag = 0;
	in[0].explain = "Print out the version information for this program";
	in[1].sng = 'u';
	in[1].dbl = "usage";
	in[1].which = fk_FLAG;
	in[1].dfault.flag = 0;
	in[1].explain = "Print out this message";
	in[OUTPUT_FILE_FLAG].sng = 'o';
	in[OUTPUT_FILE_FLAG].dbl = "output";
	in[OUTPUT_FILE_FLAG].which = fk_STRING;
	in[OUTPUT_FILE_FLAG].dfault.string = 0;
	in[OUTPUT_FILE_FLAG].explain = "Optional output file name";
	in[3].sng = '?';
	in[3].dbl = "help";
	in[3].which = fk_FLAG;
	in[3].dfault.flag = 0;
	in[3].explain = "Print out this message";
}

void std_handler(flags_out out, int flag_count,
		 flags_in *in, char *cl_info, char *info) 
{
	int quit = 0;
	
	if (out.flgs[0].flag) /* Version */ {
		quit = 1;
		fprintf(stderr, "%s version %s\n",
			out.progname, FLICK_VERSION);
	}
	if (out.error || out.flgs[1].flag || out.flgs[3].flag) {
		/* Print the usage */
		quit = 1;
		print_args_usage(out.progname, flag_count, in, cl_info, info);
	}
	if (quit)
		exit(out.error);
}

/* End of file. */

