/*
 * Copyright (c) 1995, 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#ifndef __STRINGS_H
#define __STRINGS_H

#ifdef STREAM_STRINGS
#include <iostream.h>
#endif

// Define this for copies of the string with the char * cast..

class Strings {  
public:

  enum Directions {left, right, leftAndRight};

  Strings ();
  Strings (char);
  Strings (const char*);
  Strings (const Strings&);
  virtual ~Strings();

  static Strings emptyString();

  int          beginsWith (const char*) const;
  unsigned int count (const char) const;
  int          endsWith (const char*) const;
  int          empty() const;
  unsigned int length() const;
  Strings      lowercaseOf() const;
  int          matches (const char*) const;
  int          matches (const Strings&) const;
  int          soundsLike (const char*) const;
  int          soundsLike (const Strings&) const;
  double       toDouble() const;
  int          toInt() const;
  long         toLong() const;
  Strings      token() const;
  Strings      token (const char delimitor) const;
  Strings      uppercaseOf() const;

  void append (char);
  void append (const char*);
  void append (const Strings&); 
  void append (const int);
  void clear();
  void lowercase();
  void prepend (char);
  void prepend (const char*);
  void prepend (const Strings&);
  void prepend (const int);
  void remove (const char);
  void replace (const char, const char);
  void reverse();
  void slice (unsigned int, const Directions = left);
  void sliceLeft (Strings&, unsigned int);
  void sliceToken (const unsigned int = 1);
  void sliceToken (const char delimitor, int removeDelimitor = 0);
  void sliceToken (Strings& token);
  void sliceToken (Strings& token, const char delimitor, int removeDelimitor = 0);
  void trim (const Directions = leftAndRight);
  void trim (const char, const Directions = leftAndRight);
  void uppercase(); 

  operator const char*() const;

	// WARNING:  This is LEAKY!!!
	operator char *() const;

#ifdef STREAM_STRINGS	
  friend ostream& operator << (ostream&, const Strings&);
#endif
	
  const Strings& operator = (char);
  const Strings& operator = (const char*);
  const Strings& operator = (const Strings&);
  Strings& operator + (char) const;
  Strings& operator + (const char*) const;
  Strings& operator + (const Strings&) const;
  friend Strings operator + (char, const Strings&);
  friend Strings operator + (const char*, const Strings&);
  const Strings& operator += (char);
  const Strings& operator += (const char*);
  const Strings& operator += (const Strings&);
  int operator == (const Strings&) const;
  int operator == (const char*) const;
  int operator != (const Strings&) const;
  int operator != (const char*) const;
  int operator <  (const Strings&) const;
  int operator <  (const char*) const; 
  int operator >  (const Strings&) const;
  int operator >  (const char*) const;  
  int operator <= (const Strings&) const;
  int operator <= (const char*) const;  
  int operator >= (const Strings&) const; 
  int operator >= (const char*) const;
  const char& operator [] (const unsigned int) const;

protected:
  
  int  checkMatch (char*, char*) const;
  void sliceLeft (unsigned int);
  void sliceRight (unsigned int);
  Strings soundexCodeOf (const char*) const;
  int soundexNumericOf (const char) const;
  void trimLeft();
  void trimRight();
  void trimLeft (const char);
  void trimRight (const char);

  char* theString;
};


#endif


// Copyright (c) 1995-1996 D J Supel and ISX Corporation


// beginsWith (const char*): Returns true if the prefix of the String 
// exactly matches the client specified character string.
//
//   Example: beginsWith ("hello")       on "hello world"    returns 1;
//   Example: beginsWith ("hello")       on "  hello world"  returns 0;
//   Example: beginsWith ("hello world") on "hello world!!!" returns 1;

// matches (const char* pattern) and matches (const Strings& pattern)
// provide UNIX shell-like pattern matching for ?, [], and *.
//
//   Example: matches ("?b?") on "abc" returns 1.
//   Example: matches ("[abc]*") on "apple" returns 1.
//   Example: matches ("[abc]*") on "cat" returns 1.
//   Example: matches ("[abc]*") on "dog" returns 0.

// soundsLike (const char*) and soundsLike (const Strings&) returns
// true if the strings compare equally by 'sound', not 'spelling'.
// The method is based on the "soundex" algorithm so best results
// occur in word comparisions as opposed to sentence/phrases. The
// algorithm is 'English' based.
//
//   Example: soundsLike ("Beatles") on "beedles" returns 1.
//   Example: soundsLike ("Botany Bay") on "Botneebay" returns 1.
//   Example: soundsLike ("Rugby") on "rugbug" returns 0.

// token() : Returns the first non-space sequence of characters.
//
//   Example: token() on "  hello world!"  returns "hello"
//   Example: token() on "hello"           returns "hello"
//   Example: token() on "  hello   "      returns "hello"
//
// token (const char) : Returns the sequence of characters up to 
// the delimitor. If the delimitor is not found, the original string, 
// less any preceeding blank spaces, is returned.
//
//   Example: token (",") on "  hello world, how"  returns "hello world"
//   Example: token (",") on "hello world"         returns "hello world"
//   Example: token (",") on "   hello world,,, "  returns "hello world"

// remove (char): remove ALL occurences of the specified character.
//
//   Example: remove (',') on "12,000"     is set to "12000"
//   Example: remove (',') on "12,000,000" is set to "12000000" 

// replace (char old, char new): replace ALL occurences of the old
// character with the new character.
//
//   Example: replace (',', '_') on "12,000"     is set to "12_000"
//   Example: replace (',', '_') on "12,000,000" is set to "12_000_000" 

// slice (unsigned int, const Directions): cuts off the number of characters
// specified. The client can specify which side of the string the characters 
// are 'sliced' off.
//
//   Example: slice ((unsigned int)4) on "hello world" returns "o world"
//   Example: slice ((unsigned int)2, Strings::right) on "happy" returns "hap"

// sliceToken (unsigned int) : Removes the number of 'tokens' specified and  
// sets the string to the begining of the next token.
//
//   Example: sliceToken()  on "   hello world!"  is set to "world!"
//   Example: sliceToken()  on "   $ world!"      is set to "world!"
//   Example: sliceToken()  on "good-day world!"  is set to "world!"
//   Example: sliceToken(2) on "good day world!"  is set to "world!"
//
// sliceToken (const char, int) : Removes the sequence of characters up 
// to the delimitor, including delimitor if int value is true. If the 
// delimitor is not found, the string is altered by removal of any 
// preceeding blank spaces only.
//
//   Example: sliceToken(",") on    "  hello world, how"   is set to ", how"
//   Example: sliceToken(",", 1) on " hello world, how"    is set to "how"
//   Example: sliceToken(",", 1) on " hello world,,, how"  is set to ",,how"
//
// sliceToken (Strings&) and sliceToken (Strings&, const char, int): Perform
// the same operations as the similar sliceToken methods above except the
// sliced off token is returned to the user. This sliced off token will
// have no white-space to its left or right.

// trim(): Removes ALL occurences of white-space. The client can specify 
// from which direction trimming takes place.
//
//   Example: trim()                 on " xyz "  is set to "xyz"
//   Example: trim (Strings::left)   on " xyz "  is set to "xyz "
//   Example: trim (Strings::right)  on " xyz "  is set to " xyz"
//
// trim (const char, const Directions): Removes ALL occurences of the
// specified character. The client can specify from which direction 
// trimming of this character takes place.
